import datetime
import logging
import time
from string import Template

logger = logging.getLogger('custom')


class Constants(object):
    eta = 1e-6


class Timer:
    def __init__(self, name, event_frequency='high'):
        self.name = name
        self.event_frequency = event_frequency

    def __enter__(self):
        self.begin = time.time()
        self.begin = datetime.datetime.now()
        if self.event_frequency == 'low':
            sep = f'\n{"#" * 100}\n'
            msg = f' {self.name} ({self.strform(self.begin)})'
            msg = sep + msg + sep
        elif self.event_frequency == 'medium':
            msg = f'\n==> {self.name} ({self.strform(self.begin)})'
        elif self.event_frequency == 'high':
            msg = f'\n====> {self.name} ({self.strform(self.begin)})'
        else:
            raise ValueError
        logger.info(msg)
        return self

    def __exit__(self, *args):
        self.end = datetime.datetime.now()
        elapsed = self.end - self.begin
        elapsed = strfdelta(elapsed, "%D days %H:%M:%S")
        msg = f'Time for "{self.name}": {elapsed} ({self.strform(self.end)})'
        logger.info(msg)

    @staticmethod
    def strform(x):
        return x.strftime("%Y-%m-%d %H:%M")


# https://stackoverflow.com/questions/8906926/formatting-timedelta-objects
class DeltaTemplate(Template):
    delimiter = "%"


# https://stackoverflow.com/questions/8906926/formatting-timedelta-objects
def strfdelta(tdelta, fmt):
    d = {"D": tdelta.days}
    hours, rem = divmod(tdelta.seconds, 3600)
    minutes, seconds = divmod(rem, 60)
    d["H"] = f'{hours:02d}'
    d["M"] = f'{minutes:02d}'
    d["S"] = f'{seconds:02d}'
    t = DeltaTemplate(fmt)
    return t.substitute(**d)
