#include <stdio.h>
#include <stdlib.h>
#include <time.h>
#include <math.h>
#include <petsc.h>

int main(int argc, char **args)
{
    PetscCall(PetscInitialize(&argc, &args, (char *)0, NULL));
    Vec x, b;                                                       // 初始解向量,右端项,精确解向量
    Mat A;                                                          // 系数矩阵
    KSP ksp;                                                        // 求解器
    PC pc;                                                          // 预处理器
    PetscRandom rctx;                                               // 随机数生成器
    KSPConvergedReason reason;                                      // 收敛原因
    PetscViewer viewer;                                             // 视图器
    PetscInt its;                                                   // 系数矩阵边长,迭代次数
    PetscLogDouble t1, t2;                                          // 求解时间
    unsigned long seed;                                             // 随机数种子
    PetscReal emax, emin, rcond, norm_b, norm_x;                    // 最大奇异值,最小奇异值,条件数,右端项模,初始解向量模
    char file_A[PETSC_MAX_PATH_LEN], file_b[PETSC_MAX_PATH_LEN];    // 文件地址

    // 读取文件地址
    PetscOptionsBegin(PETSC_COMM_WORLD, NULL, "Path to input file", NULL);
    PetscOptionsString("-file_A", "Path to input file", NULL, file_A, file_A, sizeof(file_A), NULL);
    PetscOptionsString("-file_b", "Path to input file", NULL, file_b, file_b, sizeof(file_b), NULL);
    PetscOptionsEnd();

    // 读取系数矩阵及右端项
    PetscCall(MatCreate(PETSC_COMM_WORLD, &A));
    PetscCall(PetscViewerBinaryOpen(PETSC_COMM_WORLD, file_A, FILE_MODE_READ, &viewer));
    PetscCall(MatLoad(A, viewer));
    PetscCall(PetscViewerDestroy(&viewer));
    PetscCall(VecCreate(PETSC_COMM_WORLD, &b));
    PetscCall(PetscViewerBinaryOpen(PETSC_COMM_WORLD, file_b, FILE_MODE_READ, &viewer));
    PetscCall(VecLoad(b, viewer));
    PetscCall(PetscViewerDestroy(&viewer));

    // 创建初始解向量
    PetscCall(VecCreate(PETSC_COMM_WORLD, &x));
    PetscCall(VecDuplicate(b, &x));
    PetscCall(VecSetFromOptions(x));
    PetscCall(PetscRandomCreate(PETSC_COMM_WORLD, &rctx));
    PetscCall(PetscRandomSetType(rctx, PETSCRAND));
    PetscCall(PetscRandomSetInterval(rctx, 0.0, 1.0));
    seed = (unsigned long)time(NULL);
    PetscCall(PetscRandomSetSeed(rctx, seed));
    PetscCall(PetscRandomSeed(rctx));
    PetscCall(VecSetRandom(x, rctx));

    // 计算初始解向量的模
    PetscCall(VecNorm(x, NORM_2, &norm_x));

    // 创建ksp求解器
    PetscCall(KSPCreate(PETSC_COMM_WORLD, &ksp));
    PetscCall(KSPSetComputeSingularValues(ksp, PETSC_TRUE));
    PetscCall(KSPSetOperators(ksp, A, A));
    // 创建pc预处理器
    PetscCall(KSPGetPC(ksp, &pc));
    PetscCall(PCSetType(pc, PCGAMG));
    PetscCall(KSPSetFromOptions(ksp));

    // 计算求解时间
    PetscCall(PetscTime(&t1));
    PetscCall(KSPSolve(ksp, b, x));
    PetscCall(PetscTime(&t2));

    // 获取收敛原因
    PetscCall(KSPGetConvergedReason(ksp, &reason));

    // 计算最大最小奇异值与条件数
    PetscCall(KSPComputeExtremeSingularValues(ksp, &emax, &emin));
    rcond = emax / emin;

    // 得到迭代次数
    PetscCall(KSPGetIterationNumber(ksp, &its));

    // 依次输出求解时间,迭代次数,最大奇异值,最小奇异值,右端项模,初始解向量模,条件数,收敛原因
    PetscCall(PetscPrintf(PETSC_COMM_WORLD, "%g ", (t2 - t1)));
    PetscCall(PetscPrintf(PETSC_COMM_WORLD, "%D ", its));
    PetscCall(PetscPrintf(PETSC_COMM_WORLD, "%g %g %g %g %g %d\n", emax, emin, norm_b, norm_x, rcond, reason));

    // 释放内存
    PetscCall(MatDestroy(&A));
    PetscCall(VecDestroy(&x));
    PetscCall(VecDestroy(&b));
    PetscCall(PetscRandomDestroy(&rctx));

    PetscCall(VecDestroy(&b));
    PetscCall(PetscFinalize());
    return 0;
}