#include <stdio.h>
#include <stdlib.h>
#include <time.h>
#include <math.h>
#include <petsc.h>

int main(int argc, char **args) {
    PetscCall(PetscInitialize(&argc, &args, (char*)0, NULL));
    Vec             x, b, xeaxct;                       //初始解向量,右端项,精确解向量
    Mat             A;                                  //系数矩阵
    KSP             ksp;                                //求解器
    PC              pc;                                 //预处理器
    PetscRandom     rctx;                               //随机数生成器
    PetscInt        N = 10000, its;                     //系数矩阵边长,迭代次数
    int             M = 1000, colss = 6;                //数据集大小,PDE参数个数
    int             rows, cols, nnz;                    //中间数据请忽略
    PetscInt        *row_indices, *col_indices;         //中间数据请忽略
    PetscLogDouble  t1, t2;                             //求解时间
    unsigned long   seed;                               //随机数种子
    PetscScalar     *values;                            //中间数据请忽略
    double          *omegas;                            //PDE参数,本次计算所使用的松弛系数
    PetscReal       emax, emin, rcond, norm_b, norm_x;  //最大奇异值,最小奇异值,条件数,右端项模,初始解向量模
    PetscReal       omega;                              //松弛系数

    //读取A.bin,即系数矩阵
    FILE *file0 = fopen("A.bin", "rb");
    if (!file0) {
        PetscCall(PetscPrintf(PETSC_COMM_WORLD, "Unable to open A.bin!\n"));
        return 1;
    }

    //创建精确解向量
    PetscCall(VecCreate(PETSC_COMM_WORLD, &xeaxct));
    PetscCall(VecSetSizes(xeaxct, PETSC_DECIDE, N));
    PetscCall(VecSetFromOptions(xeaxct));
    PetscCall(PetscRandomCreate(PETSC_COMM_WORLD, &rctx));
    PetscCall(PetscRandomSetType(rctx, PETSCRAND));
    PetscCall(PetscRandomSetInterval(rctx, 0.0, 1));
    PetscCall(VecSetRandom(xeaxct, rctx));
    PetscCall(PetscRandomDestroy(&rctx));

    fread(&rows, sizeof(PetscInt), 1, file0);
    fread(&cols, sizeof(PetscInt), 1, file0);
    fread(&nnz, sizeof(PetscInt), 1, file0);

    // 分配内存
    row_indices = malloc(sizeof(PetscInt)*rows+1);
    col_indices = malloc(sizeof(PetscInt)*nnz);
    values = malloc(sizeof(PetscScalar)*nnz);

    // 读取矩阵数据
    fread(row_indices, sizeof(PetscInt), rows+1, file0);
    fread(col_indices, sizeof(PetscInt), nnz, file0);
    fread(values, sizeof(PetscScalar), nnz, file0);

    for (int i=0;i<nnz;i++) {
        *(values+i) = *(values+i) / (10);
    }

    //创建系数矩阵
    PetscCall(MatCreateSeqAIJWithArrays(PETSC_COMM_SELF, rows, cols, row_indices, col_indices, values, &A));
        
    //计算右端项
    PetscCall(VecDuplicate(xeaxct, &b));
    PetscCall(MatMult(A, xeaxct, b));

    //计算右端项的模
    PetscCall(VecNorm(b, NORM_2, &norm_b));
        
    //创建初始解向量
    PetscCall(VecCreate(PETSC_COMM_WORLD, &x));
    PetscCall(VecSetSizes(x, PETSC_DECIDE, N));
    PetscCall(VecSetFromOptions(x));
    PetscCall(PetscRandomCreate(PETSC_COMM_WORLD, &rctx));
    PetscCall(PetscRandomSetType(rctx, PETSCRAND));
    PetscCall(PetscRandomSetInterval(rctx, 0.0, 1.0));
    seed = (unsigned long)time(NULL);
    PetscCall(PetscRandomSetSeed(rctx, seed));
    PetscCall(PetscRandomSeed(rctx));
    PetscCall(VecSetRandom(x, rctx));

    //计算初始解向量的模
    PetscCall(VecNorm(x, NORM_2, &norm_x));
        
    //创建ksp求解器
    PetscCall(KSPCreate(PETSC_COMM_WORLD, &ksp));
    PetscCall(KSPSetComputeSingularValues(ksp, PETSC_TRUE));
    PetscCall(KSPSetOperators(ksp, A, A));
    //创建pc预处理器
    PetscCall(KSPGetPC(ksp, &pc));
    PetscCall(PCSetType(pc, PCGAMG));
    PetscCall(KSPSetFromOptions(ksp));

    //计算求解时间
    PetscCall(PetscTime(&t1));
    PetscCall(KSPSolve(ksp, b, x));
    PetscCall(PetscTime(&t2));
        
    //计算最大最小奇异值与条件数
    PetscCall(KSPComputeExtremeSingularValues(ksp, &emax, &emin));
    rcond = emax/emin;
        
    //得到迭代次数
    PetscCall(KSPGetIterationNumber(ksp, &its));

    //依次输出数据序号,求解时间,迭代次数,最大奇异值,最小奇异值,右端项模,初始解向量模,条件数
    PetscCall(PetscPrintf(PETSC_COMM_WORLD, "%g ", (t2 - t1)));
    PetscCall(PetscPrintf(PETSC_COMM_WORLD, "%D ", its));
    PetscCall(PetscPrintf(PETSC_COMM_WORLD, "%g %g %g %g %g\n", emax, emin, norm_b, norm_x, rcond));
        
    //释放内存
    PetscCall(MatDestroy(&A));
    PetscCall(VecDestroy(&x));
    PetscCall(VecDestroy(&b));
    PetscCall(PetscRandomDestroy(&rctx));
    free(row_indices);
    free(col_indices);
    free(values);
    
    fclose(file0);
    PetscCall(VecDestroy(&b));
    PetscCall(PetscFinalize());
    return 0;
}