#include <stdio.h>
#include <stdlib.h>
#include <time.h>
#include <math.h>
#include <petsc.h>

PetscErrorCode MyKSPMonitor(KSP ksp, PetscInt n, PetscReal rnorm, void *ctx)
{
    //该监视器仅输出初始残差
    if (n == 0)
    {
        PetscCall(PetscPrintf(PETSC_COMM_WORLD, "%g ", (double)rnorm));
    }
    return 0;
}

int main(int argc, char **args) {
    PetscCall(PetscInitialize(&argc, &args, (char*)0, NULL));
    Vec             x, b, xeaxct;                       //初始解向量,右端项,精确解向量
    Mat             A;                                  //系数矩阵
    KSP             ksp;                                //求解器
    PC              pc;                                 //预处理器
    PetscRandom     rctx;                               //随机数生成器
    KSPConvergedReason reason;                          //收敛原因
    PetscInt        N = 4900, its;                      //系数矩阵边长,迭代次数
    int             rows, cols, nnz;                    //中间数据请忽略
    PetscInt        *row_indices, *col_indices;         //中间数据请忽略
    PetscLogDouble  t1, t2;                             //求解时间
    unsigned long   seed;                               //随机数种子
    PetscScalar     *values;                            //中间数据请忽略
    PetscReal       omega, norm_ksp, norm_b, norm_x;    //松弛系数,残差,右端项模,初始解向量模

    //读取A.bin,即系数矩阵
    FILE *file0 = fopen("A.bin", "rb");
    if (!file0) {
        PetscCall(PetscPrintf(PETSC_COMM_WORLD, "Unable to open A.bin!\n"));
        return 1;
    }

    //创建精确解向量
    PetscCall(VecCreate(PETSC_COMM_WORLD, &xeaxct));
    PetscCall(VecSetSizes(xeaxct, PETSC_DECIDE, N));
    PetscCall(VecSetFromOptions(xeaxct));
    PetscCall(PetscRandomCreate(PETSC_COMM_WORLD, &rctx));
    PetscCall(PetscRandomSetType(rctx, PETSCRAND));
    PetscCall(PetscRandomSetInterval(rctx, 0.0, 1));
    PetscCall(VecSetRandom(xeaxct, rctx));
    PetscCall(PetscRandomDestroy(&rctx));

    fread(&rows, sizeof(PetscInt), 1, file0);
    fread(&cols, sizeof(PetscInt), 1, file0);
    fread(&nnz, sizeof(PetscInt), 1, file0);

    // 分配内存
    row_indices = malloc(sizeof(PetscInt)*rows+1);
    col_indices = malloc(sizeof(PetscInt)*nnz);
    values = malloc(sizeof(PetscScalar)*nnz);

    // 读取矩阵数据
    fread(row_indices, sizeof(PetscInt), rows+1, file0);
    fread(col_indices, sizeof(PetscInt), nnz, file0);
    fread(values, sizeof(PetscScalar), nnz, file0);

    for (int i=0;i<nnz;i++) {
        *(values+i) = *(values+i) / (10);
    }

    //创建系数矩阵
    PetscCall(MatCreateSeqAIJWithArrays(PETSC_COMM_SELF, rows, cols, row_indices, col_indices, values, &A));
        
    //计算右端项
    PetscCall(VecDuplicate(xeaxct, &b));
    PetscCall(MatMult(A, xeaxct, b));

    //计算右端项的模
    PetscCall(VecNorm(b, NORM_2, &norm_b));
        
    //创建初始解向量
    PetscCall(VecCreate(PETSC_COMM_WORLD, &x));
    PetscCall(VecSetSizes(x, PETSC_DECIDE, N));
    PetscCall(VecSetFromOptions(x));
    PetscCall(PetscRandomCreate(PETSC_COMM_WORLD, &rctx));
    PetscCall(PetscRandomSetType(rctx, PETSCRAND));
    PetscCall(PetscRandomSetInterval(rctx, 0.0, 1.0));
    seed = (unsigned long)time(NULL);
    PetscCall(PetscRandomSetSeed(rctx, seed));
    PetscCall(PetscRandomSeed(rctx));
    PetscCall(VecSetRandom(x, rctx));

    //计算初始解向量的模
    PetscCall(VecNorm(x, NORM_2, &norm_x));

    //创建ksp求解器
    PetscCall(KSPCreate(PETSC_COMM_WORLD, &ksp));
    PetscCall(KSPSetType(ksp, KSPGMRES));
    PetscCall(KSPSetComputeSingularValues(ksp, PETSC_TRUE));
    PetscCall(KSPSetOperators(ksp, A, A));
    //创建pc预处理器
    PetscCall(KSPGetPC(ksp, &pc));
    PetscCall(PCSetType(pc, PCEISENSTAT));
    omega = 0.8;
    PetscCall(PCEisenstatSetOmega(pc, omega));
    PetscCall(KSPSetFromOptions(ksp));

    //设置迭代监视器
    PetscCall(KSPMonitorSet(ksp, MyKSPMonitor, NULL, NULL));

    //计算求解时间
    PetscCall(PetscTime(&t1));
    PetscCall(KSPSolve(ksp, b, x));
    PetscCall(PetscTime(&t2));

    //获取收敛原因
    PetscCall(KSPGetConvergedReason(ksp, &reason));

    //计算结束时残差
    PetscCall(KSPGetResidualNorm(ksp, &norm_ksp));
    PetscCall(PetscPrintf(PETSC_COMM_WORLD, "%g ", norm_ksp));

    //得到迭代次数
    PetscCall(KSPGetIterationNumber(ksp, &its));

    //依次输出求解时间,迭代次数,右端项模,初始解向量模
    PetscCall(PetscPrintf(PETSC_COMM_WORLD, "%g ", (t2 - t1)));
    PetscCall(PetscPrintf(PETSC_COMM_WORLD, "%D ", its));
    PetscCall(PetscPrintf(PETSC_COMM_WORLD, "%g %g %d\n", norm_b, norm_x, reason));
        
    //释放内存
    PetscCall(MatDestroy(&A));
    PetscCall(VecDestroy(&x));
    PetscCall(VecDestroy(&b));
    PetscCall(PetscRandomDestroy(&rctx));
    free(row_indices);
    free(col_indices);
    free(values);
    
    fclose(file0);
    PetscCall(VecDestroy(&b));
    PetscCall(PetscFinalize());
    return 0;
}