import json
from pathlib import Path
from typing import Any
import click
from gluonts.dataset.stat import calculate_dataset_statistics
from tqdm.auto import tqdm
from tsbench.config import DATASET_REGISTRY


@click.command()
@click.argument(
    "path",
    type=click.Path(exists=True),
    nargs=1,
    default=Path.home() / "data" / "stats",
)
def main(path: str):
    """
    Computes the dataset features for all datasets in the registry.
    """
    directory = Path(path)
    for dataset, config in tqdm(list(DATASET_REGISTRY.items())):
        file = directory / f"{dataset}.json"
        if file.exists():
            continue

        stats = calculate_dataset_statistics(config.data.train(val=False).gluonts())
        with file.open("w+") as f:
            json.dump(
                {
                    "integer_dataset": stats.integer_dataset,
                    "mean_target_length": stats.mean_target_length,
                    "num_time_observations": stats.num_time_observations,
                    "num_time_series": stats.num_time_series,
                },
                f,
            )


def is_json_compatible(x: Any):
    try:
        json.dumps(x)
        return True
    except (TypeError, OverflowError):
        return False


if __name__ == "__main__":
    # pylint: disable=no-value-for-parameter
    main()
