from __future__ import annotations
from dataclasses import asdict, dataclass
from typing import Callable, Dict, List, Union
import pandas as pd
from .dataset import DatasetConfig
from .model import ModelConfig, TrainConfig


@dataclass(frozen=True)
class Config:
    """
    A configuration is a tuple of a model configuration and a dataset configuration.
    """

    @classmethod
    def to_model_configs(cls, configs: List[Config]) -> List[ModelConfig]:
        """
        Returns the unique model configurations from the given configurations.
        """
        return list({c.model for c in configs})

    @classmethod
    def to_dataframe(
        cls,
        configs: List[Config],
        dataset_fn: Callable[
            [DatasetConfig], Dict[str, Union[bool, float, int, str]]
        ] = lambda _: {},
    ) -> pd.DataFrame:
        """
        Returns a data frame representing the provided configurations. The model is translated into
        columns of the data frame automatically. The dataset, however, needs to be converted
        according to the provided callable. By default, this callable is a noop and does not add
        any dataset information to the data frame.
        """
        rows: List[Dict[str, Union[str, float, int, bool]]] = [
            {
                "model": c.model.name(),
                **{
                    ("dataset" if k == "" else f"dataset_{k}"): v
                    for k, v in dataset_fn(c.dataset).items()
                },
                **{
                    (
                        f"model_{k}"
                        if k in TrainConfig.training_hyperparameters()
                        else f"model_{c.model.name()}_{k}"
                    ): v
                    for k, v in asdict(c.model).items()
                },
            }
            for c in configs
        ]
        return pd.DataFrame(rows)

    model: ModelConfig
    dataset: DatasetConfig
