import numpy as np


def nrmse(y_pred: np.ndarray, y_true: np.ndarray) -> np.ndarray:
    """
    Computes the normalized root mean squared error of the predictions.

    Args:
        y_pred: Array of shape [N, D] with the predictions (N: number of elements, D: number of
            metrics).
        y_true: Array of shape [N, D] with the true metrics.

    Returns:
        Array of shape [D] with the average NRMSE for each metric.
    """
    rmse = np.sqrt((y_pred - y_true) ** 2).mean(0)
    return rmse / np.abs(y_true).mean(0)


def smape(y_pred: np.ndarray, y_true: np.ndarray) -> np.ndarray:
    """
    Computes the symmetric mean absolute percentage error of the predictions.

    Args:
        y_pred: Array of shape [N, D] with the predictions (N: number of elements, D: number of
            metrics).
        y_true: Array of shape [N, D] with the true metrics.

    Returns:
        Array of shape [D] with the average sMAPE for each metric.
    """
    num = np.abs(y_pred - y_true)
    denom = (np.abs(y_pred) + np.abs(y_true)) / 2
    return 100 * (num / denom).mean(0)


def mrr(y_pred: np.ndarray, y_true: np.ndarray) -> np.ndarray:
    """
    Computes the mean reciprocal rank of the predictions.

    Args:
        y_pred: Array of shape [N, D] with the predictions (N: number of elements, D: number of
            metrics).
        y_true: Array of shape [N, D] with the true metrics.

    Returns:
        Array of shape [D] with the average MRR for each metric.
    """
    minimum_indices = y_pred.argmin(0)  # [D]
    true_ranks = y_true.argsort(0).argsort(0)  # [N, D]
    ranks = np.take_along_axis(true_ranks, minimum_indices[None, :], axis=0)  # [N, D]
    result = 1 / (ranks + 1)
    return result.mean(0)


def precision_k(k: int, y_pred: np.ndarray, y_true: np.ndarray) -> np.ndarray:
    """
    Computes the precision@k of the predictions.

    Args:
        k: The number of items that are looked at for computing the precision.
        y_pred: Array of shape [N, D] with the predictions (N: number of elements, D: number of
            metrics).
        y_true: Array of shape [N, D] with the true metrics.

    Returns:
        Array of shape [D] with the precisions@k for each metric.
    """
    pred_ranks = y_pred.argsort(0).argsort(0)  # [N, D]
    true_ranks = y_true.argsort(0).argsort(0)  # [N, D]

    pred_relvance = (pred_ranks <= k) / k  # [N, D]
    true_relevance = true_ranks <= k  # [N, D]

    return (pred_relvance * true_relevance).sum(0)


def ndcg(y_pred: np.ndarray, y_true: np.ndarray) -> np.ndarray:
    """
    Computes the normalized discounted cumulative gain of the predictions.

    Args:
        y_pred: Array of shape [N, D] with the predictions (N: number of elements, D: number of
            metrics).
        y_true: Array of shape [N, D] with the true metrics.

    Returns:
        Array of shape [D] with the nDCG for each metric.
    """
    n = y_pred.shape[0]

    # First, get the relevance
    true_argsort = y_true.argsort(0)  # [N, D]
    true_ranks = true_argsort.argsort(0)  # [N, D]
    relevance = 1 / (true_ranks + 1)  # [N, D]
    relevance[true_ranks >= 10] = 0

    # Then, compute the iDCG
    num = np.take_along_axis(relevance, true_argsort, axis=0)  # [N, D]
    denom = np.log2(np.arange(n) + 2)[:, None]  # [N, D]
    idcg = (num / denom).sum(0)

    # And compute the DCG
    pred_argsort = y_pred.argsort(0)
    num = np.take_along_axis(relevance, pred_argsort, axis=0)
    denom = np.log2(np.arange(n) + 2)[:, None]
    dcg = (num / denom).sum(0)

    return dcg / idcg
