from typing import Iterator, List, Tuple
from sklearn.model_selection import LeaveOneGroupOut
from tsbench.config import Config
from tsbench.experiments.metrics import Performance
from tsbench.experiments.tracking import Tracker


def loocv_split(
    tracker: Tracker,
) -> Iterator[
    Tuple[Tuple[List[Config], List[Performance], Tuple[List[Config], List[Performance]]]]
]:
    """
    Iterates over the configurations and associated performances obtained from a collector. For
    each item it yields, it leaves out configurations/performances for a single dataset for testing
    and provides the configurations/performances for training.

    Args:
        tracker: The tracker to retrieve data from.
        show_progerss: Whether to show progress via tqdm.

    Yields:
        The training data, including configurations and performances.
        The test data, including configurations and performances.
    """
    data = tracker.get_results()
    groups = [c.dataset.name for c in data.configurations]

    # Split the data according to the datasets
    loocv = LeaveOneGroupOut()
    for I_train, I_test in loocv.split(data.configurations, groups=groups):
        X_train = [data.configurations[i] for i in I_train]
        y_train = [data.performances[i] for i in I_train]

        X_test = [data.configurations[i] for i in I_test]
        y_test = [data.performances[i] for i in I_test]

        yield (X_train, y_train), (X_test, y_test)
