from __future__ import annotations
from dataclasses import asdict, dataclass
from typing import Dict, List, Union
import pandas as pd
from .metric import Metric


@dataclass
class Performance:
    """
    The performance class encapsulates the metrics that are recorded for configurations.
    """

    training_time: Metric
    latency: Metric
    num_model_parameters: Metric
    num_gradient_updates: Metric

    mean_weighted_quantile_loss: Metric
    mase: Metric
    smape: Metric
    nrmse: Metric
    nd: Metric

    @classmethod
    def from_dict(cls, metrics: Dict[str, Union[float, int]]) -> Performance:
        """
        Initializes a new performance object from the given 1D dictionary. Metrics are expected to
        be provided via `<metric>_mean` and `<metric>_std` keys.
        """
        kwargs = {m: Metric(metrics[f"{m}_mean"], metrics[f"{m}_std"]) for m in cls.metrics()}
        return Performance(**kwargs)  # type: ignore

    @classmethod
    def metrics(cls) -> List[str]:
        """
        Returns the list of metrics that are exposed by the performance class.
        """
        # pylint: disable=no-member
        return list(cls.__dataclass_fields__.keys())  # type: ignore

    @classmethod
    def to_dataframe(cls, performances: List[Performance], std: bool = True) -> pd.DataFrame:
        """
        Returns a data frame representing the provided performances.
        """
        result = []
        for p in performances:
            d = {}
            for key, subdict in asdict(p).items():
                if isinstance(subdict, dict):
                    for subkey, value in subdict.items():
                        if subkey == "std" and not std:
                            continue
                        d[f"{key}_{subkey}"] = value
                else:
                    d[key] = subdict
            result.append(d)
        return pd.DataFrame(result)
