from typing import List, Optional
from tsbench.config import Config
from tsbench.experiments.metrics import Performance
from tsbench.experiments.tracking import Tracker
from .base import Recommender
from .generator import CandidateGenerator
from .registry import register_recommender


@register_recommender("optimal")
class OptimalRecommender(Recommender):
    """
    The surrogate recommender recommends models by predicting their performance using a surrogate
    model.
    """

    def __init__(
        self,
        tracker: Tracker,
        generator: Optional[CandidateGenerator] = None,
        maximize: Optional[List[str]] = None,
        minimize: Optional[List[str]] = None,
        focus: Optional[str] = None,
    ):
        """
        Args:
            tracker: The tracker to obtain true performance metrics from.
            generator: The generator that generates configurations for recommendations. By default,
                this is the replay candidate generator.
            maximize: The list of performance metrics to maximize.
            minimize: The list of performance metrics to minimize.
            focus: The metric to prefer. Must be either in the list of the metrics to maximize or
                minimize. If not provided, the first metric to minimize is chosen or, if none such
                metric is provided, the first metric to maximize.
        """
        super().__init__(generator, maximize, minimize, focus)
        self.tracker = tracker

    def _get_performances(self, configs: List[Config]) -> List[Performance]:
        return [self.tracker.get_performance(c) for c in configs]
