from typing import Sequence, Union
import torch
from ignite.exceptions import NotComputableError
# noinspection PyProtectedMember
from ignite.metrics.metric import Metric, reinit__is_reduced, sync_all_reduce

__all__ = ["RelativeRootMeanSquaredError", "RelativeRootMeanSquaredErrorScore"]


class RelativeRootMeanSquaredError(Metric):
    r"""Calculates the `mean squared error <https://en.wikipedia.org/wiki/Mean_squared_error>`_.
    .. math:: \text{MSE} = \frac{1}{N} \sum_{i=1}^N \left(y_{i} - x_{i} \right)^2
    where :math:`y_{i}` is the prediction tensor and :math:`x_{i}` is ground true tensor.
    - ``update`` must receive output of the form ``(y_pred, y)`` or ``{'y_pred': y_pred, 'y': y}``.
    Args:
        output_transform: a callable that is used to transform the
            :class:`~ignite.engine.engine.Engine`'s ``process_function``'s output into the
            form expected by the metric. This can be useful if, for example, you have a multi-output model and
            you want to compute the metric with respect to one of the outputs.
            By default, metrics require the output as ``(y_pred, y)`` or ``{'y_pred': y_pred, 'y': y}``.
        device: specifies which device updates are accumulated on. Setting the
            metric's device to be the same as your ``update`` arguments ensures the ``update`` method is
            non-blocking. By default, CPU.
    """

    @reinit__is_reduced
    def reset(self) -> None:
        # noinspection PyAttributeOutsideInit
        self._sum_of_relative_errors = torch.tensor(0.0, device=self._device)
        # noinspection PyAttributeOutsideInit
        self._num_examples = 0

    @reinit__is_reduced
    def update(self, output: Sequence[torch.Tensor]) -> None:
        y_pred, y = output[0].detach(), output[1].detach()
        squared_errors = torch.sqrt(torch.sum(torch.pow(y_pred - y.view_as(y_pred), 2), 1))
        squared_base = torch.sqrt(torch.sum(torch.pow(y.view_as(y_pred), 2), 1))
        eps = 1e-6
        relative_errors = squared_errors / (squared_base + torch.tensor(eps))
        self._sum_of_relative_errors += torch.sum(relative_errors).to(self._device)
        self._num_examples += y.shape[0]

    @sync_all_reduce("_sum_of_squared_errors", "_num_examples")
    def compute(self) -> Union[float, torch.Tensor]:
        if self._num_examples == 0:
            raise NotComputableError(
                "RelativeRootMeanSquaredError must have at least one example before it can be computed."
            )
        return self._sum_of_relative_errors.item() / self._num_examples


class RelativeRootMeanSquaredErrorScore(RelativeRootMeanSquaredError):
    @sync_all_reduce("_sum_of_squared_errors", "_num_examples")
    def compute(self) -> Union[float, torch.Tensor]:
        return super().compute() * -1
