import argparse
import torch
from torch.nn import Linear
import torch.nn.functional as F
from torch_geometric.nn import APPNP

from citation import get_planetoid_dataset, random_planetoid_splits, run
from citation.train_eval import random_geom_splits, random_splits
from other_datasets import *
import torch_geometric.transforms as T

parser = argparse.ArgumentParser()
parser.add_argument('--dataset', type=str, default='cora')
parser.add_argument('--random_splits', type=bool, default=False)
parser.add_argument('--runs', type=int, default=1)
parser.add_argument('--epochs', type=int, default=1000)
parser.add_argument('--lr', type=float, default=0.01)
parser.add_argument('--weight_decay', type=float, default=0.0005)
parser.add_argument('--early_stopping', type=int, default=100)
parser.add_argument('--hidden', type=int, default=64)
parser.add_argument('--dropout', type=float, default=0.5)
parser.add_argument('--normalize_features', type=bool, default=True)
parser.add_argument('--K', type=int, default=10)
parser.add_argument('--alpha', type=float, default=0.1)
parser.add_argument('--attack', type=str, default='no')
parser.add_argument('--ptb_rate', type=float, default=0.05)
args = parser.parse_args()


class Net(torch.nn.Module):
    def __init__(self, dataset):
        super(Net, self).__init__()
        self.lin1 = Linear(dataset.num_features, args.hidden)
        self.lin2 = Linear(args.hidden, dataset.num_classes)
        self.prop1 = APPNP(args.K, args.alpha)

    def reset_parameters(self):
        self.lin1.reset_parameters()
        self.lin2.reset_parameters()

    def forward(self, data):
        x, edge_index = data.x, data.edge_index
        x = F.dropout(x, p=args.dropout, training=self.training)
        x = F.relu(self.lin1(x))
        x = F.dropout(x, p=args.dropout, training=self.training)
        x = self.lin2(x)
        x = self.prop1(x, edge_index)
        return F.log_softmax(x, dim=1)


if args.attack in ['random', 'meta']:
    dataset = AttackedDataset('', args.dataset, transforms=T.NormalizeFeatures(), args=args)
    permute_masks = None
    permute_masks = random_splits if args.random_splits else None
else:
    if args.dataset in ['cora', 'citeseer', 'pubmed']:
        dataset = get_planetoid_dataset(args.dataset, args.normalize_features)
        # permute_masks = random_planetoid_splits if args.random_splits else None
        permute_masks = random_geom_splits if args.random_splits else None
    else:
        import torch_geometric.transforms as T
        dataset = OtherDataset('', args.dataset, transform=T.NormalizeFeatures())
        permute_masks = random_splits if args.random_splits else None
        # if args.dataset != 'nell':
        #     permute_masks = random_planetoid_splits
        if args.dataset in ['chameleon', 'squirrel', 'film', 'cornell', 'texas', 'wisconsin']:
            permute_masks = random_geom_splits

import logging
LOG_FILENAME = 'log/appnp_noise.log'
logging.basicConfig(filename=LOG_FILENAME,level=logging.DEBUG)
logging.debug(f'Dataset {args.dataset}, alpha {args.alpha}, hidden {args.hidden}, weight_decay {args.weight_decay}, lr {args.lr}, ptb_rate {args.ptb_rate}')

run(dataset, Net(dataset), args.runs, args.epochs, args.lr, args.weight_decay,
    args.early_stopping, permute_masks, name=args.dataset, ptb_rate=args.ptb_rate)

