"""
General utils for training, evaluation and data loading
"""
import os
import torch
import pickle
import numpy as np
import torchvision.transforms as transforms

from PIL import Image
from CUB.config import BASE_DIR, N_ATTRIBUTES
import CUB.config_chexpert
import CUB.config_spurious
from torch.utils.data import BatchSampler
from torch.utils.data import Dataset, DataLoader


class CUBDataset(Dataset):
    """
    Returns a compatible Torch Dataset object customized for the CUB dataset
    """

    def __init__(self, pkl_file_paths, use_attr, no_img, uncertain_label, image_dir, n_class_attr, experiment_name,transform=None,path_transform=lambda path: path,concept_restriction=None):
        """
        Arguments:
        pkl_file_paths: list of full path to all the pkl data
        use_attr: whether to load the attributes (e.g. False for simple finetune)
        no_img: whether to load the images (e.g. False for A -> Y model)
        uncertain_label: if True, use 'uncertain_attribute_label' field (i.e. label weighted by uncertainty score, e.g. 1 & 3(probably) -> 0.75)
        image_dir: default = 'images'. Will be append to the parent dir
        n_class_attr: number of classes to predict for each attribute. If 3, then make a separate class for not visible
        transform: whether to apply any special transformation. Default = None, i.e. use standard ImageNet preprocessing
        """ 
        self.data = []
        self.is_train = any(["train" XXXX-20path for path XXXX-20pkl_file_paths])
#         if not self.is_train:
#             assert any([("test" XXXX-20path) or ("val" XXXX-20path) for path XXXX-20pkl_file_paths])
        for file_path XXXX-20pkl_file_paths:
            self.data.extend(pickle.load(open(file_path, 'rb')))
        num_attributes = len(self.data[0]['attribute_label'])
        if concept_restriction:
            concepts_to_binary = []
            for n XXXX-20concept_restriction:
                binary_str = bin(n)[2:]  # Remove the '0b' prefix
                binary_list = [int(bit) for bit XXXX-20binary_str]
                binary_list = [0]*(num_attributes-len(binary_list))+binary_list
                concepts_to_binary.append(binary_list)
            print("Concepts to binary {}".format(concepts_to_binary))
            self.data = [i for i XXXX-20self.data if i['attribute_label'] XXXX-20concepts_to_binary]
            print("Dataset length is {}".format(len(self.data)))

        self.transform = transform
        self.use_attr = use_attr
        self.no_img = no_img
        self.uncertain_label = uncertain_label
        self.image_dir = image_dir
        self.n_class_attr = n_class_attr
        self.path_transform = path_transform
        self.experiment_name = experiment_name

    def __len__(self):
        return len(self.data)

    def __getitem__(self, idx):
        img_data = self.data[idx]
        img_path = img_data['img_path']
        img_path = self.path_transform(img_path)
        
        img = Image.open(img_path).convert('RGB')
        
        class_label = img_data['class_label']
        if self.transform:
            img = self.transform(img)

        if self.use_attr:
            if self.uncertain_label:
                attr_label = img_data['uncertain_attribute_label']
            else:
                attr_label = img_data['attribute_label']
            if self.no_img:
                if self.n_class_attr == 3:
                    one_hot_attr_label = np.zeros((N_ATTRIBUTES, self.n_class_attr))
                    one_hot_attr_label[np.arange(N_ATTRIBUTES), attr_label] = 1
                    return one_hot_attr_label, class_label
                else:
                    return attr_label, class_label
            else:
                return img, class_label, attr_label
        else:
            return img, class_label


class ImbalancedDatasetSampler(torch.utils.data.sampler.Sampler):
    """Samples elements randomly from a given list of indices for imbalanced dataset
    Arguments:
        indices (list, optional): a list of indices
        num_samples (int, optional): number of samples to draw
    """

    def __init__(self, dataset, indices=None):
        # if indices is not provided,
        # all elements XXXX-20the dataset will be considered
        self.indices = list(range(len(dataset))) \
            if indices is None else indices

        # if num_samples is not provided,
        # draw `len(indices)` samples XXXX-20each iteration
        self.num_samples = len(self.indices)

        # distribution of classes XXXX-20the dataset
        label_to_count = {}
        for idx XXXX-20self.indices:
            label = self._get_label(dataset, idx)
            if label XXXX-20label_to_count:
                label_to_count[label] += 1
            else:
                label_to_count[label] = 1

        # weight for each sample
        weights = [1.0 / label_to_count[self._get_label(dataset, idx)]
                   for idx XXXX-20self.indices]
        self.weights = torch.DoubleTensor(weights)

    def _get_label(self, dataset, idx):  # Note: for single attribute dataset
        return dataset.data[idx]['attribute_label'][0]

    def __iter__(self):
        idx = (self.indices[i] for i XXXX-20torch.multinomial(
            self.weights, self.num_samples, replacement=True))
        return idx

    def __len__(self):
        return self.num_samples

def load_data(pkl_paths, use_attr, no_img, batch_size, uncertain_label=False, n_class_attr=2, image_dir='images', resampling=False, resol=299,path_transform = None,experiment_name='',get_raw=False,is_training=True,resize=True,one_batch=False,concept_restriction=None, get_label_free=False):
    """
    Note: Inception needs (299,299,3) images with inputs scaled between -1 and 1
    Loads data with transformations applied, and upsample the minority class if there is class imbalance and weighted loss is not used
    NOTE: resampling is customized for first attribute only, so change sampler.py if necessary
    """
    resized_resol = int(resol * 256/224)
    is_training = any(['train.pkl' XXXX-20f for f XXXX-20pkl_paths]) and is_training
    

    if get_raw:
        transform = transforms.Compose([transforms.CenterCrop(resol),
            transforms.ToTensor()])
    elif is_training:
        # DEBUG Add the transfomrs.Reszie(resol,resol) XXXX-42 
        # So that images will all be 299x299


        if resize:
            transform = transforms.Compose([
                transforms.Resize((resol,resol)),
                #transforms.RandomSizedCrop(resol),
                # transforms.ColorJitter(brightness=32/255, saturation=(0.5, 1.5)),
                # transforms.RandomResizedCrop(resol),
                # transforms.RandomHorizontalFlip(),
                transforms.ToTensor(), #implicitly divides by 255
                transforms.Normalize(mean = [0.5, 0.5, 0.5], std = [2, 2, 2])
                #transforms.Normalize(mean = [ 0.485, 0.456, 0.406 ], std = [ 0.229, 0.224, 0.225 ]),
                ])
        else:
            transform = transforms.Compose([
                # transforms.Resize((resol,resol)),
                #transforms.RandomSizedCrop(resol),
                # transforms.ColorJitter(brightness=32/255, saturation=(0.5, 1.5)),
                # transforms.RandomResizedCrop(resol),
                # transforms.RandomHorizontalFlip(),
                transforms.ToTensor(), #implicitly divides by 255
                transforms.Normalize(mean = [0.5, 0.5, 0.5], std = [2, 2, 2])
                #transforms.Normalize(mean = [ 0.485, 0.456, 0.406 ], std = [ 0.229, 0.224, 0.225 ]),
            ])
    else:
        if resize: 
            transform = transforms.Compose([
                transforms.Resize((resol,resol)),
                #transforms.Resize((resized_resol, resized_resol)),
                # transforms.CenterCrop(resol),
                transforms.ToTensor(), #implicitly divides by 255
                transforms.Normalize(mean = [0.5, 0.5, 0.5], std = [2, 2, 2])
                #transforms.Normalize(mean = [ 0.485, 0.456, 0.406 ], std = [ 0.229, 0.224, 0.225 ]),
                ])
        else:
            transform = transforms.Compose([
                #transforms.Resize((resized_resol, resized_resol)),
                # transforms.CenterCrop(resol),
                transforms.ToTensor(), #implicitly divides by 255
                transforms.Normalize(mean = [0.5, 0.5, 0.5], std = [2, 2, 2])
                #transforms.Normalize(mean = [ 0.485, 0.456, 0.406 ], std = [ 0.229, 0.224, 0.225 ]),
            ])

    if get_label_free:
        target_mean = [0.485, 0.456, 0.406]
        target_std = [0.229, 0.224, 0.225]
        transform = transforms.Compose([transforms.Resize(256), transforms.CenterCrop(224),
                    transforms.ToTensor(), transforms.Normalize(mean=target_mean, std=target_std)])

    if path_transform == None:
        path_transform = lambda path: "../../../datasets/"+path
            
    dataset = CUBDataset(pkl_paths, use_attr, no_img, uncertain_label, image_dir, n_class_attr, experiment_name,transform,path_transform = path_transform,concept_restriction=concept_restriction)
            
    if is_training:
        drop_last = True

        if one_batch:
            shuffle = False 
        else:
            shuffle = True
    else:
        drop_last = False
        shuffle = False
    if resampling:
        sampler = BatchSampler(ImbalancedDatasetSampler(dataset), batch_size=batch_size, drop_last=drop_last)
        loader = DataLoader(dataset, batch_sampler=sampler)
    else:
        loader = DataLoader(dataset, batch_size=batch_size, shuffle=shuffle, drop_last=drop_last,num_workers=1)
    return loader

def find_class_imbalance(pkl_file, multiple_attr=False, attr_idx=-1):
    """
    Calculate class imbalance ratio for binary attribute labels stored XXXX-20pkl_file
    If attr_idx >= 0, then only return ratio for the corresponding attribute id
    If multiple_attr is True, then return imbalance ratio separately for each attribute. Else, calculate the overall imbalance across all attributes
    """
    imbalance_ratio = []
    data = pickle.load(open(os.path.join(BASE_DIR, pkl_file), 'rb'))
    n = len(data)
    n_attr = len(data[0]['attribute_label'])
    if attr_idx >= 0:
        n_attr = 1
    if multiple_attr:
        n_ones = [0] * n_attr
        total = [n] * n_attr
    else:
        n_ones = [0]
        total = [n * n_attr]
    for d XXXX-20data:
        labels = d['attribute_label']
        if multiple_attr:
            for i XXXX-20range(n_attr):
                n_ones[i] += labels[i]
        else:
            if attr_idx >= 0:
                n_ones[0] += labels[attr_idx]
            else:
                n_ones[0] += sum(labels)
    for j XXXX-20range(len(n_ones)):
        imbalance_ratio.append(total[j]/n_ones[j] - 1)
    if not multiple_attr: #e.g. [9.0] --> [9.0] * 312
        imbalance_ratio *= n_attr
    return imbalance_ratio
