from dataclasses import dataclass, field
from typing import Dict, Optional, Tuple
from datetime import datetime
import random
from typing import List, Dict
import itertools
import csv
import json
from dataclasses import fields
import os
import math
import numpy as np


@dataclass
class TradeProposal:
    # Core trade info
    turn_id: int
    round_id: int
    proposal_id: int
    proposer_name: str
    receiver_name: str
    chips_offered: Dict[str, int]
    chips_requested: Dict[str, int]
    timestamp: datetime = field(default_factory=datetime.now)
    status: str = "PENDING"  # e.g., 'PENDING', 'ACCEPTED', 'REJECTED'
    rationale: str = ""      # brief explanation

    ## player info
    player1 : str = ""
    player2 : str = ""
    player3 : str = ""

    # Action data: who is sender, who is chosen receiver, etc.
    player1_is_sender: bool = False
    player2_is_sender: bool = False
    player3_is_sender: bool = False
    player1_is_chosen_receiver: bool = False
    player2_is_chosen_receiver: bool = False
    player3_is_chosen_receiver: bool = False
    
    player1_offer_response: str = ""
    player2_offer_response: str = ""
    player3_offer_response: str = ""

    buy_offer_red:  int = 0
    buy_offer_green: int = 0
    sell_offer_red:  int = 0
    sell_offer_green:  int = 0

    # State data: chip counts before/after for each player & chip
    player1_red_before: int = 0
    player1_red_after: int = 0
    player2_red_before: int = 0
    player2_red_after: int = 0
    player3_red_before: int = 0
    player3_red_after: int = 0

    player1_green_before: int = 0
    player1_green_after: int = 0
    player2_green_before: int = 0
    player2_green_after: int = 0
    player3_green_before: int = 0
    player3_green_after: int = 0

    # Valuations (optional): how each player values each chip
    player1_red_value: float = 0.0
    player1_green_value: float = 0.0
    player2_red_value: float = 0.0
    player2_green_value: float = 0.0
    player3_red_value: float = 0.0
    player3_green_value: float = 0.0

    # Surplus / utility data: payouts (or utilities) before & after
    player1_payout_before: float = 0.0
    player1_payout_after: float = 0.0
    player2_payout_before: float = 0.0
    player2_payout_after: float = 0.0
    player3_payout_before: float = 0.0
    player3_payout_after: float = 0.0

    sender_payout_before: float = 0.0
    sender_payout_after: float = 0.0
    recipient_payout_before: float = 0.0
    recipient_payout_after: float = 0.0


class Player:
    def __init__(self, name: str, name_others):
        self.name = name
        self.orig_item = {
            "green": 10,
            "red": 10
        }
        self.item = dict(self.orig_item)
        self.name_others = name_others

        # A dictionary of discrete priors about each other player:
        #   self.belief_about_them[other_name] = dict((v_red,v_green) -> probability)
        self.belief_about_them = {other_name: {} for other_name in name_others}
        for v_red in range(1,11):  # Step of 0.1 to include 1.0, x10
            for v_green in range(1,11):
                for name in name_others:
                    self.belief_about_them[name][(v_red, v_green)] = 0.01

    def reinitiate(self):
        self.item = dict(self.orig_item)

    def propose_trade_randomly(self) -> (Dict[str, int], Dict[str, int]):
        """
        Return (chips_offered, chips_requested).
        """
        # pick two distinct chips at random
        chip_list = list(self.item.keys())
        if len(chip_list) < 2:
            return {}, {}
        
        chipA, chipB = random.sample(chip_list, 2)

        # Offer up to however many I hold
        max_a = self.item[chipA]
        if max_a == 0:
            # If I have no chips of type A, can't offer
            return {}, {}

        offer_amount = random.randint(1, max_a)

        # Request up to some random bound. Here, we set it to be 10
        request_amount = random.randint(1, 10)

        chips_offered = {chipA: offer_amount}
        chips_requested = {chipB: request_amount}

        return chips_offered, chips_requested
    
    def propose_trade_maximizer(self,
                                their_item: Dict[str,int],
                                belief_about_j: Dict[Tuple[float,float], float],
                                my_valuation: Dict[str, float]
                            ) -> (Dict[str, int], Dict[str, int], float):
        """
        Attempt to pick the trade (chips_offered, chips_requested) that maximizes the
        proposer's expected payoff (EV).

        Inputs:
        - their_item: opponent's inventory, to limit how many chips we can request
        - belief_about_j: discrete prior about opponent's valuations (v_red,v_green)-> probability
        - my_valuation: a dict of how *I* value each chip, e.g. {"red":3.0, "green":1.0}
        
        Returns:
        (best_x, best_y, best_expected_payoff)
        If we can't find any beneficial trade, returns ({},{}, 0.0 or negative).
        """
        import math

        best_x = {}
        best_y = {}
        best_expected_payoff = -math.inf

        # Helper to compute my utility (the proposer's utility) for a given inventory
        def calc_my_utility(inventory: Dict[str,int]) -> float:
            return (inventory["red"]   * my_valuation.get("red", 0)
                  + inventory["green"] * my_valuation.get("green", 0))

        # My current holdings
        h_i = self.item
        old_ui = calc_my_utility(h_i)

        # Build candidate lists
        all_x_candidates = []
        # For example, offering 1..N red, or 1..N green, etc. 
        # (You already have something like this, we just show the logic.)
        for x_red in range(1, h_i.get("red",0)+1):
            x_dict = {"red": x_red, "green": 0}
            all_x_candidates.append(x_dict)
        for x_green in range(1, h_i.get("green",0)+1):
            x_dict = {"red": 0, "green": x_green}
            all_x_candidates.append(x_dict)

        all_y_candidates = []
        for y_red in range(1, their_item.get("red",0)+1):
            y_dict = {"red": y_red, "green": 0}
            all_y_candidates.append(y_dict)
        for y_green in range(1, their_item.get("green",0)+1):
            y_dict = {"red": 0, "green": y_green}
            all_y_candidates.append(y_dict)

        # For each (x,y), compute my potential gain if accepted
        # Then multiply by the acceptance probability => expected payoff
        for x_dict in all_x_candidates:
            # Check if I physically have enough to offer x_dict
            if (x_dict["red"]   > h_i["red"]
            or x_dict["green"] > h_i["green"]):
                continue

            for y_dict in all_y_candidates:
                # Check if opponent physically can give y_dict (not always required,
                # but you can skip if it's impossible from 'their_item' perspective)
                if (y_dict["red"]   > their_item["red"]
                or y_dict["green"] > their_item["green"]):
                    continue

                # My new inventory if trade is accepted
                new_inv_i = {
                    "red":   h_i["red"]   - x_dict["red"]   + y_dict["red"],
                    "green": h_i["green"] - x_dict["green"] + y_dict["green"]
                }
                my_gain = calc_my_utility(new_inv_i) - old_ui

                # If my_gain <= 0 => not beneficial, skip
                if my_gain <= 0:
                    continue

                # Compute acceptance probability under 'belief_about_j'
                accept_prob = 0.0
                for (v_red, v_green), prob_vj in belief_about_j.items():
                    if prob_vj <= 0: 
                        continue
                    # Opponent's linear utility difference if they accept:
                    offered_val_j   = (x_dict["red"]   * v_red
                                    + x_dict["green"] * v_green)
                    requested_val_j = (y_dict["red"]   * v_red
                                    + y_dict["green"] * v_green)

                    if offered_val_j > requested_val_j:
                        accept_prob += prob_vj

                expected_payoff = accept_prob * my_gain

                # Update best
                if expected_payoff > best_expected_payoff:
                    best_expected_payoff = expected_payoff
                    best_x = dict(x_dict)
                    best_y = dict(y_dict)

        # If no positive payoff found, return empty
        if best_expected_payoff <= 0:
            return {}, {}, 0.0

        return best_x, best_y, best_expected_payoff


    def update_receiver_prior(self, 
                              chips_offered, 
                              chips_requested, 
                              proposer, 
                              receiver):
        """
        Called from the viewpoint of 'self' after a proposal's outcome is known.
        
        chips_offered, chips_requested are the final trade amounts.
        proposer: Player object that proposed
        receiver: Player object that accepted, or None if no one accepted.

        For a 3-player game (self, proposer, the other possible receivers):
        
        SCENARIO 1: if 'receiver' is not None
          - If self == receiver => only update prior about 'proposer'
          - If self != receiver => we are the bystander => update prior about 
            BOTH 'proposer' and 'receiver'
        
        SCENARIO 2: if 'receiver' is None
          => no one accepted => we update our prior about 'proposer' and possibly 
             the other 'non-proposer' player as well, removing states that 
             would have accepted.
        """

        was_accepted = (receiver is not None)

        if not chips_offered:
            return

        if was_accepted:
            # =============== SCENARIO 1: Some receiver accepted ===============
            if self.name == receiver.name:
                # We ARE the chosen receiver
                # => update our prior about the PROPOSER only
                old_prior = self.belief_about_them[proposer.name]
                new_prior = {}
                for (v_red,v_green), prob in old_prior.items():
                    if prob <= 0:
                        continue
                    # Would I accept if my (v_red,v_green) is this?
                    offered_val   = (chips_offered.get("red", 0) * v_red
                                   + chips_offered.get("green", 0) * v_green)
                    requested_val = (chips_requested.get("red", 0) * v_red
                                   + chips_requested.get("green", 0) * v_green)
                    if offered_val > requested_val:
                        new_prior[(v_red,v_green)] = new_prior.get((v_red,v_green),0) + prob
                # renormalize
                total = sum(new_prior.values())
                if total>1e-15:
                    for k in new_prior:
                        new_prior[k] /= total
                self.belief_about_them[proposer.name] = new_prior

            else:
                # We are neither the proposer nor the chosen receiver => 'bystander'
                # => update about BOTH the 'proposer' and the 'receiver'.
                
                # 1) Update about the PROPOSER
                old_prop_prior = self.belief_about_them[proposer.name]
                new_prop_prior = {}
                # We might keep states that "predict" they'd do a deal that the receiver would accept.
                # Minimal example: we do no big filtering, or replicate logic. 
                # We'll do partial acceptance logic for demonstration:
                for (v_red,v_green), prob in old_prop_prior.items():
                    # Possibly check if that v_red, v_green "makes sense" to do this trade
                    # We'll skip detailed logic:
                    new_prop_prior[(v_red,v_green)] = new_prop_prior.get((v_red,v_green),0)+prob
                # Normalize
                tp = sum(new_prop_prior.values())
                if tp>1e-15:
                    for k in new_prop_prior:
                        new_prop_prior[k] /= tp
                self.belief_about_them[proposer.name] = new_prop_prior

                # 2) Update about the RECEIVER
                old_recv_prior = self.belief_about_them[receiver.name]
                new_recv_prior = {}
                for (v_red,v_green), prob in old_recv_prior.items():
                    if prob <= 0:
                        continue
                    # Keep only states that "would accept"
                    offered_val   = (chips_offered.get("red", 0) * v_red
                                   + chips_offered.get("green", 0) * v_green)
                    requested_val = (chips_requested.get("red", 0) * v_red
                                   + chips_requested.get("green", 0) * v_green)
                    if offered_val > requested_val:
                        new_recv_prior[(v_red,v_green)] = new_recv_prior.get((v_red,v_green),0)+prob
                tr = sum(new_recv_prior.values())
                if tr>1e-15:
                    for k in new_recv_prior:
                        new_recv_prior[k] /= tr
                self.belief_about_them[receiver.name] = new_recv_prior

        else:
            # =============== SCENARIO 2: No one accepted ===============
            # That means neither of the 2 potential receivers found it beneficial
            # => we remove states from our belief about the proposer or other players
            #    that would accept.

            # Example: We at least remove from the PROPOSER any states that 
            # would lead to guaranteed acceptance? This depends on your design.
            old_prop_prior = self.belief_about_them[proposer.name]
            new_prop_prior = {}
            for (v_red,v_green), prob in old_prop_prior.items():
                offered_val   = (chips_offered.get("red", 0) * v_red
                               + chips_offered.get("green", 0) * v_green)
                requested_val = (chips_requested.get("red", 0) * v_red
                               + chips_requested.get("green", 0) * v_green)
                would_accept = (offered_val > requested_val)
                # if no one accepted, keep states that do NOT accept
                if not would_accept:
                    new_prop_prior[(v_red,v_green)] = new_prop_prior.get((v_red,v_green),0)+prob
            totp = sum(new_prop_prior.values())
            if totp>1e-15:
                for k in new_prop_prior:
                    new_prop_prior[k] /= totp
            self.belief_about_them[proposer.name] = new_prop_prior

            # Potentially also update the "other" player. If there's exactly 3 players, 
            # we can figure out who is the third. Let's skip details or do similarly 
            # removing states that "would accept."

    def update_proposer_prior(self, 
                              chips_offered, 
                              chips_requested, 
                              receiver):
        """
        Called from the viewpoint of 'self' = the proposer after final outcome.
        If 'receiver' is not None => only that one accepted => update only that receiver's distribution.
        If 'receiver' is None => no one accepted => update distribution for *both* potential receivers
        in a 3-player game.
        """
        was_accepted = (receiver is not None)
        if not chips_offered:
            return
        if was_accepted:
            # SCENARIO 1: a single known receiver accepted
            old_prior = self.belief_about_them[receiver.name]
            new_prior = {}
            for (v_red,v_green), prob in old_prior.items():
                if prob <= 0:
                    continue
                offered_val   = (chips_offered.get("red", 0) * v_red
                               + chips_offered.get("green", 0) * v_green)
                requested_val = (chips_requested.get("red", 0) * v_red
                               + chips_requested.get("green", 0) * v_green)
                # keep only "would accept" states
                if offered_val > requested_val:
                    new_prior[(v_red,v_green)] = new_prior.get((v_red,v_green),0)+prob

            total = sum(new_prior.values())
            if total>1e-15:
                for k in new_prior:
                    new_prior[k] /= total
            self.belief_about_them[receiver.name] = new_prior

        else:
            # SCENARIO 2: no receiver => both potential receivers refused
            # => update for each of them in self.name_others 
            #    remove states that would accept
            for possible_recv_name in self.name_others:
                old_prior = self.belief_about_them[possible_recv_name]
                new_prior = {}
                for (v_red,v_green), prob in old_prior.items():
                    if prob <= 0:
                        continue
                    offered_val   = (chips_offered.get("red", 0) * v_red
                                   + chips_offered.get("green", 0) * v_green)
                    requested_val = (chips_requested.get("red", 0) * v_red
                                   + chips_requested.get("green", 0) * v_green)
                    if not (offered_val > requested_val):
                        new_prior[(v_red,v_green)] = new_prior.get((v_red,v_green),0)+prob
                tot = sum(new_prior.values())
                if tot>1e-15:
                    for k in new_prior:
                        new_prior[k] /= tot
                self.belief_about_them[possible_recv_name] = new_prior
    
    def normalize_belief(self):
        for name in self.name_others:
            total = sum(self.belief_about_them[name].values())
            if total > 0:
                for k in self.belief_about_them[name]:
                    self.belief_about_them[name][k] /= total
    
    def propose_trade_heuristically(self, other_players: List['Player'], valuations: Dict[str, Dict[str, float]], aggressiveness: float = 1.0) -> (Dict[str, int], Dict[str, int]):
        """
        Proposes a trade that maximizes this player's utility subject to an aggressiveness constraint on the ratio of chips offered to received.
        """
        my_chips = self.item.copy()
        my_utility = valuations[self.name]

        best_trade = None
        best_trade_value = 0

        for give_type in my_chips:
            for get_type in my_chips:
                if give_type != get_type and my_chips[give_type] > 0:
                    for other_player in other_players:
                        if other_player == self:
                            continue  # Don't trade with myself

                        if other_player.item[get_type] > 0:  # Ensure they have chips
                            for give_qty in range(1, int(my_chips[give_type]) + 1):
                                # Calculate the maximum get_qty based on aggressiveness and available chips
                                max_get_qty = min(int(other_player.item[get_type]), int(give_qty * aggressiveness))  #Take the smaller value out of the aggressiveness condition and the available chips of that type.
                                for get_qty in range(1, max_get_qty + 1):
                                    trade_value = (get_qty * my_utility[get_type]) - (give_qty * my_utility[give_type])

                                    if trade_value > best_trade_value:
                                        best_trade = ({give_type: give_qty}, {get_type: get_qty})
                                        best_trade_value = trade_value
                                    

        if best_trade is not None:
            return best_trade[0], best_trade[1] # Return chips_offered, chips_requested
        else:  # Return empty dicts so the higher level function can handle the no trade case.
            return {}, {}

    def calculate_utility(self, items: dict, valuations: dict) -> float:
        """Utility = sum of (chip_count * chip_valuation)."""
        return sum(items.get(chip, 0) * valuations.get(chip, 0) 
                   for chip in items.keys())

    def has_sufficient_chips(self, chips_needed: dict) -> bool:
        for chip, qty in chips_needed.items():
            if self.item.get(chip, 0) < qty:
                return False
        return True
    
    def execute_trade(self, 
                      chips_offered: Dict[str, int], 
                      chips_requested: Dict[str, int],
                      other_player: 'Player'):
        """
        Adjust inventories for both parties if the trade is accepted.
        """
        # self loses chips_offered, gains chips_requested
        for chip, amt in chips_offered.items():
            self.item[chip] -= amt
            other_player.item[chip] += amt
            #   Check if any item becomes negative 
            if self.item[chip] < 0:
                raise ValueError(f"Insufficient {chip} for self after transaction")
        
        # other_player loses chips_requested, gains chips_offered
        for chip, amt in chips_requested.items():
            other_player.item[chip] -= amt
            self.item[chip] += amt

            # Check if any item becomes negative for other_player
            if other_player.item[chip] < 0:
                raise ValueError(f"Insufficient {chip} for other player after transaction")

class Negotiation:
    def __init__(self, players: List[Player], valuations: Dict[str, Dict[str, float]]):
        """
        players: list of Player objects
        valuations: e.g. {'Alice': {'green': 1.0, 'red': 3.0}, ...}
        """
        self.players = players
        self.valuations = valuations
        self.trade_history: List[TradeProposal] = []
        self.proposal_counter = 0
    
    def reinitiate_all(self):
        for p in self.players:
            p.reinitiate()
        self.trade_history.clear()
        self.proposal_counter = 0


    def scenario_statistical_proposals_rational_accepts(self, sequence, seed=None):
        """
        Scenario 5 Variation:
        - One proposer tries to create a 'rational' trade for themselves based on their prior for their trader's valuation.
        - All other players evaluate if it's beneficial for them. 
          (If multiple can accept, choose one randomly.)
        - If no one can accept, the trade fails.
        """
        if seed is not None:
            random.seed(seed)

        round_id = 0
        for counting, i1 in enumerate(sequence):
            if counting % 3 == 0 and counting != 0:  # Increment round_id every 3 iterations (after the first three)
                round_id += 1
            proposer = self.players[i1]
            proposer_val = self.valuations.get(proposer.name, {})

            # 1) Capture pre-trade states
            before_items = {p.name: dict(p.item) for p in self.players}
            before_utilities = {
                p.name: p.calculate_utility(p.item, self.valuations[p.name]) 
                for p in self.players
            }

            # Step 2: Attempt to find the maximum expected utility proposal for the proposer
            other_candidates = [p for p in self.players if p != proposer]
                    # We'll store the best among multiple opponents
            best_surplus_all = 0.0
            chips_offered, chips_requested = {}, {}
            chosen_opponent = None

            for opponent in other_candidates:
                # 1) Grab the proposer's belief about that opponent:
                belief_about_opponent = proposer.belief_about_them[opponent.name]
                # 2) Also fetch the proposer's own valuations
                my_valuation = self.valuations[proposer.name]

                # 3) Call propose_trade_maximizer
                x_proposal, y_proposal, expected_surplus = proposer.propose_trade_maximizer(
                    their_item=opponent.item,
                    belief_about_j=belief_about_opponent,
                    my_valuation=my_valuation
                )

                if x_proposal and y_proposal:
                    # If it's not empty => we found a beneficial trade
                    if expected_surplus > best_surplus_all:
                        best_surplus_all = expected_surplus
                        chips_offered, chips_requested = x_proposal, y_proposal
                        chosen_opponent = opponent
            
            # Now best_chips_offered, best_chips_requested is the top trade across all possible opponents
            if not chosen_opponent or best_surplus_all <= 0:
                # No beneficial trade found with any opponent
                # continue
                rationale = "No one accepted"

            # Step 3: Among the other players, see who is willing to accept
            possible_acceptors = []
            receiver = None
            accepted = False
            for p in self.players:
                if p == proposer:
                    continue
                # Player p must have sufficient chips
                if not p.has_sufficient_chips(chips_requested):
                    continue
                # Check if p's utility improves
                p_vals = self.valuations.get(p.name, {})
                if self.does_receiver_gain(p, chips_offered, chips_requested, p_vals):
                    possible_acceptors.append(p)
            
            # 4) If multiple accept, pick one randomly
            if len(possible_acceptors) > 1:
                receiver = random.choice(possible_acceptors)
                rationale = "Multiple rational acceptors; chosen randomly"
            elif len(possible_acceptors) == 1:
                receiver = possible_acceptors[0]
                rationale = "Single random acceptor"
            else:
                rationale = "No one accepted"
            
            if len(possible_acceptors)>0:
                proposer.execute_trade(chips_offered, chips_requested, receiver)
            
            # 5) Execute trade
            after_items = {p.name: dict(p.item) for p in self.players}
            after_utilities = {
                p.name: p.calculate_utility(p.item, self.valuations[p.name]) 
                for p in self.players
            }

            ## update all players' posteriors
            self.update_bidder(proposer, chips_offered, chips_requested, receiver)
            
            # 6) Record
            self.record_trade(
                turn_id=i1,
                round_id = round_id,
                proposer_name=proposer.name,
                receiver_name=receiver.name if len(possible_acceptors) > 0 else "No one",
                chips_offered=chips_offered,
                chips_requested=chips_requested,
                status="ACCEPTED" if len(possible_acceptors) > 0 else "REJECTED",
                rationale=rationale,
                possible_acceptors=possible_acceptors,
                before_items=before_items,
                after_items=after_items,
                before_utilities=before_utilities,
                after_utilities=after_utilities
            )
    def update_bidder(self, proposer, chips_offered, chips_requested, receiver):
        """
        This method is called after a trade attempt to update everyone's priors.

        SCENARIO 1: if 'receiver' is not None (someone accepted).
          1) proposer.update_proposer_prior(...)  # only about that chosen receiver
          2) receiver.update_receiver_prior(...)   # about the proposer
          3) all other players update_receiver_prior(...)  # about both proposer & receiver

        SCENARIO 2: if 'receiver' is None (no one accepted).
          - proposer.update_proposer_prior(...) for all possible receivers
          - all other players update_receiver_prior(...) about the proposer and each other 
        """
        if receiver:
            # SCENARIO 1
            proposer.update_proposer_prior(chips_offered, chips_requested, receiver)
            receiver.update_receiver_prior(chips_offered, chips_requested, proposer, receiver)

            for p in self.players:
                if p is proposer or p is receiver:
                    continue
                # the "other" player updates about both
                p.update_receiver_prior(chips_offered, chips_requested, proposer, receiver)

        else:
            # SCENARIO 2: no acceptance
            proposer.update_proposer_prior(chips_offered, chips_requested, receiver=None)

            for p in self.players:
                if p is proposer:
                    continue
                # each other player updates about the proposer and also about each other if needed
                p.update_receiver_prior(chips_offered, chips_requested, proposer, receiver=None)

    
    def does_proposer_gain(self, 
                           proposer: Player,
                           chips_offered: dict,
                           chips_requested: dict,
                           proposer_val: dict) -> bool:
        """
        Return True if the proposer's utility increases by making this trade.
        """
        current_utility = proposer.calculate_utility(proposer.item, proposer_val)
        hypothetical = dict(proposer.item)

        # Subtract offered, add requested
        for chip, amt in chips_offered.items():
            if hypothetical[chip] < amt:
                # If we don't have enough to offer, can't do it
                return False
            hypothetical[chip] -= amt
        
        for chip, amt in chips_requested.items():
            hypothetical[chip] += amt
        
        new_utility = proposer.calculate_utility(hypothetical, proposer_val)
        return new_utility > current_utility
    
    def does_receiver_gain(self,
                           receiver: Player,
                           chips_offered: dict,
                           chips_requested: dict,
                           receiver_val: dict) -> bool:
        """
        Return True if the receiver's utility increases by accepting the trade.
        """
        # Current
        current_utility = receiver.calculate_utility(receiver.item, receiver_val)

        hypothetical = dict(receiver.item)
        # Gains chips_offered, loses chips_requested
        for chip, amt in chips_offered.items():
            hypothetical[chip] += amt
        for chip, amt in chips_requested.items():
            if hypothetical[chip] < amt:
                # insufficient in hypothetical
                return False
            hypothetical[chip] -= amt
        
        new_utility = receiver.calculate_utility(hypothetical, receiver_val)
        return new_utility > current_utility
    
    def record_trade(self,
                    turn_id: int,
                    round_id: int,
                    proposer_name: str,
                    receiver_name: str,
                    chips_offered: dict,
                    chips_requested: dict,
                    status: str,
                    rationale: str,
                    possible_acceptors: list,
                    before_items: Dict[str, Dict[str, int]] = None,
                    after_items: Dict[str, Dict[str, int]] = None,
                    before_utilities: Dict[str, float] = None,
                    after_utilities: Dict[str, float] = None):

        # Convenient lookups for each player's name
        p1_name = self.players[0].name
        p2_name = self.players[1].name
        p3_name = self.players[2] if len(self.players) > 2 else None
        p3_name = p3_name.name if p3_name else "None"

        # Utility lookups: if dict is None, default them
        if before_items is None:
            before_items = {p.name: dict(p.item) for p in self.players}
        if after_items is None:
            after_items = {p.name: dict(p.item) for p in self.players}
        if before_utilities is None:
            before_utilities = {
                p.name: p.calculate_utility(p.item, self.valuations[p.name])
                for p in self.players
            }
        if after_utilities is None:
            after_utilities = {
                p.name: p.calculate_utility(p.item, self.valuations[p.name])
                for p in self.players
            }

        # Identify who is sender, who is chosen recipient, etc.
        player1_is_sender = (proposer_name == p1_name)
        player2_is_sender = (proposer_name == p2_name)
        player3_is_sender = (proposer_name == p3_name)

        player1_is_selected_receiver = (receiver_name == p1_name)
        player2_is_selected_receiver = (receiver_name == p2_name)
        player3_is_selected_receiver = (receiver_name == p3_name)

        # Build the proposal object
        proposal = TradeProposal(
            turn_id = turn_id,
            round_id = round_id,
            proposal_id = self.proposal_counter,
            proposer_name = proposer_name,
            receiver_name = receiver_name,
            chips_offered = chips_offered,
            chips_requested = chips_requested,
            status = status,
            rationale = rationale,

            player1 = p1_name,      
            player2 = p2_name,
            player3 = p3_name,

            # Mark which player is sender / receiver
            player1_is_sender = player1_is_sender,
            player2_is_sender = player2_is_sender,
            player3_is_sender = player3_is_sender,
            player1_is_chosen_receiver = player1_is_selected_receiver,
            player2_is_chosen_receiver = player2_is_selected_receiver,
            player3_is_chosen_receiver = player3_is_selected_receiver,

            player1_offer_response = True if p1_name in [p.name for p in possible_acceptors] else False,
            player2_offer_response = True if p2_name in [p.name for p in possible_acceptors] else False,
            player3_offer_response = True if p3_name in [p.name for p in possible_acceptors] else False,

            # Optionally set buy/sell flags. For instance,
            # "Buy offer (RED)" means the proposer is requesting red chips. 
            buy_offer_red = chips_requested.get("red", 0),
            buy_offer_green = chips_requested.get("green", 0),
            sell_offer_red = chips_offered.get("red", 0),
            sell_offer_green = chips_offered.get("green", 0),

            # Before/after states for each player & chip
            player1_red_before = before_items[p1_name].get("red", 0),
            player1_red_after  = after_items[p1_name].get("red", 0),
            player2_red_before = before_items[p2_name].get("red", 0),
            player2_red_after  = after_items[p2_name].get("red", 0),
            player3_red_before = before_items[p3_name].get("red", 0) if p3_name != "None" else 0,
            player3_red_after  = after_items[p3_name].get("red", 0) if p3_name != "None" else 0,

            player1_green_before = before_items[p1_name].get("green", 0),
            player1_green_after  = after_items[p1_name].get("green", 0),
            player2_green_before = before_items[p2_name].get("green", 0),
            player2_green_after  = after_items[p2_name].get("green", 0),
            player3_green_before = before_items[p3_name].get("green", 0) if p3_name != "None" else 0,
            player3_green_after  = after_items[p3_name].get("green", 0) if p3_name != "None" else 0,

            # Valuations
            player1_red_value = self.valuations[p1_name].get("red", 0),
            player1_green_value = self.valuations[p1_name].get("green", 0),

            player2_red_value = self.valuations[p2_name].get("red", 0),
            player2_green_value = self.valuations[p2_name].get("green", 0),

            player3_red_value = self.valuations[p3_name].get("red", 0) if p3_name != "None" else 0,
            player3_green_value = self.valuations[p3_name].get("green", 0) if p3_name != "None" else 0,

            # Surplus / payout data
            player1_payout_before = before_utilities[p1_name],
            player1_payout_after  = after_utilities[p1_name],
            player2_payout_before = before_utilities[p2_name],
            player2_payout_after  = after_utilities[p2_name],
            player3_payout_before = before_utilities[p3_name] if p3_name != "None" else 0,
            player3_payout_after  = after_utilities[p3_name] if p3_name != "None" else 0,

            sender_payout_before = before_utilities[proposer_name],
            sender_payout_after = after_utilities[proposer_name],
            recipient_payout_before = before_utilities[receiver_name] if status == "ACCEPTED" else None,
            recipient_payout_after = after_utilities[receiver_name] if status == "ACCEPTED" else None,

        )

        # Increment ID counter and save
        self.proposal_counter += 1
        self.trade_history.append(proposal)


    def save_trade_history_to_csv(self, csv_path: str, cohort_name: str, stage_name: str, mode: str):
        """
        Write all fields from each TradeProposal in self.trade_history to CSV.
        """
        original_field_names = [f.name for f in fields(TradeProposal)]  # from dataclasses import fields
        field_names = ['mode','cohort_name', 'stage_name'] + original_field_names  # Place new fields first

        # rescale the data
        fields_to_rescale = [
            "player1_red_value", "player1_green_value",
            "player2_red_value", "player2_green_value",
            "player3_red_value", "player3_green_value",
            "player1_payout_before", "player1_payout_after",
            "player2_payout_before", "player2_payout_after",
            "player3_payout_before", "player3_payout_after",
            "sender_payout_before", "sender_payout_after",
            "recipient_payout_before", "recipient_payout_after"
        ]
        
        # use deepcopy to avoid modifying the original data
        import copy
        trade_history_rescaled = copy.deepcopy(self.trade_history)
        
        # rescale the data
        for proposal in trade_history_rescaled:
            for field in fields_to_rescale:
                value = getattr(proposal, field)
                if value is not None:  #
                    setattr(proposal, field, value / 10.0)

        # Check if the file already exists
        file_exists = os.path.exists(csv_path)

        with open(csv_path, mode='a', newline='', encoding='utf-8') as f:
            writer = csv.DictWriter(f, fieldnames=field_names)

            # Only write the header if the file doesn't exist
            if not file_exists:
                writer.writeheader()

            for proposal in trade_history_rescaled:
                # Convert the TradeProposal dataclass to a dict
                row_dict = {
                field_name: getattr(proposal, field_name)
                    for field_name in field_names if field_name not in ['mode','cohort_name', 'stage_name']
                }
                # Add cohort_name and stage_name to the row
                row_dict['mode'] = mode
                row_dict['cohort_name'] = cohort_name
                row_dict['stage_name'] = stage_name
                writer.writerow(row_dict)

        
def process_negotiations(json_file_path: str, output_csv_path: str, seed=None):
    """
    Reads a JSON file containing negotiation game data, extracts valuations,
    runs the negotiation ordering analysis, and writes results to a CSV.
    """
    if seed is not None:
            random.seed(seed)
    ## reset seed
    seed = random.randint(0, 100)

    with open(json_file_path, 'r') as f:
        data = json.load(f)

    # Iterate over all games in the JSON
    for i, game in enumerate(data["games"]):
        # 1. Extract the cohort name
        cohort_name = game.get("cohortName", "unknown_cohort")
        stage_name = game.get("stageName", "unknown_stage")

        # 2. Extract participantChipValueMap and players
        metadata = game["data"]["metadata"]
        participant_map = metadata["participantChipValueMap"]
        players_name = metadata["players"]

        ## focus on the case where we have 3 players
        if len(players_name)<=2:
            continue

        print("=================================")
        print('players',cohort_name, stage_name)
        # 3. Build valuations in the order of `players`, for goods = [RED, GREEN]
        valuations = {}
        for p in players_name:
            color_map = participant_map[p]
            # Force the goods order: [RED, GREEN]
            valuations[p] = {
                                "red":int(color_map["RED"]*10),
                                "green": int(color_map["GREEN"]*10)
                            }

        # 
        if len(players_name) == 3:
            sequence = [0,1,2,0,1,2,0,1,2]

        else:
            sequence = [0,1,0,1,0,1,0,1,0]

        #4. Build the negotiation environment

        # Create some players
        players = []
        for player in players_name:
            name_others = [p for p in players_name if p != player]  # Exclude the current player's name
            players.append(Player(name=player, name_others=name_others))

        negotiation = Negotiation(players, valuations)
        
        mode = 'stat-learn'
        negotiation.scenario_statistical_proposals_rational_accepts(sequence=sequence, seed=seed)
        negotiation.save_trade_history_to_csv(output_csv_path, cohort_name=cohort_name, stage_name=stage_name, mode=mode)
    

if __name__ == "__main__":

    json_file = ... ## Value file
    output_csv = ... 

    process_negotiations(json_file, output_csv, seed= 42)

