import math
import numpy as np
import re

import torch
from torch import nn
import torch.nn.functional as F
from torch import distributions as torchd

import tools


class RSSM(nn.Module):
    def __init__(
        self,
        stoch=30,
        deter=200,
        hidden=200,
        layers_input=1,
        layers_output=1,
        rec_depth=1,
        shared=False,
        discrete=False,
        act="SiLU",
        norm="LayerNorm",
        mean_act="none",
        std_act="softplus",
        temp_post=True,
        min_std=0.1,
        cell="gru",
        unimix_ratio=0.01,
        initial="learned",
        num_actions=None,
        embed=None,
        device=None,
        c_stoch=None,
        c_discrete=None,
        c_deter=None,
    ):
        super(RSSM, self).__init__()
        self._stoch = stoch
        self._deter = deter
        self._hidden = hidden
        self._min_std = min_std
        self._layers_input = layers_input
        self._layers_output = layers_output
        self._rec_depth = rec_depth
        self._shared = shared
        self._discrete = discrete
        act = getattr(torch.nn, act)
        norm = getattr(torch.nn, norm)
        self._mean_act = mean_act
        self._std_act = std_act
        self._temp_post = temp_post
        self._unimix_ratio = unimix_ratio
        self._initial = initial
        self._embed = embed
        self._device = device

        self._c_stoch = c_stoch
        self._c_discrete = c_discrete
        self._c_deter = c_deter
        self._c_layers_input = self._layers_input
        self._c_hidden = self._hidden

        self._z_infer_hidden = self._c_hidden

        z_inp_layers = []
        if self._discrete:
            z_h_inp_dim = self._stoch * self._discrete + num_actions
        else:
            raise NotImplementedError(f"self._discrete={self._discrete}")
            # inp_dim = self._stoch + num_actions
        if self._shared:
            raise NotImplementedError(f"self._shared={self._shared}")
            # inp_dim += self._embed
        for i in range(self._layers_input):
            z_inp_layers.append(nn.Linear(z_h_inp_dim, self._hidden, bias=False))
            z_inp_layers.append(norm(self._hidden, eps=1e-03))
            z_inp_layers.append(act())
            if i == 0:
                z_h_inp_dim = self._hidden
        self._z_h_inp_layers = nn.Sequential(*z_inp_layers)
        self._z_h_inp_layers.apply(tools.weight_init)

        c_inp_layers = []
        if self._c_discrete:
            c_h_inp_dim = self._c_stoch * self._c_discrete
        else:
            raise NotImplementedError(f"self._discrete={self._c_discrete}")
            # inp_dim = self._stoch + num_actions
        if self._shared:
            raise NotImplementedError(f"self._shared={self._shared}")
            # inp_dim += self._embed
        for i in range(self._c_layers_input):
            c_inp_layers.append(nn.Linear(c_h_inp_dim, self._c_hidden, bias=False))
            c_inp_layers.append(norm(self._c_hidden, eps=1e-03))
            c_inp_layers.append(act())
            if i == 0:
                c_h_inp_dim = self._c_hidden
        self._c_h_inp_layers = nn.Sequential(*c_inp_layers)
        self._c_h_inp_layers.apply(tools.weight_init)

        if cell == "gru":
            self._z_cell = GRUCell(self._hidden, self._deter)
            self._z_cell.apply(tools.weight_init)
            self._c_cell = GRUCell(self._c_hidden, self._c_deter)
            self._c_cell.apply(tools.weight_init)
        elif cell == "gru_layer_norm":
            self._z_cell = GRUCell(self._hidden, self._deter, norm=True)
            self._z_cell.apply(tools.weight_init)
            self._c_cell = GRUCell(self._c_hidden, self._c_deter, norm=True)
            self._c_cell.apply(tools.weight_init)
        else:
            raise NotImplementedError(cell)

        img_h_z_out_layers = []
        img_h_z_inp_dim = self._deter
        for i in range(self._layers_output):
            img_h_z_out_layers.append(nn.Linear(img_h_z_inp_dim, self._hidden, bias=False))
            img_h_z_out_layers.append(norm(self._hidden, eps=1e-03))
            img_h_z_out_layers.append(act())
            if i == 0:
                img_h_z_inp_dim = self._hidden
        self._img_h_z_out_layers = nn.Sequential(*img_h_z_out_layers)
        self._img_h_z_out_layers.apply(tools.weight_init)

        img_h_c_out_layers = []
        img_h_c_inp_dim = self._c_deter
        for i in range(self._c_layers_input):
            img_h_c_out_layers.append(nn.Linear(img_h_c_inp_dim, self._c_hidden, bias=False))
            img_h_c_out_layers.append(norm(self._c_hidden, eps=1e-03))
            img_h_c_out_layers.append(act())
            if i == 0:
                img_h_c_inp_dim = self._c_hidden
        self._img_h_c_out_layers = nn.Sequential(*img_h_c_out_layers)
        self._img_h_c_out_layers.apply(tools.weight_init)


        obs_h_z_out_layers = []
        if self._temp_post:
            obs_h_z_inp_dim = self._deter + self._c_deter + self._c_stoch * self._c_discrete + self._embed  # (h_z, h_c, c, o)

        else:
            raise NotImplementedError(f"self._temp_post={self._temp_post}")
            # inp_dim = self._embed
        for i in range(self._layers_output + 1):
            obs_h_z_out_layers.append(nn.Linear(obs_h_z_inp_dim, self._hidden, bias=False))
            obs_h_z_out_layers.append(norm(self._hidden, eps=1e-03))
            obs_h_z_out_layers.append(act())
            if i == 0:
                obs_h_z_inp_dim = self._hidden
        self._obs_h_z_out_layers = nn.Sequential(*obs_h_z_out_layers)
        self._obs_h_z_out_layers.apply(tools.weight_init)

        obs_h_c_out_layers = []
        if self._temp_post:
            obs_h_c_inp_dim = self._deter + self._c_deter + self._embed    # (h_z, h_c, o)
        else:
            raise NotImplementedError(f"self._temp_post={self._temp_post}")
            # inp_dim = self._embed
        for i in range(self._layers_output + 1):
            obs_h_c_out_layers.append(nn.Linear(obs_h_c_inp_dim, self._c_hidden, bias=False))
            obs_h_c_out_layers.append(norm(self._c_hidden, eps=1e-03))
            obs_h_c_out_layers.append(act())
            if i == 0:
                obs_h_c_inp_dim = self._c_hidden
        self._obs_h_c_out_layers = nn.Sequential(*obs_h_c_out_layers)
        self._obs_h_c_out_layers.apply(tools.weight_init)


        self.z_infer_out_layers = nn.Sequential(*obs_h_c_out_layers)
        self.z_infer_out_layers.apply(tools.weight_init)


        if self._discrete:
            self._ims_h_z_stat_layer = nn.Linear(self._hidden, self._stoch * self._discrete)
            self._ims_h_z_stat_layer.apply(tools.weight_init)
            self._obs_h_z_stat_layer = nn.Linear(self._hidden, self._stoch * self._discrete)
            self._obs_h_z_stat_layer.apply(tools.weight_init)

            self._ims_h_c_stat_layer = nn.Linear(self._c_hidden, self._c_stoch * self._c_discrete)
            self._ims_h_c_stat_layer.apply(tools.weight_init)
            self._obs_h_c_stat_layer = nn.Linear(self._c_hidden, self._c_stoch * self._c_discrete)
            self._obs_h_c_stat_layer.apply(tools.weight_init)

            self._z_infer_stat_layer = nn.Linear(self._z_infer_hidden, self._stoch * self._discrete)
            self._z_infer_stat_layer.apply(tools.weight_init)
        else:
            raise NotImplementedError

    def initial(self, batch_size):
        if self._discrete:
            state = dict(
                z_logit=torch.zeros([batch_size, self._stoch, self._discrete], device=self._device),
                z_stoch=torch.zeros([batch_size, self._stoch, self._discrete], device=self._device),
                z_deter=torch.zeros(batch_size, self._deter, device=self._device),
                c_logit=torch.zeros([batch_size, self._c_stoch, self._c_discrete], device=self._device),
                c_stoch=torch.zeros([batch_size, self._c_stoch, self._c_discrete], device=self._device),
                c_deter=torch.zeros(batch_size, self._c_deter, device=self._device),
                z_infer_logit=torch.zeros([batch_size, self._stoch, self._discrete], device=self._device),
            )
        else:
            raise  NotImplementedError
        if self._initial == "zeros":
            return state
        else:
            raise NotImplementedError(self._initial)

    def observe(self, embed, action, is_first, state=None):
        swap = lambda x: x.permute([1, 0] + list(range(2, len(x.shape))))
        if state is None:
            state = self.initial(action.shape[0])
        # (batch, time, ch) -> (time, batch, ch)
        embed, action, is_first = swap(embed), swap(action), swap(is_first)
        # prev_state[0] means selecting posterior of return(posterior, prior) from obs_step
        post, prior = tools.static_scan(
            lambda prev_state, prev_act, embed, is_first: self.obs_step(
                prev_state[0], prev_act, embed, is_first
            ),
            (action, embed, is_first),
            (state, state),
        )

        # (time, batch, stoch, discrete_num) -> (batch, time, stoch, discrete_num)
        post = {k: swap(v) for k, v in post.items()}
        prior = {k: swap(v) for k, v in prior.items()}
        return post, prior

    def imagine(self, action, state=None):
        swap = lambda x: x.permute([1, 0] + list(range(2, len(x.shape))))
        if state is None:
            state = self.initial(action.shape[0])
        assert isinstance(state, dict), state
        action = action
        action = swap(action)
        prior = tools.static_scan(self.z_img_step, [action], state)
        prior = prior[0]
        prior = {k: swap(v) for k, v in prior.items()}
        return prior

    def get_feat(self, state, latent_type):
        z_stoch = state["z_stoch"]
        if self._discrete:
            shape = list(z_stoch.shape[:-2]) + [self._stoch * self._discrete]
            z_stoch_rsp = z_stoch.reshape(shape)
            if latent_type=="latent_z":
                return torch.cat([z_stoch_rsp, state["z_deter"]], -1)
            elif latent_type=="latent_z_latent_c":
                c_stoch = state["c_stoch"]
                shape = list(c_stoch.shape[:-2]) + [self._c_stoch * self._c_discrete]
                c_stoch_rsp = c_stoch.reshape(shape)
                return torch.cat([z_stoch_rsp, state["z_deter"],
                                  c_stoch_rsp, state["c_deter"]], -1)
            else:
                raise NotImplementedError
        raise NotImplementedError

    def get_dist(self, state, latent_type, dtype=None):
        if self._discrete:
            if latent_type == "latent_z":
                logit = state["z_logit"]
            elif latent_type == "latent_c":
                logit = state["c_logit"]
            else:
                raise NotImplementedError(f"latent_type={latent_type}")
            dist = torchd.independent.Independent(
                tools.OneHotDist(logit, unimix_ratio=self._unimix_ratio), 1
            )
        else:
            raise NotImplementedError(f"self._discrete={self._discrete}")
            # mean, std = state["mean"], state["std"]
            # dist = tools.ContDist(
            #     torchd.independent.Independent(torchd.normal.Normal(mean, std), 1)
            # )
        return dist

    def obs_step(self, prev_state, prev_action, embed, is_first, sample=True):
        # infer_train: use infer network
        # if shared is True, prior and post both use same networks(inp_layers, _img_out_layers, _ims_stat_layer)
        # otherwise, post use different network(_obs_out_layers) with prior[deter] and embed as inputs
        prev_action *= (1.0 / torch.clip(torch.abs(prev_action), min=1.0)).detach()

        if torch.sum(is_first) > 0:
            is_first = is_first[:, None]
            prev_action *= 1.0 - is_first
            init_state = self.initial(len(is_first))
            for key, val in prev_state.items():
                is_first_r = torch.reshape(
                    is_first,
                    is_first.shape + (1,) * (len(val.shape) - len(is_first.shape)),
                )
                prev_state[key] = (
                    val * (1.0 - is_first_r) + init_state[key] * is_first_r
                )

        z_prior = self.z_img_step(prev_state, prev_action, None, sample)
        c_prior = self.c_img_step(prev_state, None, sample)
        prior = {
            "z_logit": z_prior["z_logit"],
            "z_stoch": z_prior["z_stoch"],
            "z_deter": z_prior["z_deter"],
            "c_logit": c_prior["c_logit"],
            "c_stoch": c_prior["c_stoch"],
            "c_deter": c_prior["c_deter"],
        }

        if self._shared:
            raise NotImplementedError(f"self._shared={self._shared}")
        assert self._temp_post

        c_input = torch.cat([prior["z_deter"], prior["c_deter"], embed], -1)   # (h_z, h_c, o)
        c_dist_input = self._obs_h_c_out_layers(c_input)
        c_stats = self._suff_stats_layer("obs", latent_type="latent_c", x=c_dist_input)
        if sample:
            c_stoch = self.get_dist(c_stats, latent_type="latent_c").sample()
        else:
            c_stoch = self.get_dist(c_stats, latent_type="latent_c").mode()
        c_post = {"c_stoch": c_stoch, "c_deter": prior["c_deter"], **c_stats}

        shape = list(c_stoch.shape[:-2]) + [self._c_stoch * self._c_discrete]
        c_stoch_rp = c_stoch.reshape(shape)
        z_input = torch.cat([prior["z_deter"], prior["c_deter"], c_stoch_rp, embed], -1) # (h_z, h_c, c, o)
        z_dist_input = self._obs_h_z_out_layers(z_input)
        z_stats = self._suff_stats_layer("obs", latent_type="latent_z", x=z_dist_input)
        if sample:
            z_stoch = self.get_dist(z_stats, latent_type="latent_z").sample()
        else:
            z_stoch = self.get_dist(z_stats, latent_type="latent_z").mode()
        z_post = {"z_stoch": z_stoch, "z_deter": prior["z_deter"], **z_stats}

        z_infer_input = c_input.detach()       # (h_z, h_c, o)
        z_infer_dist_input = self.z_infer_out_layers(z_infer_input)
        z_infer_dist_input = self._z_infer_stat_layer(z_infer_dist_input)
        z_infer_logit = z_infer_dist_input.reshape(list(z_infer_dist_input.shape[:-1]) + [self._stoch, self._discrete])
        post = {
            "z_logit": z_post["z_logit"],
            "z_stoch": z_post["z_stoch"],
            "z_deter": z_post["z_deter"],
            "c_logit": c_post["c_logit"],
            "c_stoch": c_post["c_stoch"],
            "c_deter": c_post["c_deter"],
            "z_infer_logit": z_infer_logit
        }
        return post, prior

    # this is used for making future image
    def z_img_step(self, prev_state, prev_action, embed=None, sample=True):
        # (batch, stoch, discrete_num)
        prev_action *= (1.0 / torch.clip(torch.abs(prev_action), min=1.0)).detach()
        prev_stoch = prev_state["z_stoch"]
        if self._discrete:
            shape = list(prev_stoch.shape[:-2]) + [self._stoch * self._discrete]
            # (batch, stoch, discrete_num) -> (batch, stoch * discrete_num)
            prev_stoch = prev_stoch.reshape(shape)
        if self._shared:
            raise NotImplementedError(f"self._shared={self._shared}")
            # if embed is None:
            #     shape = list(prev_action.shape[:-1]) + [self._embed]
            #     embed = torch.zeros(shape)
            # # (batch, stoch * discrete_num) -> (batch, stoch * discrete_num + action, embed)
            # x = torch.cat([prev_stoch, prev_action, embed], -1)
        else:
            x = torch.cat([prev_stoch, prev_action], -1)
        # (batch, stoch * discrete_num + action, embed) -> (batch, hidden)
        x = self._z_h_inp_layers(x)
        for _ in range(self._rec_depth):  # rec depth is not correctly implemented
            deter = prev_state["z_deter"]
            # (batch, hidden), (batch, deter) -> (batch, deter), (batch, deter)
            x, deter = self._z_cell(x, [deter])
            deter = deter[0]  # Keras wraps the state in a list.
        # (batch, deter) -> (batch, hidden)
        x = self._img_h_z_out_layers(x)
        # (batch, hidden) -> (batch_size, stoch, discrete_num)
        stats = self._suff_stats_layer("ims", latent_type="latent_z", x=x)
        if sample:
            stoch = self.get_dist(stats, latent_type="latent_z").sample()
        else:
            stoch = self.get_dist(stats, latent_type="latent_z").mode()
        prior = {"z_stoch": stoch, "z_deter": deter, **stats}
        return prior


    def c_img_step(self, prev_state, embed=None, sample=True):
        # (batch, stoch, discrete_num)
        # prev_action *= (1.0 / torch.clip(torch.abs(prev_action), min=1.0)).detach()
        prev_stoch = prev_state["c_stoch"]
        if self._c_discrete:
            shape = list(prev_stoch.shape[:-2]) + [self._c_stoch * self._c_discrete]
            # (batch, stoch, discrete_num) -> (batch, stoch * discrete_num)
            prev_stoch = prev_stoch.reshape(shape)
        else:
            raise NotImplementedError
        if self._shared:
            raise NotImplementedError(f"self._shared={self._shared}")
        else:
            x = torch.cat([prev_stoch], -1)
        # (batch, stoch * discrete_num, embed) -> (batch, hidden)
        x = self._c_h_inp_layers(x)
        for _ in range(self._rec_depth):  # rec depth is not correctly implemented
            deter = prev_state["c_deter"]
            # (batch, hidden), (batch, deter) -> (batch, deter), (batch, deter)
            x, deter = self._c_cell(x, [deter])
            deter = deter[0]  # Keras wraps the state in a list.
        # (batch, deter) -> (batch, hidden)
        x = self._img_h_c_out_layers(x)
        # (batch, hidden) -> (batch_size, stoch, discrete_num)
        stats = self._suff_stats_layer("ims", latent_type="latent_c", x=x)
        if sample:
            stoch = self.get_dist(stats, latent_type="latent_c").sample()
        else:
            stoch = self.get_dist(stats,  latent_type="latent_c").mode()
        prior = {"c_stoch": stoch, "c_deter": deter, **stats}
        return prior

    def get_z_stoch(self, deter):
        x = self._img_h_z_out_layers(deter)
        stats = self._suff_stats_layer("ims", latent_type="latent_z", x=x)
        dist = self.get_dist(stats, latent_type="latent_z")
        return dist.mode()

    def _suff_stats_layer(self, name, latent_type, x):
        if self._discrete and self._c_discrete:
            if name == "ims":
                if latent_type == "latent_z":
                    x = self._ims_h_z_stat_layer(x)
                elif latent_type == "latent_c":
                    x = self._ims_h_c_stat_layer(x)
                else:
                    raise NotImplementedError(f"latent_type={latent_type}")
            elif name == "obs":
                if latent_type == "latent_z":
                    x = self._obs_h_z_stat_layer(x)
                elif latent_type == "latent_c":
                    x = self._obs_h_c_stat_layer(x)
                else:
                    raise NotImplementedError(f"latent_type={latent_type}")
            else:
                raise NotImplementedError
            if latent_type == "latent_z":
                logit = x.reshape(list(x.shape[:-1]) + [self._stoch, self._discrete])
                return {"z_logit": logit}
            elif latent_type == "latent_c":
                logit = x.reshape(list(x.shape[:-1]) + [self._c_stoch, self._c_discrete])
                return {"c_logit": logit}
            else:
                raise NotImplementedError
        else:
            raise NotImplementedError(f"self._discrete={self._discrete}, self._c_discrete={self._c_discrete}")


    def kl_loss(self, post, prior, free, dyn_scale, rep_scale, latent_type):
        kld = torchd.kl.kl_divergence
        dist = lambda x: self.get_dist(x, latent_type)
        sg = lambda x: {k: v.detach() for k, v in x.items()}

        rep_loss = value = kld(
            dist(post) if self._discrete else dist(post)._dist,
            dist(sg(prior)) if self._discrete else dist(sg(prior))._dist,
        )
        dyn_loss = kld(
            dist(sg(post)) if self._discrete else dist(sg(post))._dist,
            dist(prior) if self._discrete else dist(prior)._dist,
        )
        rep_loss = torch.mean(torch.clip(rep_loss, min=free))
        dyn_loss = torch.mean(torch.clip(dyn_loss, min=free))
        loss = dyn_scale * dyn_loss + rep_scale * rep_loss

        return loss, value, dyn_loss, rep_loss


class MultiEncoder(nn.Module):
    def __init__(
        self,
        shapes,
        mlp_keys,
        cnn_keys,
        act,
        norm,
        cnn_depth,
        kernel_size,
        minres,
        mlp_layers,
        mlp_units,
        symlog_inputs,
    ):
        super(MultiEncoder, self).__init__()
        excluded = ("is_first", "is_last", "is_terminal", "reward")
        shapes = {
            k: v
            for k, v in shapes.items()
            if k not in excluded and not k.startswith("log_")
        }
        self.cnn_shapes = {
            k: v for k, v in shapes.items() if len(v) == 3 and re.match(cnn_keys, k)
        }
        self.mlp_shapes = {
            k: v
            for k, v in shapes.items()
            if len(v) in (1, 2) and re.match(mlp_keys, k)
        }
        print("Encoder CNN shapes:", self.cnn_shapes)
        print("Encoder MLP shapes:", self.mlp_shapes)

        self.outdim = 0
        if self.cnn_shapes:
            input_ch = sum([v[-1] for v in self.cnn_shapes.values()])
            input_shape = tuple(self.cnn_shapes.values())[0][:2] + (input_ch,)
            self._cnn = ConvEncoder(
                input_shape, cnn_depth, act, norm, kernel_size, minres
            )
            self.outdim += self._cnn.outdim
        if self.mlp_shapes:
            input_size = sum([sum(v) for v in self.mlp_shapes.values()])
            self._mlp = MLP(
                input_size,
                None,
                mlp_layers,
                mlp_units,
                act,
                norm,
                symlog_inputs=symlog_inputs,
            )
            self.outdim += mlp_units

    def forward(self, obs):
        outputs = []
        if self.cnn_shapes:
            inputs = torch.cat([obs[k] for k in self.cnn_shapes], -1)
            outputs.append(self._cnn(inputs))
        if self.mlp_shapes:
            inputs = torch.cat([obs[k] for k in self.mlp_shapes], -1)
            outputs.append(self._mlp(inputs))
        outputs = torch.cat(outputs, -1)
        return outputs


class MultiDecoder(nn.Module):
    def __init__(
        self,
        feat_size,
        shapes,
        mlp_keys,
        cnn_keys,
        act,
        norm,
        cnn_depth,
        kernel_size,
        minres,
        mlp_layers,
        mlp_units,
        cnn_sigmoid,
        image_dist,
        vector_dist,
    ):
        super(MultiDecoder, self).__init__()
        excluded = ("is_first", "is_last", "is_terminal")
        shapes = {k: v for k, v in shapes.items() if k not in excluded}
        self.cnn_shapes = {
            k: v for k, v in shapes.items() if len(v) == 3 and re.match(cnn_keys, k)
        }
        self.mlp_shapes = {
            k: v
            for k, v in shapes.items()
            if len(v) in (1, 2) and re.match(mlp_keys, k)
        }
        print("Decoder CNN shapes:", self.cnn_shapes)
        print("Decoder MLP shapes:", self.mlp_shapes)

        if self.cnn_shapes:
            some_shape = list(self.cnn_shapes.values())[0]
            shape = (sum(x[-1] for x in self.cnn_shapes.values()),) + some_shape[:-1]
            self._cnn = ConvDecoder(
                feat_size,
                shape,
                cnn_depth,
                act,
                norm,
                kernel_size,
                minres,
                cnn_sigmoid=cnn_sigmoid,
            )
        if self.mlp_shapes:
            self._mlp = MLP(
                feat_size,
                self.mlp_shapes,
                mlp_layers,
                mlp_units,
                act,
                norm,
                vector_dist,
            )
        self._image_dist = image_dist

    def forward(self, features):
        dists = {}
        if self.cnn_shapes:
            feat = features
            outputs = self._cnn(feat)
            split_sizes = [v[-1] for v in self.cnn_shapes.values()]
            outputs = torch.split(outputs, split_sizes, -1)
            dists.update(
                {
                    key: self._make_image_dist(output)
                    for key, output in zip(self.cnn_shapes.keys(), outputs)
                }
            )
        if self.mlp_shapes:
            dists.update(self._mlp(features))
        return dists

    def _make_image_dist(self, mean):
        if self._image_dist == "normal":
            return tools.ContDist(
                torchd.independent.Independent(torchd.normal.Normal(mean, 1), 3)
            )
        if self._image_dist == "mse":
            return tools.MSEDist(mean)
        raise NotImplementedError(self._image_dist)


class ConvEncoder(nn.Module):
    def __init__(
        self,
        input_shape,
        depth=32,
        act="SiLU",
        norm="LayerNorm",
        kernel_size=4,
        minres=4,
    ):
        super(ConvEncoder, self).__init__()
        act = getattr(torch.nn, act)
        norm = getattr(torch.nn, norm)
        h, w, input_ch = input_shape
        layers = []
        for i in range(int(np.log2(h) - np.log2(minres))):
            if i == 0:
                in_dim = input_ch
            else:
                in_dim = 2 ** (i - 1) * depth
            out_dim = 2**i * depth
            layers.append(
                Conv2dSame(
                    in_channels=in_dim,
                    out_channels=out_dim,
                    kernel_size=kernel_size,
                    stride=2,
                    bias=False,
                )
            )
            layers.append(ChLayerNorm(out_dim))
            layers.append(act())
            h, w = h // 2, w // 2

        self.outdim = out_dim * h * w
        self.layers = nn.Sequential(*layers)
        self.layers.apply(tools.weight_init)

    def forward(self, obs):
        # (batch, time, h, w, ch) -> (batch * time, h, w, ch)
        x = obs.reshape((-1,) + tuple(obs.shape[-3:]))
        # (batch * time, h, w, ch) -> (batch * time, ch, h, w)
        x = x.permute(0, 3, 1, 2)
        x = self.layers(x)
        # (batch * time, ...) -> (batch * time, -1)
        x = x.reshape([x.shape[0], np.prod(x.shape[1:])])
        # (batch * time, -1) -> (batch, time, -1)
        return x.reshape(list(obs.shape[:-3]) + [x.shape[-1]])


class ConvDecoder(nn.Module):
    def __init__(
        self,
        feat_size,
        shape=(3, 64, 64),
        depth=32,
        act=nn.ELU,
        norm=nn.LayerNorm,
        kernel_size=4,
        minres=4,
        outscale=1.0,
        cnn_sigmoid=False,
    ):
        super(ConvDecoder, self).__init__()
        act = getattr(torch.nn, act)
        norm = getattr(torch.nn, norm)
        self._shape = shape
        self._cnn_sigmoid = cnn_sigmoid
        layer_num = int(np.log2(shape[1]) - np.log2(minres))
        self._minres = minres
        self._embed_size = minres**2 * depth * 2 ** (layer_num - 1)

        self._linear_layer = nn.Linear(feat_size, self._embed_size)
        self._linear_layer.apply(tools.weight_init)
        in_dim = self._embed_size // (minres**2)

        layers = []
        h, w = minres, minres
        for i in range(layer_num):
            out_dim = self._embed_size // (minres**2) // (2 ** (i + 1))
            bias = False
            initializer = tools.weight_init
            if i == layer_num - 1:
                out_dim = self._shape[0]
                act = False
                bias = True
                norm = False
                initializer = tools.uniform_weight_init(outscale)

            if i != 0:
                in_dim = 2 ** (layer_num - (i - 1) - 2) * depth
            pad_h, outpad_h = self.calc_same_pad(k=kernel_size, s=2, d=1)
            pad_w, outpad_w = self.calc_same_pad(k=kernel_size, s=2, d=1)
            layers.append(
                nn.ConvTranspose2d(
                    in_dim,
                    out_dim,
                    kernel_size,
                    2,
                    padding=(pad_h, pad_w),
                    output_padding=(outpad_h, outpad_w),
                    bias=bias,
                )
            )
            if norm:
                layers.append(ChLayerNorm(out_dim))
            if act:
                layers.append(act())
            [m.apply(initializer) for m in layers[-3:]]
            h, w = h * 2, w * 2

        self.layers = nn.Sequential(*layers)

    def calc_same_pad(self, k, s, d):
        val = d * (k - 1) - s + 1
        pad = math.ceil(val / 2)
        outpad = pad * 2 - val
        return pad, outpad

    def forward(self, features, dtype=None):
        x = self._linear_layer(features)
        # (batch, time, -1) -> (batch * time, h, w, ch)
        x = x.reshape(
            [-1, self._minres, self._minres, self._embed_size // self._minres**2]
        )
        # (batch, time, -1) -> (batch * time, ch, h, w)
        x = x.permute(0, 3, 1, 2)
        x = self.layers(x)
        # (batch, time, -1) -> (batch * time, ch, h, w) necessary???
        mean = x.reshape(features.shape[:-1] + self._shape)
        # (batch * time, ch, h, w) -> (batch * time, h, w, ch)
        mean = mean.permute(0, 1, 3, 4, 2)
        if self._cnn_sigmoid:
            mean = F.sigmoid(mean) - 0.5
        return mean


class MLP(nn.Module):
    def __init__(
        self,
        inp_dim,
        shape,
        layers,
        units,
        act="SiLU",
        norm="LayerNorm",
        dist="normal",
        std=1.0,
        outscale=1.0,
        symlog_inputs=False,
        device="cuda",
    ):
        super(MLP, self).__init__()
        self._shape = (shape,) if isinstance(shape, int) else shape
        if self._shape is not None and len(self._shape) == 0:
            self._shape = (1,)
        self._layers = layers
        act = getattr(torch.nn, act)
        norm = getattr(torch.nn, norm)
        self._dist = dist
        self._std = std
        self._symlog_inputs = symlog_inputs
        self._device = device

        layers = []
        for index in range(self._layers):
            layers.append(nn.Linear(inp_dim, units, bias=False))
            layers.append(norm(units, eps=1e-03))
            layers.append(act())
            if index == 0:
                inp_dim = units
        self.layers = nn.Sequential(*layers)
        self.layers.apply(tools.weight_init)

        if isinstance(self._shape, dict):
            self.mean_layer = nn.ModuleDict()
            for name, shape in self._shape.items():
                self.mean_layer[name] = nn.Linear(inp_dim, np.prod(shape))
            self.mean_layer.apply(tools.uniform_weight_init(outscale))
            if self._std == "learned":
                self.std_layer = nn.ModuleDict()
                for name, shape in self._shape.items():
                    self.std_layer[name] = nn.Linear(inp_dim, np.prod(shape))
                self.std_layer.apply(tools.uniform_weight_init(outscale))
        elif self._shape is not None:
            self.mean_layer = nn.Linear(inp_dim, np.prod(self._shape))
            self.mean_layer.apply(tools.uniform_weight_init(outscale))
            if self._std == "learned":
                self.std_layer = nn.Linear(units, np.prod(self._shape))
                self.std_layer.apply(tools.uniform_weight_init(outscale))

    def forward(self, features, dtype=None):
        x = features
        if self._symlog_inputs:
            x = tools.symlog(x)
        out = self.layers(x)
        if self._shape is None:
            return out
        if isinstance(self._shape, dict):
            dists = {}
            for name, shape in self._shape.items():
                mean = self.mean_layer[name](out)
                if self._std == "learned":
                    std = self.std_layer[name](out)
                else:
                    std = self._std
                dists.update({name: self.dist(self._dist, mean, std, shape)})
            return dists
        else:
            mean = self.mean_layer(out)
            if self._std == "learned":
                std = self.std_layer(out)
            else:
                std = self._std
            return self.dist(self._dist, mean, std, self._shape)

    def dist(self, dist, mean, std, shape):
        if dist == "normal":
            return tools.ContDist(
                torchd.independent.Independent(
                    torchd.normal.Normal(mean, std), len(shape)
                )
            )
        if dist == "huber":
            return tools.ContDist(
                torchd.independent.Independent(
                    tools.UnnormalizedHuber(mean, std, 1.0), len(shape)
                )
            )
        if dist == "binary":
            return tools.Bernoulli(
                torchd.independent.Independent(
                    torchd.bernoulli.Bernoulli(logits=mean), len(shape)
                )
            )
        if dist == "symlog_disc":
            return tools.DiscDist(logits=mean, device=self._device)
        if dist == "symlog_mse":
            return tools.SymlogDist(mean)
        raise NotImplementedError(dist)


class ActionHead(nn.Module):
    def __init__(
        self,
        inp_dim,
        size,
        layers,
        units,
        act=nn.ELU,
        norm=nn.LayerNorm,
        dist="trunc_normal",
        init_std=0.0,
        min_std=0.1,
        max_std=1.0,
        temp=0.1,
        outscale=1.0,
        unimix_ratio=0.01,
    ):
        super(ActionHead, self).__init__()
        self._size = size
        self._layers = layers
        self._units = units
        self._dist = dist
        act = getattr(torch.nn, act)
        norm = getattr(torch.nn, norm)
        self._min_std = min_std
        self._max_std = max_std
        self._init_std = init_std
        self._unimix_ratio = unimix_ratio
        self._temp = temp() if callable(temp) else temp

        pre_layers = []
        for index in range(self._layers):
            pre_layers.append(nn.Linear(inp_dim, self._units, bias=False))
            pre_layers.append(norm(self._units, eps=1e-03))
            pre_layers.append(act())
            if index == 0:
                inp_dim = self._units
        self._pre_layers = nn.Sequential(*pre_layers)
        self._pre_layers.apply(tools.weight_init)

        if self._dist in ["tanh_normal", "tanh_normal_5", "normal", "trunc_normal"]:
            self._dist_layer = nn.Linear(self._units, 2 * self._size)
            self._dist_layer.apply(tools.uniform_weight_init(outscale))

        elif self._dist in ["normal_1", "onehot", "onehot_gumbel"]:
            self._dist_layer = nn.Linear(self._units, self._size)
            self._dist_layer.apply(tools.uniform_weight_init(outscale))

    def forward(self, features, dtype=None):
        x = features
        x = self._pre_layers(x)
        if self._dist == "tanh_normal":
            x = self._dist_layer(x)
            mean, std = torch.split(x, 2, -1)
            mean = torch.tanh(mean)
            std = F.softplus(std + self._init_std) + self._min_std
            dist = torchd.normal.Normal(mean, std)
            dist = torchd.transformed_distribution.TransformedDistribution(
                dist, tools.TanhBijector()
            )
            dist = torchd.independent.Independent(dist, 1)
            dist = tools.SampleDist(dist)
        elif self._dist == "tanh_normal_5":
            x = self._dist_layer(x)
            mean, std = torch.split(x, 2, -1)
            mean = 5 * torch.tanh(mean / 5)
            std = F.softplus(std + 5) + 5
            dist = torchd.normal.Normal(mean, std)
            dist = torchd.transformed_distribution.TransformedDistribution(
                dist, tools.TanhBijector()
            )
            dist = torchd.independent.Independent(dist, 1)
            dist = tools.SampleDist(dist)
        elif self._dist == "normal":
            x = self._dist_layer(x)
            mean, std = torch.split(x, [self._size] * 2, -1)
            std = (self._max_std - self._min_std) * torch.sigmoid(
                std + 2.0
            ) + self._min_std
            dist = torchd.normal.Normal(torch.tanh(mean), std)
            dist = tools.ContDist(torchd.independent.Independent(dist, 1))
        elif self._dist == "normal_1":
            mean = self._dist_layer(x)
            dist = torchd.normal.Normal(mean, 1)
            dist = tools.ContDist(torchd.independent.Independent(dist, 1))
        elif self._dist == "trunc_normal":
            x = self._dist_layer(x)
            mean, std = torch.split(x, [self._size] * 2, -1)
            mean = torch.tanh(mean)
            std = 2 * torch.sigmoid(std / 2) + self._min_std
            dist = tools.SafeTruncatedNormal(mean, std, -1, 1)
            dist = tools.ContDist(torchd.independent.Independent(dist, 1))
        elif self._dist == "onehot":
            x = self._dist_layer(x)
            dist = tools.OneHotDist(x, unimix_ratio=self._unimix_ratio)
        elif self._dist == "onehot_gumble":
            x = self._dist_layer(x)
            temp = self._temp
            dist = tools.ContDist(torchd.gumbel.Gumbel(x, 1 / temp))
        else:
            raise NotImplementedError(self._dist)
        return dist


class GRUCell(nn.Module):
    def __init__(self, inp_size, size, norm=False, act=torch.tanh, update_bias=-1):
        super(GRUCell, self).__init__()
        self._inp_size = inp_size
        self._size = size
        self._act = act
        self._norm = norm
        self._update_bias = update_bias
        self._layer = nn.Linear(inp_size + size, 3 * size, bias=False)
        if norm:
            self._norm = nn.LayerNorm(3 * size, eps=1e-03)

    @property
    def state_size(self):
        return self._size

    def forward(self, inputs, state):
        state = state[0]  # Keras wraps the state in a list.
        parts = self._layer(torch.cat([inputs, state], -1))
        if self._norm:
            parts = self._norm(parts)
        reset, cand, update = torch.split(parts, [self._size] * 3, -1)
        reset = torch.sigmoid(reset)
        cand = self._act(reset * cand)
        update = torch.sigmoid(update + self._update_bias)
        output = update * cand + (1 - update) * state
        return output, [output]


class Conv2dSame(torch.nn.Conv2d):
    def calc_same_pad(self, i, k, s, d):
        return max((math.ceil(i / s) - 1) * s + (k - 1) * d + 1 - i, 0)

    def forward(self, x):
        ih, iw = x.size()[-2:]
        pad_h = self.calc_same_pad(
            i=ih, k=self.kernel_size[0], s=self.stride[0], d=self.dilation[0]
        )
        pad_w = self.calc_same_pad(
            i=iw, k=self.kernel_size[1], s=self.stride[1], d=self.dilation[1]
        )

        if pad_h > 0 or pad_w > 0:
            x = F.pad(
                x, [pad_w // 2, pad_w - pad_w // 2, pad_h // 2, pad_h - pad_h // 2]
            )

        ret = F.conv2d(
            x,
            self.weight,
            self.bias,
            self.stride,
            self.padding,
            self.dilation,
            self.groups,
        )
        return ret


class ChLayerNorm(nn.Module):
    def __init__(self, ch, eps=1e-03):
        super(ChLayerNorm, self).__init__()
        self.norm = torch.nn.LayerNorm(ch, eps=eps)

    def forward(self, x):
        x = x.permute(0, 2, 3, 1)
        x = self.norm(x)
        x = x.permute(0, 3, 1, 2)
        return x
