import torch
import copy
import pickle
import dgl

class NET(torch.nn.Module):
    """
        ER baseline for NCGL tasks

        :param model: The backbone GNNs, e.g. GCN, GAT, GIN, etc.
        :param args: The arguments containing the configurations of the experiments including the training parameters like the learning rate, the setting confugurations like class-IL and task-IL, etc. These arguments are initialized in the train.py file and can be specified by the users upon running the code.

        """

    def __init__(self,
                 model,
                 args):
        super(NET, self).__init__()
        
        # setup network
        self.net = model

        # setup optimizer
        self.opt = torch.optim.Adam(self.net.parameters(), lr=args.lr, weight_decay=args.weight_decay)

        # setup losses
        self.ce = torch.nn.functional.cross_entropy

        # setup memories
        self.buffer_node_ids = []
        self.memory_proportion = int(args.er_args['memory_proportion'])
        self.budget = int(args.er_args['budget'])
        self.aux_g = None
        self.n_seen_examples = 0
        self.epochs = 0
        self.seen_classes = []

    def forward(self, features):
        output = self.net(features)
        return output   

    def observe_minibatch(self, args, g, features, labels, train_ids, ids_per_cls):
        """
                The method for learning the given tasks under the class-IL setting.

                :param args: Same as the args in __init__().
                :param g: The graph of the current task.
                :param features: Node features of the current task.
                :param labels: Labels of the nodes in the current task.
                :param train_ids: The indices of the nodes participating in the training.
                :param ids_per_cls: Indices of the nodes in each class.

                """
        for label in labels.unique():
            if label not in self.seen_classes:
                self.seen_classes.append(label)

        self.epochs += 1
        last_epoch = self.epochs % args.epochs

        ids_per_cls_train = [list(set(ids).intersection(set(train_ids))) for ids in ids_per_cls]
        self.net.train()
        n_nodes = len(train_ids)
        buffer_size = len(self.buffer_node_ids)

        self.net.zero_grad()
        offset1, offset2 = 0, max(self.seen_classes)+1
        nb_sampler = dgl.dataloading.NeighborSampler(args.n_nbs_sample) if args.sample_nbs else dgl.dataloading.MultiLayerFullNeighborSampler(len(self.net.gat_layers))
        _, _, blocks = nb_sampler.sample_blocks(g, torch.tensor(train_ids).to(device='cuda:{}'.format(args.gpu)))
        input_features = blocks[0].srcdata['feat']
        output, _ = self.net.forward_batch(blocks, input_features)
        output_labels = labels[train_ids]

        if args.cls_balance:
            n_per_cls = [(output_labels == j).sum() for j in range(args.n_cls)]
            loss_w_ = [1. / max(i, 1) for i in n_per_cls]  # weight to balance the loss of different class
        else:
            loss_w_ = [1. for i in range(args.n_cls)]
        loss_w_ = torch.tensor(loss_w_).to(device='cuda:{}'.format(args.gpu))
        loss = self.ce(output[:,offset1:offset2], labels[train_ids], weight=loss_w_[offset1: offset2])

        if buffer_size > 0:
            # sample batch_size nodes uniformly at random from the buffer
            sampled_mask = torch.zeros(buffer_size, dtype=torch.bool)
            n_samples = min(args.batch_size*self.memory_proportion, buffer_size)
            sampled_mask[torch.randperm(buffer_size)[:n_samples]] = True
            # calculate auxiliary loss based on replay
            output, _ = self.net(self.aux_g, self.aux_features)
            loss_aux = self.ce(output[sampled_mask, offset1:offset2], self.aux_labels[sampled_mask])
            loss = loss + loss_aux

        loss.backward()
        self.opt.step()

        if last_epoch == 0:
            # perform reservoir sampling
            place_left = max(0, self.budget - len(self.buffer_node_ids))
            if place_left:
                offset = min(place_left, n_nodes)
                self.buffer_node_ids.extend(train_ids[:offset])
                if offset < n_nodes:
                    for i in range(offset, n_nodes):
                        j = torch.randint(0, self.n_seen_examples + i, (1,))
                        if j < self.budget:
                            self.buffer_node_ids[j] = train_ids[i]
            else:
                for i in range(n_nodes):
                    j = torch.randint(0, self.n_seen_examples + i, (1,))
                    if j < self.budget:
                        self.buffer_node_ids[j] = train_ids[i]
            self.n_seen_examples += n_nodes
            subg = dgl.node_subgraph(g, self.buffer_node_ids, store_ids=True)
            n_edges = subg.edges()[0].shape[0]
            subg.remove_edges(list(range(n_edges)))
            subg = dgl.add_self_loop(subg)
            self.aux_g = subg.to(device='cuda:{}'.format(features.get_device()))
            self.aux_features, self.aux_labels = self.aux_g.srcdata['feat'], self.aux_g.dstdata['label'].squeeze()
    
