import torch
import copy

class NET(torch.nn.Module):
    """
        Jointly trained model baseline for NCGL tasks

        In this baseline, when ever a new task comes, the model will be jointly trained on all existing tasks.

        :param model: The backbone GNNs, e.g. GCN, GAT, GIN, etc.
        :param task_manager: Mainly serves to store the indices of the output dimensions corresponding to each task
        :param args: The arguments containing the configurations of the experiments including the training parameters like the learning rate, the setting confugurations like class-IL and task-IL, etc. These arguments are initialized in the train.py file and can be specified by the users upon running the code.

        """
    def __init__(self,
                 model,
                 task_manager,
                 args):
        super(NET, self).__init__()

        self.task_manager = task_manager

        # setup network
        self.net = model

        # setup optimizer
        self.opt = torch.optim.Adam(self.net.parameters(), lr=args.lr, weight_decay=args.weight_decay)

        # setup losses
        self.ce = torch.nn.functional.cross_entropy

        # setup memories
        self.current_task = -1

    def forward(self, features):
        output = self.net(features)
        return output

    def observe(self, args, g, features, labels_all, t, train_ids, ids_per_cls_all, dataset):
        """
                                The method for learning the given tasks under the class-IL setting with inter-task edges.

                                :param args: Same as the args in __init__().
                                :param g: The graphs of all the existing tasks to be jointly trained on.
                                :param features: Node features of the nodes in all the existing tasks.
                                :param labels_all: Labels of the nodes in all the existing tasks.
                                :param t: Index of the newest task.
                                :param train_ids: The indices of the nodes participating in the training in all the existing tasks.
                                :param ids_per_cls_all: Indices of the nodes in each class of all the existing tasks.
                                :param dataset: The entire dataset (not in use in the current baseline).

                                """
        self.net.train()
        if t!=self.current_task:
            self.current_task = t
            self.net.reset_params()

        self.net.zero_grad()
        loss = 0
        output, _ = self.net(g, features)
        offset1, offset2 = self.task_manager.get_label_offset(t)
        for tt,task in enumerate(args.task_seq[0:t+1]):
            ids_per_cls_current_task = [ids_per_cls_all[i] for i in task]
            ids_per_cls_train = [list(set(ids).intersection(set(train_ids))) for ids in ids_per_cls_current_task]
            train_ids_current_task = []
            for ids in ids_per_cls_train:
                train_ids_current_task.extend(ids)
            labels = labels_all[train_ids_current_task]

            if args.cls_balance:
                n_per_cls = [(labels == j).sum() for j in range(args.n_cls)]
                loss_w_ = [1. / max(i, 1) for i in n_per_cls]  # weight to balance the loss of different class
            else:
                loss_w_ = [1. for i in range(args.n_cls)]
            loss_w_ = torch.tensor(loss_w_).to(device='cuda:{}'.format(args.gpu))
            loss += self.ce(output[train_ids_current_task, offset1:offset2], labels, weight=loss_w_[offset1: offset2])
        loss.backward()
        self.opt.step()