"""
Defines the abstract base class for all quantizers.
"""
from abc import ABC, abstractmethod
from typing import Any, Tuple
import torch

class BaseQuantizer(ABC):
    """
    Abstract Base Class for quantizers.

    All custom quantizers should inherit from this class and implement the
    `quantize` and `dequantize` methods.
    """

    def __init__(self, n_bits: int, *args, **kwargs):
        """
        Initializes the base quantizer.

        Args:
            n_bits (int): The number of bits for quantization.
        """
        if not (isinstance(n_bits, int) and n_bits > 0):
            raise ValueError("Number of bits must be a positive integer.")
        self.n_bits = n_bits

    @abstractmethod
    def quantize(self, fp32_tensor: torch.Tensor) -> Tuple[torch.Tensor, Any]:
        """
        Quantizes a full-precision tensor.

        Args:
            fp32_tensor (torch.Tensor): The input tensor in FP32 or FP16.

        Returns:
            Tuple[torch.Tensor, Any]: A tuple containing the quantized tensor
            (as integers) and a state object. The state object can contain
            any information needed for dequantization, such as scaling factors,
            zero-points, or quantization maps.
        """
        pass

    @abstractmethod
    def dequantize(self, quantized_tensor: torch.Tensor, state: Any) -> torch.Tensor:
        """
        Dequantizes a tensor back to full-precision.

        Args:
            quantized_tensor (torch.Tensor): The quantized integer tensor.
            state (Any): The state object returned by the `quantize` method.

        Returns:
            torch.Tensor: The dequantized tensor in floating-point format.
        """
        pass
