"""
Entry point for RQ1: Run Benford's Law compliance analysis on a model and
generate plots comparing observed distributions with Benford's Law.
"""
import argparse
import logging
import json
import os
import torch

# script is run from the root directory.
import sys
sys.path.insert(0, os.path.abspath(os.path.join(os.path.dirname(__file__), '..')))

from benford_quant.utils import load_config, setup_logging
from benford_quant.analysis.benford_analysis import BenfordAnalyzer

try:
    from transformers import AutoModelForCausalLM
except ImportError:
    print("Please install the required libraries: pip install -r requirements.txt")
    sys.exit(1)

def main():
    """
    Run the analysis and produce plots summarizing the results.
    """
    parser = argparse.ArgumentParser(description="Run Benford's Law analysis on a Hugging Face model.")
    parser.add_argument('--config_path', type=str, required=True, help='Path to the YAML configuration file.')
    args = parser.parse_args()

    # 1. Setup logging
    setup_logging()
    logger = logging.getLogger(__name__)
    logger.info("Starting Benford's Law analysis script...")

    # 2. Load configuration
    logger.info(f"Loading configuration from: {args.config_path}")
    config = load_config(args.config_path)

    # 3. Load model
    model_config = config['model']
    logger.info(f"Loading model: {model_config['name']}")

    device = "cuda" if torch.cuda.is_available() else "cpu"
    logger.info(f"Using device: {device}")

    model = AutoModelForCausalLM.from_pretrained(
        model_config['name'],
        trust_remote_code=model_config.get('trust_remote_code', False),
        device_map='auto'
    ).to(device)

    # 4. Initialize and run the analyzer
    logger.info("Initializing BenfordAnalyzer...")
    analyzer = BenfordAnalyzer(model=model, config=config)

    logger.info("Running analysis...")
    stats = analyzer.analyze()

    # 5. Save results
    stats_file = config['output']['stats_file']
    plot_dir = config['output']['plot_dir']

    # Ensure output directories exist
    os.makedirs(os.path.dirname(stats_file), exist_ok=True)
    os.makedirs(plot_dir, exist_ok=True)

    logger.info(f"Saving statistics to: {stats_file}")
    with open(stats_file, 'w', encoding='utf-8') as f:
        json.dump(stats, f, indent=4)

    logger.info(f"Saving plots to: {plot_dir}")
    analyzer.plot_results(stats, plot_dir)

    logger.info("Analysis script finished successfully.")

if __name__ == "__main__":
    main()
