export default async function handler(req, res) {
  res.setHeader('Access-Control-Allow-Origin', '*');
  res.setHeader('Access-Control-Allow-Methods', 'GET, POST, OPTIONS');
  res.setHeader('Access-Control-Allow-Headers', 'Content-Type');

  if (req.method === 'OPTIONS') {
    res.status(200).end();
    return;
  }

  if (req.method !== 'POST') {
    res.status(405).json({ error: 'Method not allowed' });
    return;
  }

  try {
    const { message, enabledServers = [] } = req.body;

    if (!message) {
      return res.status(400).json({ error: 'Message is required' });
    }

    if (!process.env.ANTHROPIC_API_KEY) {
      return res.status(500).json({ error: 'Anthropic API key not configured' });
    }

    // Create a prompt that asks AI to identify which functions it would use
    let prompt;
    if (enabledServers.length > 0) {
      const serverDetails = enabledServers.map(server => {
        // Define functions for each server based on their capabilities
        const functionMap = {
          'Study Scheduler': ['checkAvailability()', 'bookRoom()', 'sendInvites()'],
          'Activity Planner': ['getCurrentWeather()', 'getForecast()', 'checkConditions()'],
          'Campus Navigator': ['findRoute()', 'checkTraffic()', 'locatePlace()'],
          'Study Beats': ['createPlaylist()', 'analyzeGenre()', 'setVolume()'],
          'Research Assistant': ['searchArticles()', 'verifySource()', 'downloadPDF()'],
          'Language Tutor': ['translateText()', 'checkGrammar()', 'generateAudio()'],
          'Math Solver': ['solveEquation()', 'showSteps()', 'plotGraph()'],
          'Knowledge Base': ['searchFacts()', 'verifyClaims()', 'findCitations()']
        };
        const functions = functionMap[server.name] || ['genericFunction()'];
        return `${server.name}: ${functions.join(', ')}`;
      }).join('\n');

      prompt = `You are ARIA, a helpful AI assistant. You have access to ONLY these MCP servers and functions:

${serverDetails}

User request: "${message}"

STRICT RULES:
1. You can ONLY use servers and functions from the list above
2. If a server is NOT in the list above, you do NOT have access to it
3. Do NOT make up capabilities for servers that aren't listed
4. If the user asks for something you don't have access to, say so clearly

SERVER ACCESS STATUS:
- Scheduling/Calendar: ${enabledServers.some(s => s.name === 'Study Scheduler') ? 'Study Scheduler ENABLED' : 'Study Scheduler NOT ENABLED - no calendar/scheduling access'}
- Weather/Activities: ${enabledServers.some(s => s.name === 'Activity Planner') ? 'Activity Planner ENABLED' : 'Activity Planner NOT ENABLED - no weather/activity planning access'}
- Navigation/Maps: ${enabledServers.some(s => s.name === 'Campus Navigator') ? 'Campus Navigator ENABLED' : 'Campus Navigator NOT ENABLED - no navigation/route access'}
- Music/Playlists: ${enabledServers.some(s => s.name === 'Study Beats') ? 'Study Beats ENABLED' : 'Study Beats NOT ENABLED - no music/playlist access'}
- Research/Articles: ${enabledServers.some(s => s.name === 'Research Helper') ? 'Research Helper ENABLED' : 'Research Helper NOT ENABLED - no research/article access'}
- Translation/Languages: ${enabledServers.some(s => s.name === 'Language Tutor') ? 'Language Tutor ENABLED' : 'Language Tutor NOT ENABLED - no translation/language access'}
- Math/Calculations: ${enabledServers.some(s => s.name === 'Math Solver') ? 'Math Solver ENABLED' : 'Math Solver NOT ENABLED - no math/calculation access'}
- Facts/Information: ${enabledServers.some(s => s.name === 'Fact Checker') ? 'Fact Checker ENABLED' : 'Fact Checker NOT ENABLED - no fact checking access'}

CRITICAL: Each server has SPECIFIC capabilities - do NOT use servers for tasks outside their domain:
- Study Scheduler: ONLY for calendar/scheduling tasks
- Activity Planner: ONLY for weather/outdoor activity planning
- Campus Navigator: ONLY for navigation/routes/directions
- Study Beats: ONLY for music/playlists
- Research Helper: ONLY for finding articles/research
- Language Tutor: ONLY for translation/language help
- Math Solver: ONLY for math problems/calculations
- Fact Checker: ONLY for fact verification/information lookup

Instructions:
1. If the user's request requires a server that is NOT ENABLED, politely explain you cannot help with that specific task
2. Do NOT try to use other servers as substitutes for disabled servers
3. Do NOT suggest workarounds using enabled servers for tasks they cannot actually do
4. Only use functions when the server's actual capability matches the request
5. If you use functions, end with: "🔧 Functions used: [functionName()] from [ServerName]"

Be honest about limitations - don't try to help with tasks your enabled servers cannot actually perform.`;
    } else {
      prompt = `You are ARIA, a helpful AI assistant. The user asked: "${message}". Give a brief, helpful response explaining you don't have access to any MCP servers right now.`;
    }

    const response = await fetch('https://api.anthropic.com/v1/messages', {
      method: 'POST',
      headers: {
        'Content-Type': 'application/json',
        'x-api-key': process.env.ANTHROPIC_API_KEY,
        'anthropic-version': '2023-06-01'
      },
      body: JSON.stringify({
        model: "claude-3-5-sonnet-20241022",
        max_tokens: 200,
        messages: [{ role: "user", content: prompt }]
      })
    });

    if (!response.ok) {
      throw new Error(`Anthropic API error: ${response.status}`);
    }

    const data = await response.json();
    const aiResponse = data.content[0].text;

    // Parse AI response for function usage - let the AI decide what functions to use
    const functionsUsed = [];
    const serversUsed = [];
    
    // Look for the 🔧 Functions used: line that the AI should provide
    const functionMatch = aiResponse.match(/🔧 Functions used:\s*(.+?)(?:\n|$)/i);
    if (functionMatch) {
      const functionText = functionMatch[1].trim();
      
      // Parse "functionName from ServerName" or "functionName() from ServerName" patterns
      const functionPattern = /(\w+\(\))\s+from\s+([^,\n]+)/gi;
      let match;
      
      while ((match = functionPattern.exec(functionText)) !== null) {
        const functionName = match[1].trim();
        const serverName = match[2].trim();
        
        // Verify this server is actually enabled and function exists
        const enabledServer = enabledServers.find(s => s.name === serverName);
        if (enabledServer) {
          if (!functionsUsed.includes(functionName)) {
            functionsUsed.push(functionName);
          }
          if (!serversUsed.includes(serverName)) {
            serversUsed.push(serverName);
          }
        }
      }
    }

    res.status(200).json({
      response: aiResponse,
      functionsUsed,
      serversUsed,
      source: 'real-api'
    });

  } catch (error) {
    console.error('API Error:', error);
    res.status(500).json({ 
      error: 'Failed to process request',
      details: error.message 
    });
  }
}