const Anthropic = require('@anthropic-ai/sdk');

// Generate MCP-aware prompt for Anthropic API
const generateMCPPrompt = (message, enabledServers) => {
  if (enabledServers.length === 0) {
    return `You are ARIA, a helpful AI assistant for high school students. Currently, you don't have access to any MCP (Model Context Protocol) servers, so you can only provide general advice based on your training data.

User's question: "${message}"

Since no MCP servers are enabled, respond with: "I don't have access to any MCP servers right now. Please enable some servers to unlock my enhanced capabilities! I can help with scheduling, weather, math problems, translations, research, and more once you enable the relevant servers."`;
  }

  const serverDescriptions = enabledServers.map(server => 
    `• ${server.name}: ${server.description}
   Available functions: ${server.functions.join(', ')}`
  ).join('\n');

  // Create explicit constraint checking
  const serverNames = enabledServers.map(s => s.name);
  const hasScheduler = serverNames.includes('Study Scheduler');
  const hasMath = serverNames.includes('Math Solver');
  const hasWeather = serverNames.includes('Activity Planner');
  const hasTranslation = serverNames.includes('Language Tutor');
  const hasMusic = serverNames.includes('Study Beats');

  // Determine request type and required server
  const getRequestAnalysis = (message) => {
    const msg = message.toLowerCase();
    if (msg.includes('schedul') || msg.includes('calendar') || msg.includes('availab') || msg.includes('book') || msg.includes('meet')) {
      return { type: 'scheduling', server: 'Study Scheduler', hasServer: hasScheduler };
    }
    if (msg.includes('math') || msg.includes('calculat') || msg.includes('equation') || msg.includes('solve') || msg.includes('algebra') || msg.includes('geometry')) {
      return { type: 'math', server: 'Math Solver', hasServer: hasMath };
    }
    if (msg.includes('weather') || msg.includes('rain') || msg.includes('temperature') || msg.includes('outdoor') || msg.includes('forecast')) {
      return { type: 'weather', server: 'Activity Planner', hasServer: hasWeather };
    }
    if (msg.includes('translat') || msg.includes('spanish') || msg.includes('french') || msg.includes('german') || msg.includes('italian') || msg.includes('chinese') || msg.includes('japanese')) {
      return { type: 'translation', server: 'Language Tutor', hasServer: hasTranslation };
    }
    if (msg.includes('music') || msg.includes('playlist') || msg.includes('song') || msg.includes('beats') || msg.includes('focus') || msg.includes('study session')) {
      return { type: 'music', server: 'Study Beats', hasServer: hasMusic };
    }
    return { type: 'general', server: null, hasServer: true };
  };

  const analysis = getRequestAnalysis(message);
  
  if (analysis.server && !analysis.hasServer) {
    return `You are ARIA, a helpful AI assistant. The user is asking about ${analysis.type}, but you don't have access to the ${analysis.server} MCP server which is required for this request.

User's question: "${message}"

Respond politely that you don't have access to the necessary MCP server (${analysis.server}) to help with this ${analysis.type} request. Suggest they enable the ${analysis.server} server to get enhanced capabilities for this type of question.`;
  }

  return `You are ARIA, a helpful AI assistant for high school students with access to these MCP (Model Context Protocol) servers:

${serverDescriptions}

CRITICAL CONSTRAINTS:
- Only use functions from the enabled servers listed above
- If asked about something requiring a server you don't have, politely explain you need that specific server enabled
- Be concise and helpful
- Use a friendly, encouraging tone appropriate for high school students

User's question: "${message}"

Analyze the request and respond appropriately. If you use any server functions, mention which server and functions you're using in a natural way.`;
};

// Extract function calls from AI response
const extractFunctionCalls = (text, enabledServers) => {
  const functionsUsed = [];
  const serversUsed = [];
  
  // Look for function patterns in the response
  const functionPatterns = [
    // Study Scheduler
    /checkAvailability\(\)|bookRoom\(\)|sendInvites\(\)/g,
    // Math Solver  
    /solveEquation\(\)|showSteps\(\)|plotGraph\(\)/g,
    // Activity Planner
    /getCurrentWeather\(\)|getForecast\(\)|checkConditions\(\)/g,
    // Language Tutor
    /translateText\(\)|checkGrammar\(\)|generateAudio\(\)/g,
    // Study Beats
    /createPlaylist\(\)|analyzeGenre\(\)|setVolume\(\)/g,
    // Research Assistant
    /searchArticles\(\)|verifySource\(\)|downloadPDF\(\)/g,
    // Knowledge Base
    /searchFacts\(\)|verifyClaims\(\)|findCitations\(\)/g,
    // Navigation Helper
    /findRoute\(\)|checkTraffic\(\)|locatePlace\(\)/g
  ];

  const serverFunctionMap = {
    'Study Scheduler': ['checkAvailability()', 'bookRoom()', 'sendInvites()'],
    'Math Solver': ['solveEquation()', 'showSteps()', 'plotGraph()'],
    'Activity Planner': ['getCurrentWeather()', 'getForecast()', 'checkConditions()'],
    'Language Tutor': ['translateText()', 'checkGrammar()', 'generateAudio()'],
    'Study Beats': ['createPlaylist()', 'analyzeGenre()', 'setVolume()'],
    'Research Assistant': ['searchArticles()', 'verifySource()', 'downloadPDF()'],
    'Knowledge Base': ['searchFacts()', 'verifyClaims()', 'findCitations()'],
    'Navigation Helper': ['findRoute()', 'checkTraffic()', 'locatePlace()']
  };

  // Check for function usage in the response, but only for enabled servers
  const enabledServerNames = enabledServers.map(s => s.name);
  
  for (const [serverName, functions] of Object.entries(serverFunctionMap)) {
    // Only check for functions if this server is actually enabled
    if (enabledServerNames.includes(serverName)) {
      for (const func of functions) {
        if (text.includes(func)) {
          if (!functionsUsed.includes(func)) {
            functionsUsed.push(func);
          }
          if (!serversUsed.includes(serverName)) {
            serversUsed.push(serverName);
          }
        }
      }
    }
  }

  return { functionsUsed, serversUsed };
};

module.exports = async function handler(req, res) {
  // Set CORS headers
  res.setHeader('Access-Control-Allow-Credentials', true);
  res.setHeader('Access-Control-Allow-Origin', '*');
  res.setHeader('Access-Control-Allow-Methods', 'GET,OPTIONS,PATCH,DELETE,POST,PUT');
  res.setHeader(
    'Access-Control-Allow-Headers',
    'X-CSRF-Token, X-Requested-With, Accept, Accept-Version, Content-Length, Content-MD5, Content-Type, Date, X-Api-Version'
  );

  if (req.method === 'OPTIONS') {
    res.status(200).end();
    return;
  }

  if (req.method !== 'POST') {
    res.status(405).json({ error: 'Method not allowed' });
    return;
  }

  try {
    console.log('API request received:', req.method);
    console.log('Request body:', req.body);
    
    const { message, enabledServers } = req.body;

    if (!message) {
      console.error('No message provided');
      res.status(400).json({ error: 'Message is required' });
      return;
    }

    if (!process.env.ANTHROPIC_API_KEY) {
      console.error('Anthropic API key not configured');
      res.status(500).json({ error: 'Anthropic API key not configured' });
      return;
    }

    console.log('Generating prompt for message:', message);
    const prompt = generateMCPPrompt(message, enabledServers || []);
    
    console.log('Calling Anthropic API...');
    
    // Initialize Anthropic client here instead of at module level
    const anthropicClient = new Anthropic({
      apiKey: process.env.ANTHROPIC_API_KEY,
    });
    
    const response = await anthropicClient.messages.create({
      model: "claude-3-5-sonnet-20241022",
      max_tokens: 150,
      system: "Only use real functions from enabled servers. For general requests, do NOT add function usage.",
      messages: [{ role: "user", content: prompt }]
    });

    console.log('Anthropic API response received');
    const aiResponse = response.content[0].text;
    const { functionsUsed, serversUsed } = extractFunctionCalls(aiResponse, enabledServers || []);
    
    console.log('Sending successful response');
    res.status(200).json({
      response: aiResponse,
      functionsUsed,
      serversUsed
    });

  } catch (error) {
    console.error('API Error details:', {
      message: error.message,
      stack: error.stack,
      name: error.name
    });
    res.status(500).json({ 
      error: 'Failed to process request',
      details: error.message,
      timestamp: new Date().toISOString()
    });
  }
}