const express = require('express');
const cors = require('cors');
const path = require('path');
const Anthropic = require('@anthropic-ai/sdk');
require('dotenv').config();

const app = express();
const PORT = process.env.PORT || 5001;

// Initialize Anthropic client
const anthropic = new Anthropic({
  apiKey: process.env.ANTHROPIC_API_KEY,
});

// Middleware
app.use(cors());
app.use(express.json());

// Serve static files from the React app build directory
app.use(express.static(path.join(__dirname, 'build')));

// Generate MCP-aware prompt for Anthropic API
const generateMCPPrompt = (message, enabledServers) => {
  if (enabledServers.length === 0) {
    return `You are ARIA, a helpful AI assistant for high school students. Currently, you don't have access to any MCP (Model Context Protocol) servers, so you can only provide general advice based on your training data.

User's question: "${message}"

Since no MCP servers are enabled, respond with: "I don't have access to any MCP servers right now. Please enable some servers to unlock my enhanced capabilities! I can help with scheduling, weather, math problems, translations, research, and more once you enable the relevant servers."`;
  }

  const serverDescriptions = enabledServers.map(server => 
    `• ${server.name}: ${server.description}
   Available functions: ${server.functions.join(', ')}`
  ).join('\n');

  // Create explicit constraint checking
  const serverNames = enabledServers.map(s => s.name);
  const hasScheduler = serverNames.includes('Study Scheduler');
  const hasMath = serverNames.includes('Math Solver');
  const hasWeather = serverNames.includes('Activity Planner');
  const hasTranslation = serverNames.includes('Language Tutor');
  const hasMusic = serverNames.includes('Study Beats');

  // Determine request type and required server
  const getRequestAnalysis = (message) => {
    const msg = message.toLowerCase();
    if (msg.includes('schedul') || msg.includes('calendar') || msg.includes('availab') || msg.includes('book') || msg.includes('meet')) {
      return { type: 'scheduling', server: 'Study Scheduler', hasServer: hasScheduler };
    }
    if (msg.includes('math') || msg.includes('calculat') || msg.includes('equation') || msg.includes('solve') || msg.includes('formula')) {
      return { type: 'math', server: 'Math Solver', hasServer: hasMath };
    }
    if (msg.includes('weather') || msg.includes('forecast') || msg.includes('temperature') || msg.includes('condition')) {
      return { type: 'weather', server: 'Activity Planner', hasServer: hasWeather };
    }
    if (msg.includes('translat') || msg.includes('language') || msg.includes('grammar') || 
        (msg.includes('what is') && (msg.includes('in ') || msg.includes('french') || msg.includes('spanish') || msg.includes('german') || msg.includes('italian') || msg.includes('portuguese') || msg.includes('chinese') || msg.includes('japanese')))) {
      return { type: 'translation', server: 'Language Tutor', hasServer: hasTranslation };
    }
    if (msg.includes('music') || msg.includes('playlist') || msg.includes('song') || msg.includes('audio')) {
      return { type: 'music', server: 'Study Beats', hasServer: hasMusic };
    }
    return { type: 'general', server: null, hasServer: true }; // Allow general requests
  };

  const analysis = getRequestAnalysis(message);

  if (!analysis.hasServer) {
    return `You don't have ${analysis.server}. Respond exactly: "I don't have access to the right MCP servers to help with ${analysis.type}. You may need to enable ${analysis.server} to get help with ${analysis.type} functions."`;
  }

  if (analysis.server) {
    const serverFunctions = enabledServers.find(s => s.name === analysis.server)?.functions.join(', ') || '';
    return `${message}

Server: ${analysis.server} 
Functions: ${serverFunctions}

Answer the user's request and end with: "🔧 Functions used: translateText()" (or whichever function you used)`;
  }

  return `${message}. Give brief helpful advice.`;
};

// Real AI response function using Anthropic API
const generateAIResponse = async (message, enabledServers) => {
  try {
    if (!process.env.ANTHROPIC_API_KEY) {
      throw new Error('ANTHROPIC_API_KEY not configured');
    }

    const prompt = generateMCPPrompt(message, enabledServers);
    
    console.log('=== DEBUGGING PROMPT ===');
    console.log('Enabled servers:', enabledServers.map(s => s.name));
    console.log('Available functions:', enabledServers.flatMap(s => s.functions));
    console.log('Full prompt sent to AI:');
    console.log(prompt);
    console.log('========================');
    
    const response = await anthropic.messages.create({
      model: "claude-3-5-sonnet-20241022",
      max_tokens: 150, // Allow for function reporting
      system: "Only use real functions from enabled servers. For general requests, do NOT add function usage.",
      messages: [
        {
          role: "user",
          content: prompt
        }
      ]
    });

    const aiResponse = response.content[0].text;
    
    console.log('AI Response:', aiResponse); // Debug logging
    
    // Extract which servers and functions were used
    let serversUsed = [];
    let functionsUsed = [];
    
    // Extract functions from the "🔧 Functions used:" line first (most reliable)
    const functionsMatch = aiResponse.match(/🔧 Functions used:\s*([^\n]+)/);
    if (functionsMatch) {
      let rawFunctions = functionsMatch[1].trim();
      
      // Clean up the functions string by removing all brackets and parsing properly
      const extractedFunctions = rawFunctions
        .replace(/[\[\]]/g, '') // Remove all brackets
        .split(',')
        .map(f => f.trim())
        .filter(f => f.length > 0);
      
      // Only keep functions that actually exist in enabled servers
      const validFunctions = [];
      const allServerFunctions = enabledServers.flatMap(server => server.functions);
      
      extractedFunctions.forEach(func => {
        if (allServerFunctions.includes(func)) {
          validFunctions.push(func);
        } else {
          console.log('Filtering out invalid function:', func); // Debug logging
        }
      });
      
      functionsUsed = validFunctions;
      console.log('Functions extracted from 🔧 line:', functionsUsed); // Debug logging
    }
    
    // If no functions were extracted but we should have used them, infer based on request type
    if (functionsUsed.length === 0) {
      const msg = message.toLowerCase();
      if (msg.includes('translat') || msg.includes('french') || msg.includes('spanish') || (msg.includes('what is') && msg.includes('in '))) {
        const languageServer = enabledServers.find(s => s.name === 'Language Tutor');
        if (languageServer) {
          functionsUsed.push('translateText()');
          console.log('Inferred translation function usage'); // Debug logging
        }
      } else if (msg.includes('playlist') || msg.includes('music')) {
        const musicServer = enabledServers.find(s => s.name === 'Study Beats');
        if (musicServer) {
          functionsUsed.push('createPlaylist()');
          console.log('Inferred music function usage'); // Debug logging
        }
      } else if (msg.includes('schedul')) {
        const scheduleServer = enabledServers.find(s => s.name === 'Study Scheduler');
        if (scheduleServer) {
          functionsUsed.push('checkAvailability()');
          console.log('Inferred scheduling function usage'); // Debug logging
        }
      }
    }
    
    // Also check for functions mentioned in the response text
    enabledServers.forEach(server => {
      server.functions.forEach(func => {
        if (aiResponse.includes(func) && !functionsUsed.includes(func)) {
          functionsUsed.push(func);
        }
      });
    });
    
    // Now determine which servers were used based on the functions called
    enabledServers.forEach(server => {
      const serverFunctionsUsed = server.functions.filter(func => 
        functionsUsed.some(usedFunc => usedFunc === func)
      );
      
      if (serverFunctionsUsed.length > 0) {
        serversUsed.push(server.name);
      }
    });
    
    console.log('Final servers used:', serversUsed); // Debug logging
    console.log('Final functions used:', functionsUsed); // Debug logging
    
    // Clean up the response by removing the functions line and extra quotes
    let cleanResponse = aiResponse;
    if (functionsMatch) {
      cleanResponse = aiResponse.replace(/🔧 Functions used:\s*[^\n]+\n?/g, '').trim();
    }
    
    // Remove extra quotes if the entire response is quoted
    if (cleanResponse.startsWith('"') && cleanResponse.endsWith('"')) {
      cleanResponse = cleanResponse.slice(1, -1);
    }
    
    return {
      response: cleanResponse,
      serversUsed,
      functionsUsed
    };
  } catch (error) {
    console.error('Error calling Anthropic API:', error);
    
    // Fallback response if API fails
    if (error.message.includes('ANTHROPIC_API_KEY')) {
      return {
        response: "⚠️ API key not configured. Please add your Anthropic API key to the .env file to enable real AI responses.",
        serversUsed: [],
        functionsUsed: []
      };
    }
    
    return {
      response: "Sorry, I encountered an error processing your request. Please try again in a moment.",
      serversUsed: [],
      functionsUsed: []
    };
  }
};

// API Routes
app.post('/api/chat', async (req, res) => {
  try {
    console.log('Chat API called with:', { message: req.body.message, serverCount: req.body.enabledServers?.length });
    const { message, enabledServers } = req.body;
    
    if (!message || typeof message !== 'string') {
      return res.status(400).json({ error: 'Message is required and must be a string' });
    }
    
    // Generate AI response
    const result = await generateAIResponse(message, enabledServers || []);
    
    res.json(result);
  } catch (error) {
    console.error('Error in /api/chat:', error);
    res.status(500).json({ error: 'Internal server error' });
  }
});

// Health check endpoint
app.get('/api/health', (req, res) => {
  res.json({ status: 'ok', message: 'MCP Explorer API is running' });
});


// The "catchall" handler: send back React's index.html file for any non-API routes
app.get('*', (req, res) => {
  res.sendFile(path.join(__dirname, 'build', 'index.html'));
});

app.listen(PORT, () => {
  console.log(`Server is running on port ${PORT}`);
});

module.exports = app;