import React, { useState, useEffect } from 'react';
import { motion, AnimatePresence } from 'framer-motion';
import useProgress from './hooks/useProgress';
import NameInput from './pages/NameInput';
import Introduction from './pages/Introduction';
import Discovery from './pages/Discovery';
import TechnicalOverview from './pages/TechnicalOverview';
import Exploration from './pages/Exploration';
import Playground from './pages/Playground';
import Challenge from './pages/Challenge';
import RealAI from './pages/RealAI';
import Congratulations from './pages/Congratulations';
import QuizPage from './pages/Quiz';
import ProgressBar from './components/ProgressBar';
import SkipConfirmationModal from './components/SkipConfirmationModal';

const STEPS = [
  { id: 0, name: 'Introduction', component: Introduction },
  { id: 1, name: 'Discovery', component: Discovery },
  { id: 2, name: 'How It Works', component: TechnicalOverview },
  { id: 3, name: 'Exploration', component: Exploration },
  { id: 4, name: 'Playground', component: Playground },
  { id: 5, name: 'Try Live', component: RealAI },
  { id: 6, name: 'Challenge', component: Challenge },
  { id: 7, name: 'Quiz', component: QuizPage }
];


function App() {
  const { currentStep, setStep, stepTimes, userName } = useProgress();
  const [showSkipModal, setShowSkipModal] = useState(false);
  const [targetStep, setTargetStep] = useState(0);
  const [showCongratulations, setShowCongratulations] = useState(false);
  
  // Determine which component to show
  let CurrentStepComponent, currentStepForProgressBar;
  if (showCongratulations) {
    console.log('Showing congratulations page');
    CurrentStepComponent = Congratulations;
    currentStepForProgressBar = 6; // Keep Challenge highlighted in progress bar
  } else {
    const step = STEPS[currentStep];
    CurrentStepComponent = step.component;
    currentStepForProgressBar = currentStep;
  }
  
  console.log('Current step:', currentStep, 'Show congratulations:', showCongratulations);
  
  // Initialize timing for the first step
  useEffect(() => {
    if (userName && !stepTimes[0]) {
      setStep(0); // This will initialize timing for step 0
    }
  }, [stepTimes, setStep, userName]);
  
  // Show name input if no name is set
  if (!userName) {
    return (
      <div className="min-h-screen bg-gradient-to-br from-blue-50 to-indigo-100">
        <NameInput />
      </div>
    );
  }

  const handleStepClick = (stepIndex) => {
    if (stepIndex < currentStep) {
      setStep(stepIndex);
    } else if (stepIndex > currentStep) {
      setTargetStep(stepIndex);
      setShowSkipModal(true);
    }
  };

  const handleSkipConfirm = () => {
    setStep(targetStep);
    setShowSkipModal(false);
  };

  const handleSkipCancel = () => {
    setShowSkipModal(false);
  };

  // Custom next step handler for Challenge page
  const handleChallengeComplete = () => {
    console.log('Challenge complete - showing congratulations');
    setShowCongratulations(true);
    // Scroll to top when showing congratulations
    window.scrollTo({ top: 0, behavior: 'smooth' });
  };

  // Custom next step handler for Congratulations page
  const handleCongratulationsNext = () => {
    setShowCongratulations(false);
    setStep(7); // Go to Quiz (step 7)
    // Scroll to top when going to quiz
    window.scrollTo({ top: 0, behavior: 'smooth' });
  };

  return (
    <div className="min-h-screen bg-gradient-to-br from-blue-50 to-indigo-100">
      <ProgressBar steps={STEPS} currentStep={currentStepForProgressBar} onStepClick={handleStepClick} />
      
      <AnimatePresence mode="wait">
        <motion.div
          key={showCongratulations ? 'congratulations' : currentStep}
          initial={{ opacity: 0 }}
          animate={{ opacity: 1 }}
          exit={{ opacity: 0 }}
          transition={{ duration: 0.2 }}
          className="container mx-auto px-4 py-8"
        >
          {showCongratulations ? (
            React.createElement(CurrentStepComponent, { nextStep: handleCongratulationsNext })
          ) : currentStep === 6 ? (
            React.createElement(CurrentStepComponent, { nextStep: handleChallengeComplete })
          ) : (
            React.createElement(CurrentStepComponent)
          )}
        </motion.div>
      </AnimatePresence>
      
      <SkipConfirmationModal
        isOpen={showSkipModal}
        onClose={handleSkipCancel}
        onConfirm={handleSkipConfirm}
        targetStep={targetStep}
      />
    </div>
  );
}

export default App;