import React from 'react';
import { motion, AnimatePresence } from 'framer-motion';
import { X, Power, AlertCircle, CheckCircle, Code } from 'lucide-react';

// Helper function to get server functions with explanations
const getServerFunctions = (serverId) => {
  const functionDetails = {
    'calendar': [
      { name: 'checkAvailability()', description: 'Checks when people are free and finds common available time slots' },
      { name: 'bookRoom()', description: 'Reserves study rooms, library spaces, or meeting locations' },
      { name: 'sendInvites()', description: 'Sends calendar invitations to group members with meeting details' }
    ],
    'weather': [
      { name: 'getCurrentWeather()', description: 'Gets real-time weather conditions for any location' },
      { name: 'getForecast()', description: 'Provides detailed weather predictions for upcoming days' },
      { name: 'checkConditions()', description: 'Analyzes weather suitability for specific outdoor activities' }
    ],
    'maps': [
      { name: 'findRoute()', description: 'Calculates the fastest or safest path between two locations' },
      { name: 'checkTraffic()', description: 'Monitors current traffic conditions and suggests alternate routes' },
      { name: 'locatePlace()', description: 'Finds nearby businesses, landmarks, or points of interest' }
    ],
    'music': [
      { name: 'createPlaylist()', description: 'Builds custom playlists based on mood, activity, or preferences' },
      { name: 'analyzeGenre()', description: 'Identifies music styles that best match your study or work needs' },
      { name: 'setVolume()', description: 'Adjusts audio levels for optimal focus without distraction' }
    ],
    'news': [
      { name: 'searchArticles()', description: 'Finds current news articles and reports on specific topics' },
      { name: 'verifySource()', description: 'Checks credibility and reliability of news sources for research' },
      { name: 'downloadPDF()', description: 'Retrieves full articles and reports for offline reading' }
    ],
    'translation': [
      { name: 'translateText()', description: 'Converts text between different languages accurately' },
      { name: 'checkGrammar()', description: 'Reviews and corrects grammar, spelling, and sentence structure' },
      { name: 'generateAudio()', description: 'Creates pronunciation guides and spoken examples' }
    ],
    'calculator': [
      { name: 'solveEquation()', description: 'Computes complex mathematical problems and calculations' },
      { name: 'showSteps()', description: 'Provides step-by-step solutions to help you learn the process' },
      { name: 'plotGraph()', description: 'Creates visual graphs and charts for equations and data' }
    ],
    'wikipedia': [
      { name: 'searchFacts()', description: 'Looks up verified information from reliable knowledge sources' },
      { name: 'verifyClaims()', description: 'Cross-checks information against multiple trusted sources' },
      { name: 'findCitations()', description: 'Provides proper citations and references for research papers' }
    ],
    'timer': [
      { name: 'setReminder()', description: 'Creates alerts for assignments, tests, and important deadlines' },
      { name: 'trackTime()', description: 'Monitors study sessions and productivity periods' },
      { name: 'organizeDeadlines()', description: 'Sorts and prioritizes upcoming assignments and tasks' }
    ],
    'social': [
      { name: 'analyzeEngagement()', description: 'Studies how your posts perform and suggests improvements' },
      { name: 'predictTiming()', description: 'Identifies the best times to post for maximum visibility' },
      { name: 'checkTrends()', description: 'Monitors popular topics and hashtags relevant to your interests' }
    ]
  };
  
  return functionDetails[serverId] || [];
};

const ComparisonModal = ({ server, isOpen, onClose, isEnabled, onToggle }) => {
  if (!isOpen) return null;

  const Icon = server.icon;

  return (
    <AnimatePresence>
      <div className="fixed inset-0 bg-black/50 flex items-center justify-center z-50 p-4">
        <motion.div
          initial={{ scale: 0.9, opacity: 0 }}
          animate={{ scale: 1, opacity: 1 }}
          exit={{ scale: 0.9, opacity: 0 }}
          className="bg-white rounded-2xl shadow-2xl max-w-3xl w-full max-h-[90vh] overflow-hidden flex flex-col"
        >
          <div className="p-6 border-b bg-white sticky top-0 z-10">
            <div className="flex items-center justify-between">
              <div className="flex items-center">
                <div className={`inline-flex items-center justify-center w-12 h-12 ${server.color} bg-opacity-10 rounded-xl mr-3`}>
                  <Icon className={`w-6 h-6 text-white ${server.color.replace('bg-', 'text-').replace('500', '600')}`} />
                </div>
                <div className="flex-1">
                  <h2 className="text-2xl font-bold text-gray-800">{server.name}</h2>
                  <p className="text-sm text-gray-600">{server.useCase}</p>
                </div>
              </div>
              
              <div className="flex items-center space-x-4">
                <div className="flex items-center space-x-3">
                  <span className="text-sm text-gray-600">Enable this server:</span>
                  <button
                    onClick={onToggle}
                    className={`w-14 h-7 rounded-full transition-colors relative ${
                      isEnabled ? 'bg-primary' : 'bg-gray-300'
                    }`}
                  >
                    <motion.div
                      animate={{ x: isEnabled ? 28 : 0 }}
                      transition={{ type: "spring", stiffness: 500, damping: 30 }}
                      className="absolute top-1 left-1 w-5 h-5 bg-white rounded-full shadow-md"
                    />
                  </button>
                  <span className="text-sm font-medium">
                    {isEnabled ? 'Enabled' : 'Disabled'}
                  </span>
                </div>
                
                <button
                  onClick={onClose}
                  className="p-2 hover:bg-gray-100 rounded-full transition-colors"
                >
                  <X className="w-5 h-5 text-gray-500" />
                </button>
              </div>
            </div>
          </div>

          <div className="p-6 overflow-y-auto flex-1">
            <div className="mb-6 bg-blue-50 rounded-lg p-4">
              <div className="flex items-start">
                <Power className="w-5 h-5 text-primary mr-2 mt-0.5" />
                <div>
                  <p className="text-sm text-gray-700">
                    <strong>How it works:</strong> {server.name} connects ARIA to real {server.useCase.toLowerCase()} data, 
                    allowing her to provide accurate, up-to-date information and take actions on your behalf.
                  </p>
                </div>
              </div>
            </div>

            {/* Functions Section */}
            <div className="mb-6 bg-gray-50 rounded-lg p-4">
              <div className="flex items-center mb-3">
                <Code className="w-5 h-5 text-gray-600 mr-2" />
                <h3 className="font-semibold text-gray-800">Available Functions</h3>
              </div>
              <div className="space-y-3">
                {getServerFunctions(server.id).map((func, index) => (
                  <div key={index} className="bg-white rounded-lg p-3 border border-gray-200">
                    <div className="flex items-start">
                      <code className="text-sm font-mono bg-gray-100 px-2 py-1 rounded text-blue-600 mr-3 flex-shrink-0">
                        {func.name}
                      </code>
                      <p className="text-sm text-gray-700 leading-relaxed">
                        {func.description}
                      </p>
                    </div>
                  </div>
                ))}
              </div>
            </div>

            <div className="grid md:grid-cols-2 gap-6">
              <div className="bg-red-50 rounded-xl p-5">
                <div className="flex items-center mb-3">
                  <AlertCircle className="w-5 h-5 text-red-600 mr-2" />
                  <h3 className="font-semibold text-red-800">Without MCP</h3>
                </div>
                
                <div className="bg-white rounded-lg p-4 mb-3">
                  <p className="text-sm font-medium text-gray-700 mb-2">You ask:</p>
                  <p className="text-sm text-gray-600 italic">"{server.examples.without.query}"</p>
                </div>
                
                <div className="bg-white rounded-lg p-4">
                  <p className="text-sm font-medium text-gray-700 mb-2">AI responds:</p>
                  <p className="text-sm text-gray-600">{server.examples.without.response}</p>
                </div>
              </div>

              <div className="bg-green-50 rounded-xl p-5">
                <div className="flex items-center mb-3">
                  <CheckCircle className="w-5 h-5 text-green-600 mr-2" />
                  <h3 className="font-semibold text-green-800">With MCP</h3>
                </div>
                
                <div className="bg-white rounded-lg p-4 mb-3">
                  <p className="text-sm font-medium text-gray-700 mb-2">You ask:</p>
                  <p className="text-sm text-gray-600 italic">"{server.examples.with.query}"</p>
                </div>
                
                <div className="bg-white rounded-lg p-4">
                  <p className="text-sm font-medium text-gray-700 mb-2">AI responds:</p>
                  <p className="text-sm text-gray-600 whitespace-pre-line">{server.examples.with.response}</p>
                </div>
              </div>
            </div>

          </div>
        </motion.div>
      </div>
    </AnimatePresence>
  );
};

export default ComparisonModal;