import React, { useState } from "react";
import { motion } from "framer-motion";
import { CheckCircle, XCircle, HelpCircle } from "lucide-react";
import useProgress from "../hooks/useProgress";

const Quiz = ({ onComplete }) => {
  const { updateQuizScore } = useProgress();
  const [currentQuestion, setCurrentQuestion] = useState(0);
  const [selectedAnswer, setSelectedAnswer] = useState(null);
  const [showFeedback, setShowFeedback] = useState(false);
  const [correctAnswers, setCorrectAnswers] = useState(0);

  const questions = [
    {
      question: "What does MCP stand for?",
      options: [
        "Model Context Protocol",
        "Model Connection Protocol",
        "Machine Context Platform",
        "Modern Communication Protocol",
      ],
      correct: 0,
      explanation:
        "MCP stands for Model Context Protocol - a standardized way for AI to connect with external tools.",
    },
    {
      question: "What is the main benefit of using MCP servers?",
      options: [
        "They make AI responses faster",
        "They let AI access real-time data and tools",
        "They reduce computational costs",
        "They improve AI memory capacity",
      ],
      correct: 1,
      explanation:
        "MCP servers allow AI to connect to real tools and data sources, providing current information and enabling actions.",
    },
    {
      question: "What type of data can MCP servers access?",
      options: [
        "Only local text files",
        "Only cloud databases",
        "Various data sources like files, APIs, and databases",
        "Only web-based content",
      ],
      correct: 2,
      explanation:
        "MCP servers are versatile and can connect to many different data sources including files, APIs, databases, and more!",
    },
    {
      question: "What happens when you enable multiple MCP servers?",
      options: [
        "Only the first server remains active",
        "Each server operates independently",
        "They work together to provide more capabilities",
        "The system automatically selects one server",
      ],
      correct: 2,
      explanation:
        "Multiple MCP servers work together, giving your AI assistant combined capabilities for more complex tasks.",
    },
    {
      question:
        "What is the trade-off when connecting more MCP servers to your AI?",
      options: [
        "More servers make AI less accurate",
        "More servers cost more money to use",
        "More servers provide amazing capabilities but take longer to process",
        "More servers make AI forget previous conversations",
      ],
      correct: 2,
      explanation:
        "The 'Magic Balance' shows that while more MCP servers give your AI amazing capabilities, it takes a bit longer for the AI to figure out which tools to use for each request. Fewer servers mean super fast responses but with fewer abilities! However, modern AI architectures are continuously optimizing this trade-off to make multi-server processing faster and more efficient.",
    },
  ];

  const handleAnswer = (answerIndex) => {
    setSelectedAnswer(answerIndex);
    setShowFeedback(true);

    if (answerIndex === questions[currentQuestion].correct) {
      setCorrectAnswers(correctAnswers + 1);
      updateQuizScore(1);
    }
  };

  const handleNext = () => {
    if (currentQuestion < questions.length - 1) {
      setCurrentQuestion(currentQuestion + 1);
      setSelectedAnswer(null);
      setShowFeedback(false);
    } else {
      onComplete();
    }
  };

  const question = questions[currentQuestion];
  const isCorrect = selectedAnswer === question.correct;

  return (
    <div className='max-w-3xl mx-auto'>
      <motion.div
        initial={{ opacity: 0, y: 20 }}
        animate={{ opacity: 1, y: 0 }}
        className='text-center mb-8'
      >
        <h2 className='text-3xl font-bold text-gray-800 mb-4'>
          Quick Knowledge Check!
        </h2>
        <p className='text-gray-600'>Let's see what you've learned about MCP</p>
      </motion.div>

      <div className='bg-white rounded-2xl shadow-xl p-8'>
        <div className='mb-6'>
          <div className='flex justify-between items-center mb-4'>
            <span className='text-sm text-gray-500'>
              Question {currentQuestion + 1} of {questions.length}
            </span>
            <span className='text-sm font-medium text-primary'>
              Score: {correctAnswers}/{currentQuestion + (showFeedback ? 1 : 0)}
            </span>
          </div>

          <div className='h-2 bg-gray-200 rounded-full overflow-hidden'>
            <motion.div
              className='h-full bg-gradient-to-r from-primary to-secondary'
              initial={{ width: 0 }}
              animate={{
                width: `${((currentQuestion + 1) / questions.length) * 100}%`,
              }}
              transition={{ duration: 0.5 }}
            />
          </div>
        </div>

        <motion.div
          key={currentQuestion}
          initial={{ opacity: 0, x: 20 }}
          animate={{ opacity: 1, x: 0 }}
          transition={{ duration: 0.3 }}
        >
          <h3 className='text-xl font-semibold text-gray-800 mb-6 flex items-start'>
            <HelpCircle className='w-6 h-6 text-primary mr-3 mt-0.5 flex-shrink-0' />
            {question.question}
          </h3>

          <div className='space-y-3 mb-6'>
            {question.options.map((option, index) => (
              <motion.button
                key={index}
                onClick={() => !showFeedback && handleAnswer(index)}
                disabled={showFeedback}
                whileHover={!showFeedback ? { scale: 1.02 } : {}}
                whileTap={!showFeedback ? { scale: 0.98 } : {}}
                className={`w-full text-left p-4 rounded-lg border-2 transition-all ${
                  showFeedback
                    ? index === question.correct
                      ? "border-green-500 bg-green-50"
                      : index === selectedAnswer && !isCorrect
                      ? "border-red-500 bg-red-50"
                      : "border-gray-200 bg-gray-50"
                    : selectedAnswer === index
                    ? "border-primary bg-blue-50"
                    : "border-gray-200 hover:border-primary hover:bg-blue-50"
                } ${showFeedback ? "cursor-not-allowed" : "cursor-pointer"}`}
              >
                <div className='flex items-center justify-between'>
                  <span
                    className={`${
                      showFeedback && index === question.correct
                        ? "font-medium text-green-700"
                        : showFeedback && index === selectedAnswer && !isCorrect
                        ? "text-red-700"
                        : "text-gray-700"
                    }`}
                  >
                    {option}
                  </span>
                  {showFeedback && index === question.correct && (
                    <CheckCircle className='w-5 h-5 text-green-500' />
                  )}
                  {showFeedback && index === selectedAnswer && !isCorrect && (
                    <XCircle className='w-5 h-5 text-red-500' />
                  )}
                </div>
              </motion.button>
            ))}
          </div>

          {showFeedback && (
            <motion.div
              initial={{ opacity: 0, y: -10 }}
              animate={{ opacity: 1, y: 0 }}
              className={`p-4 rounded-lg mb-6 ${
                isCorrect
                  ? "bg-green-50 border border-green-200"
                  : "bg-blue-50 border border-blue-200"
              }`}
            >
              <p
                className={`text-sm ${
                  isCorrect ? "text-green-700" : "text-blue-700"
                }`}
              >
                {isCorrect ? "✨ Correct! " : "💡 Not quite. "}
                {question.explanation}
              </p>
            </motion.div>
          )}
        </motion.div>

        {showFeedback && (
          <motion.div
            initial={{ opacity: 0 }}
            animate={{ opacity: 1 }}
            className='text-center'
          >
            <button
              onClick={handleNext}
              className='px-8 py-3 bg-gradient-to-r from-primary to-secondary text-white font-medium rounded-full hover:shadow-lg transition-shadow'
            >
              {currentQuestion < questions.length - 1
                ? "Next Question"
                : "See Your Results"}
            </button>
          </motion.div>
        )}
      </div>
    </div>
  );
};

export default Quiz;
