import { create } from 'zustand';

const useProgress = create((set, get) => ({
  currentStep: 0,
  completedSteps: [],
  quizScore: 0,
  enabledServers: [],
  startTime: Date.now(),
  stepTimes: {}, // tracks time spent on each step
  totalTimeSpent: 0,
  userName: '',
  
  nextStep: () => set((state) => ({ 
    currentStep: state.currentStep + 1,
    completedSteps: [...state.completedSteps, state.currentStep]
  })),
  
  previousStep: () => set((state) => ({ 
    currentStep: Math.max(0, state.currentStep - 1) 
  })),
  
  setStep: (step) => set((state) => {
    const now = Date.now();
    const timeOnCurrentStep = now - (state.stepTimes[state.currentStep]?.startTime || state.startTime);
    
    return {
      currentStep: step,
      stepTimes: {
        ...state.stepTimes,
        [state.currentStep]: {
          ...state.stepTimes[state.currentStep],
          totalTime: (state.stepTimes[state.currentStep]?.totalTime || 0) + timeOnCurrentStep
        },
        [step]: {
          ...state.stepTimes[step],
          startTime: now
        }
      }
    };
  }),
  
  toggleServer: (serverId) => set((state) => ({
    enabledServers: state.enabledServers.includes(serverId)
      ? state.enabledServers.filter(id => id !== serverId)
      : [...state.enabledServers, serverId]
  })),
  
  resetServers: () => set({ enabledServers: [] }),
  
  updateQuizScore: (points) => set((state) => ({
    quizScore: state.quizScore + points
  })),
  
  setUserName: (name) => set({ userName: name }),
  
  getTotalTimeSpent: () => {
    const state = get();
    const now = Date.now();
    const currentStepTime = now - (state.stepTimes[state.currentStep]?.startTime || state.startTime);
    
    let totalTime = currentStepTime;
    Object.values(state.stepTimes).forEach(step => {
      if (step.totalTime) totalTime += step.totalTime;
    });
    
    return totalTime;
  },
  
  getStepStats: () => {
    const state = get();
    const stepNames = ['Introduction', 'Discovery', 'Exploration', 'Playground', 'Conclusion'];
    
    return stepNames.map((name, index) => ({
      name,
      timeSpent: state.stepTimes[index]?.totalTime || 0,
      completed: state.completedSteps.includes(index) || index < state.currentStep
    }));
  },
  
  resetProgress: () => set({
    currentStep: 0,
    completedSteps: [],
    quizScore: 0,
    enabledServers: [],
    startTime: Date.now(),
    stepTimes: {},
    totalTimeSpent: 0,
    userName: ''
  })
}));

export default useProgress;