import React, { useState, useEffect } from "react";
import { motion } from "framer-motion";
import {
  ArrowRight,
  ArrowLeft,
  Target,
  CheckCircle,
  XCircle,
  AlertTriangle,
  Lightbulb,
  Sparkles,
  User,
} from "lucide-react";
import useProgress from "../hooks/useProgress";
import { mcpServers } from "../data/mcpServers";
import MarkdownText from "../components/MarkdownText";
import ServerCard from "../components/ServerCard";
import {
  getFunctionsForServer,
  getServerFunctionCount,
} from "../utils/serverFunctions";

// Helper function to get server functions with explanations
const getServerFunctions = (serverId) => {
  const functionDetails = {
    calendar: [
      {
        name: "checkAvailability()",
        description:
          "Checks when people are free and finds common available time slots",
      },
      {
        name: "bookRoom()",
        description:
          "Reserves study rooms, library spaces, or meeting locations",
      },
      {
        name: "sendInvites()",
        description:
          "Sends calendar invitations to group members with meeting details",
      },
    ],
    weather: [
      {
        name: "getCurrentWeather()",
        description: "Gets real-time weather conditions for any location",
      },
      {
        name: "getForecast()",
        description: "Provides detailed weather predictions for upcoming days",
      },
      {
        name: "checkConditions()",
        description:
          "Analyzes weather suitability for specific outdoor activities",
      },
    ],
    maps: [
      {
        name: "findRoute()",
        description:
          "Calculates the fastest or safest path between two locations",
      },
      {
        name: "checkTraffic()",
        description:
          "Monitors current traffic conditions and suggests alternate routes",
      },
      {
        name: "locatePlace()",
        description:
          "Finds nearby businesses, landmarks, or points of interest",
      },
    ],
    music: [
      {
        name: "createPlaylist()",
        description:
          "Builds custom playlists based on mood, activity, or preferences",
      },
      {
        name: "analyzeGenre()",
        description:
          "Identifies music styles that best match your study or work needs",
      },
      {
        name: "setVolume()",
        description:
          "Adjusts audio levels for optimal focus without distraction",
      },
    ],
    news: [
      {
        name: "searchArticles()",
        description:
          "Finds current news articles and reports on specific topics",
      },
      {
        name: "verifySource()",
        description:
          "Checks credibility and reliability of news sources for research",
      },
      {
        name: "downloadPDF()",
        description: "Retrieves full articles and reports for offline reading",
      },
    ],
    translation: [
      {
        name: "translateText()",
        description: "Converts text between different languages accurately",
      },
      {
        name: "checkGrammar()",
        description:
          "Reviews and corrects grammar, spelling, and sentence structure",
      },
      {
        name: "generateAudio()",
        description: "Creates pronunciation guides and spoken examples",
      },
    ],
    calculator: [
      {
        name: "solveEquation()",
        description: "Computes complex mathematical problems and calculations",
      },
      {
        name: "showSteps()",
        description:
          "Provides step-by-step solutions to help you learn the process",
      },
      {
        name: "plotGraph()",
        description: "Creates visual graphs and charts for equations and data",
      },
    ],
    wikipedia: [
      {
        name: "searchFacts()",
        description:
          "Looks up verified information from reliable knowledge sources",
      },
      {
        name: "verifyClaims()",
        description:
          "Cross-checks information against multiple trusted sources",
      },
      {
        name: "findCitations()",
        description:
          "Provides proper citations and references for research papers",
      },
    ],
    timer: [
      {
        name: "setReminder()",
        description:
          "Creates alerts for assignments, tests, and important deadlines",
      },
      {
        name: "trackTime()",
        description: "Monitors study sessions and productivity periods",
      },
      {
        name: "organizeDeadlines()",
        description: "Sorts and prioritizes upcoming assignments and tasks",
      },
    ],
    social: [
      {
        name: "analyzeEngagement()",
        description: "Studies how your posts perform and suggests improvements",
      },
      {
        name: "predictTiming()",
        description: "Identifies the best times to post for maximum visibility",
      },
      {
        name: "checkTrends()",
        description:
          "Monitors popular topics and hashtags relevant to your interests",
      },
    ],
  };

  return functionDetails[serverId] || [];
};

const challenges = [
  {
    id: 1,
    title: "Chapter 1: The Science Fair Presentation Crisis",
    scenario:
      "Sam, a junior at Westfield High, has a science fair presentation on renewable energy tomorrow, but just realized the research is outdated! The teacher emphasized using current data and credible sources. Can ARIA help Sam find recent articles and verify facts before the big presentation?",
    question:
      "Help me prepare my science fair presentation on renewable energy with current data and verified facts",
    requiredServers: ["news", "wikipedia"],
    requiredFunctions: [
      {
        server: "news",
        functions: ["searchArticles()", "verifySource()"],
        reason:
          "Find current renewable energy articles and check source credibility",
      },
      {
        server: "wikipedia",
        functions: ["searchFacts()", "verifyClaims()", "findCitations()"],
        reason: "Verify energy statistics and provide proper citations",
      },
    ],
    optimalAnswer:
      "Perfect timing, Sam! I've got your science fair presentation covered:\n\n**Current Research** (Research Helper):\n• 'Teen Invents Solar Panel Efficiency Breakthrough' - Science News (yesterday!)\n• 'School District Goes 100% Renewable' - Local Tribune (this week)\n• 'Youth Climate Action Impacts Policy' - National Geographic\n\n**Fact-Checked Data** (Fact Checker):\n✓ Solar costs dropped 85% since 2010 (verified from 3 sources)\n✓ Wind power now cheapest energy source (confirmed)\n✓ Your school's solar panels save $50,000/year (district verified)\n\nYou're going to crush this presentation! All sources are perfect for citation. 🌟🔬",
    limitedAnswer:
      "I can share some general knowledge about renewable energy from my training data, but I can't access current news articles or verify recent statistics for your science fair presentation.",
    hint: "Think about what makes a great science presentation: current research and verified facts...",
  },
  {
    id: 2,
    title: "Chapter 2: The College Visit Coordination Challenge",
    scenario:
      "After Sam's amazing science fair win, college visits are next! Sam's been accepted to visit three universities next weekend, but coordinating with parents, checking weather for outdoor campus tours, and managing everyone's schedules is overwhelming. Can ARIA help organize this important trip?",
    question:
      "Help me coordinate my college visit weekend with my parents, plan around the weather, and schedule the campus tours",
    requiredServers: ["calendar", "weather"],
    requiredFunctions: [
      {
        server: "calendar",
        functions: ["checkAvailability()", "bookRoom()", "sendInvites()"],
        reason: "Coordinate family schedules and book campus tour slots",
      },
      {
        server: "weather",
        functions: ["getForecast()", "checkConditions()"],
        reason: "Plan outdoor tours based on weather conditions",
      },
    ],
    optimalAnswer:
      "Congratulations on your science fair success, Sam! Let's nail this college visit:\n\n**Perfect Schedule** (Study Scheduler):\n• Saturday: State University (9am-2pm) - your parents are free\n• Sunday: Tech Institute (10am-3pm) - avoid your mom's meeting\n• Monday: Liberal Arts College (11am-4pm) - skip your free period\n\n**Weather Planning** (Activity Planner):\n• Saturday: 68°F, sunny (perfect for outdoor campus tour!)\n• Sunday: Light rain expected (indoor info sessions recommended)\n• Monday: Clear skies, 72°F (ideal for walking the campus)\n\nBonus: I found campus parking spots and sent your parents the GPS coordinates! 🎓🚗",
    limitedAnswer:
      "I can give general advice about college visits, but I can't check everyone's calendars, coordinate schedules, or provide weather forecasts to plan your campus tours.",
    hint: "College visits need careful coordination: scheduling and weather planning...",
  },
  {
    id: 3,
    title: "Chapter 3: The International Scholarship Application",
    scenario:
      "Sam's college visits went amazing, and now there's a prestigious international scholarship to apply for! The application requires: a personal statement translated to Spanish for the Madrid program, solving a complex calculus problem for the STEM scholarship, and calculating financial need percentages. It's due tonight - can ARIA help Sam finish strong?",
    question:
      "Help me complete my international scholarship application: translate my essay to Spanish, solve ∫(3x²+2x)dx for the STEM section, and calculate financial aid for a $50,000 program with 80% scholarship coverage",
    requiredServers: ["translation", "calculator"],
    requiredFunctions: [
      {
        server: "translation",
        functions: ["translateText()", "checkGrammar()"],
        reason: "Translate essay to academic-level Spanish with proper grammar",
      },
      {
        server: "calculator",
        functions: ["solveEquation()", "showSteps()"],
        reason:
          "Solve calculus integral and calculate scholarship percentages with work shown",
      },
    ],
    optimalAnswer:
      "¡Vamos, Sam! Let's secure that international scholarship:\n\n**Essay Translation** (Language Tutor):\nYour personal statement is beautifully translated to Spanish:\n• Enhanced cultural references for Spanish reviewers\n• Fixed 3 grammar nuances for native-level flow\n• Added proper academic formality expected in Spain\n\n**STEM Problem Solved** (Math Solver):\n∫(3x²+2x)dx = x³ + x² + C\n✓ Step-by-step solution attached with graphical proof\n✓ Work shown in competition format\n\n**Financial Calculation** (Math Solver):\n• Scholarship covers: $40,000 (80% of $50,000)\n• Your family pays: $10,000/year\n• 4-year total savings: $160,000!\n\nYour application is scholarship-worthy! From science fair winner to international scholar - you've got this! 🌍✨",
    limitedAnswer:
      "I can help with basic translation and math concepts, but I can't provide precise grammar checking for academic Spanish, verify complex calculus solutions, or handle detailed financial calculations for scholarship applications.",
    hint: "Scholarship applications demand precision: perfect translation and mathematical accuracy...",
  },
];

const Challenge = ({ nextStep: customNextStep }) => {
  const {
    nextStep: defaultNextStep,
    previousStep,
    enabledServers,
    toggleServer,
    resetServers,
  } = useProgress();
  const nextStep = customNextStep || defaultNextStep;
  const [showIntro, setShowIntro] = useState(true);
  const [showServerSelection, setShowServerSelection] = useState(false);
  const [currentChallenge, setCurrentChallenge] = useState(0);
  const [showResult, setShowResult] = useState(false);
  const [challengeResults, setChallengeResults] = useState([]);
  const [selectedServerForModal, setSelectedServerForModal] = useState(null);
  const [showFunctionModal, setShowFunctionModal] = useState(false);
  const [isReloading, setIsReloading] = useState(false);

  const challenge = challenges[currentChallenge];

  // Reset servers only when first entering the Challenge intro page to ensure clean start
  useEffect(() => {
    // Only reset if we're showing the intro page (first time entering challenges)
    if (showIntro && currentChallenge === 0) {
      resetServers();
    }
  }, []); // Only run once when component mounts

  // Reset result when challenge changes (must be before conditional return)
  useEffect(() => {
    if (!showIntro && !showServerSelection) {
      setShowResult(false);
      setChallengeResults((prev) =>
        prev.filter((r) => r.challengeId !== challenge.id)
      );
    }
  }, [currentChallenge, challenge.id, showIntro, showServerSelection]);

  // Don't reset results when servers change during story - let users adjust and manually re-test

  const handleServerToggle = (serverId) => {
    toggleServer(serverId);
  };

  // Show intro page first
  if (showIntro) {
    return (
      <div
        className='max-w-4xl mx-auto
      '
      >
        <motion.div
          initial={{ opacity: 0, y: 20 }}
          animate={{ opacity: 1, y: 0 }}
          className='text-center mb-12'
        >
          <div className='inline-flex items-center justify-center w-16 h-16 bg-gradient-to-r from-primary to-secondary rounded-full mb-6'>
            <Target className='w-8 h-8 text-white' />
          </div>

          <h2 className='text-4xl font-bold text-gray-800 mb-6'>
            Ready for a Challenge?
          </h2>
          <p className='text-xl text-gray-600 mb-8 max-w-2xl mx-auto'>
            Now that you've experienced MCP firsthand, let's put your
            understanding to the test with a real-world scenario.
          </p>
        </motion.div>

        <motion.div
          initial={{ opacity: 0, y: 30 }}
          animate={{ opacity: 1, y: 0 }}
          transition={{ delay: 0.2 }}
          className='bg-gradient-to-r from-blue-50 to-purple-50 rounded-2xl p-8 mb-8'
        >
          <div className='flex items-start space-x-4 mb-6'>
            <div className='flex-shrink-0'>
              <div className='w-12 h-12 bg-primary rounded-full flex items-center justify-center'>
                <User className='w-6 h-6 text-white' />
              </div>
            </div>
            <div>
              <h3 className='text-xl font-semibold text-gray-800 mb-2'>
                Meet Sam
              </h3>
              <p className='text-gray-600 mb-4'>
                Sam is a busy high school student juggling academics,
                extracurriculars, and social life. They need an AI assistant
                that can truly understand and help with their complex,
                multi-faceted needs.
              </p>
            </div>
          </div>

          <div className='bg-white rounded-xl p-6 border-l-4 border-primary'>
            <h4 className='font-semibold text-gray-800 mb-3 flex items-center'>
              <Sparkles className='w-5 h-5 text-primary mr-2' />
              Sam's Situation
            </h4>
            <p className='text-gray-700 mb-4'>
              "I have a huge history presentation due next week, and I need to
              coordinate with my study group, find reliable sources, and somehow
              fit it around my soccer practice schedule. Plus, I want to create
              some background music for the presentation. Can AI really help me
              with all of this?"
            </p>
            <div className='bg-gray-50 rounded-lg p-4'>
              <p className='text-sm text-gray-600 italic'>
                This is exactly the kind of complex, multi-domain challenge
                where MCP shines. Without it, AI would struggle to provide
                comprehensive help across all these different areas.
              </p>
            </div>
          </div>
        </motion.div>

        <motion.div
          initial={{ opacity: 0, y: 30 }}
          animate={{ opacity: 1, y: 0 }}
          transition={{ delay: 0.4 }}
          className='text-center mb-8'
        >
          <h3 className='text-2xl font-semibold text-gray-800 mb-4'>
            Your Challenge
          </h3>
          <p className='text-lg text-gray-600 max-w-3xl mx-auto'>
            Help Sam by choosing the right combination of MCP servers. You'll
            need to think about which servers can work together to address Sam's
            multi-faceted request effectively.
          </p>
        </motion.div>

        {/* Navigation */}
        <div className='flex justify-between items-center'>
          <button
            onClick={previousStep}
            className='inline-flex items-center px-6 py-3 text-gray-600 bg-gray-100 rounded-full hover:bg-gray-200 transition-colors'
          >
            <ArrowLeft className='mr-2 w-4 h-4' />
            Back to Try Live
          </button>

          <button
            onClick={() => {
              setShowIntro(false);
              setShowServerSelection(true);
            }}
            className='inline-flex items-center px-6 py-3 bg-gradient-to-r from-primary to-secondary text-white font-medium rounded-full hover:shadow-lg transition-shadow'
          >
            Choose Your MCP Servers
            <ArrowRight className='ml-2 w-4 h-4' />
          </button>
        </div>
      </div>
    );
  }

  // Server selection page
  if (showServerSelection) {
    return (
      <div className='max-w-5xl mx-auto py-12'>
        <motion.div
          initial={{ opacity: 0, y: 20 }}
          animate={{ opacity: 1, y: 0 }}
          className='text-center mb-10'
        >
          <h2 className='text-3xl font-bold text-gray-800 mb-4'>
            🎯 Choose ARIA's MCP Servers
          </h2>
          <p className='text-lg text-gray-600 mb-4'>
            Select the servers you think ARIA will need to help Sam through all
            three challenges.
          </p>
          <div className='bg-blue-50 border border-blue-200 rounded-xl py-3 px-6 inline-block'>
            <p className='text-sm text-blue-700'>
              💡 <strong>Think strategically:</strong> Sam needs help with
              research, scheduling, and complex problem-solving!
            </p>
          </div>
        </motion.div>

        <div className='grid md:grid-cols-2 lg:grid-cols-3 gap-6 mb-10'>
          {mcpServers.map((server) => (
            <ServerCard
              key={server.id}
              server={server}
              isEnabled={enabledServers.includes(server.id)}
              onToggle={() => handleServerToggle(server.id)}
              onClick={() => {
                setSelectedServerForModal(server);
                setShowFunctionModal(true);
              }}
            />
          ))}
        </div>

        {/* Selection Summary */}
        <div className='bg-white rounded-xl shadow-lg p-6 mb-8'>
          <h3 className='font-semibold text-gray-800 mb-4 flex items-center'>
            <Target className='w-5 h-5 text-primary mr-2' />
            Your Selection Summary
          </h3>

          {enabledServers.length > 0 ? (
            <div className='space-y-2'>
              <p className='text-sm text-gray-600 mb-3'>
                You've selected <strong>{enabledServers.length}</strong> server
                {enabledServers.length !== 1 ? "s" : ""} for ARIA to use:
              </p>
              <div className='flex flex-wrap gap-2'>
                {enabledServers.map((serverId) => {
                  const server = mcpServers.find((s) => s.id === serverId);
                  const Icon = server.icon;
                  return (
                    <div
                      key={serverId}
                      className='inline-flex items-center bg-primary/10 text-primary px-3 py-1 rounded-full text-sm'
                    >
                      <Icon className='w-4 h-4 mr-2' />
                      {server.name}
                    </div>
                  );
                })}
              </div>
            </div>
          ) : (
            <p className='text-gray-500 text-sm'>
              No servers selected yet. Choose the servers you think ARIA will
              need!
            </p>
          )}
        </div>

        {/* Navigation */}
        <div className='flex justify-between items-center'>
          <button
            onClick={() => {
              setShowServerSelection(false);
              setShowIntro(true);
            }}
            className='inline-flex items-center px-6 py-3 text-gray-600 bg-gray-100 rounded-full hover:bg-gray-200 transition-colors'
          >
            <ArrowLeft className='mr-2 w-4 h-4' />
            Back to Introduction
          </button>

          <button
            onClick={() => {
              setShowServerSelection(false);
            }}
            disabled={enabledServers.length === 0}
            className='inline-flex items-center px-6 py-3 bg-gradient-to-r from-primary to-secondary text-white font-medium rounded-full hover:shadow-lg transition-shadow disabled:opacity-50 disabled:cursor-not-allowed'
          >
            Start Sam's Story
            <ArrowRight className='ml-2 w-4 h-4' />
          </button>
        </div>

        {/* Function Details Modal */}
        {showFunctionModal && selectedServerForModal && (
          <div className='fixed inset-0 bg-black bg-opacity-50 flex items-center justify-center z-50 p-4'>
            <motion.div
              initial={{ opacity: 0, scale: 0.9 }}
              animate={{ opacity: 1, scale: 1 }}
              exit={{ opacity: 0, scale: 0.9 }}
              className='bg-white rounded-2xl shadow-xl max-w-2xl w-full max-h-[80vh] overflow-hidden'
            >
              <div className='p-6 border-b border-gray-100'>
                <div className='flex items-center justify-between'>
                  <div className='flex items-center'>
                    <selectedServerForModal.icon className='w-8 h-8 text-primary mr-3' />
                    <div>
                      <h3 className='text-xl font-semibold text-gray-800'>
                        {selectedServerForModal.name} Functions
                      </h3>
                      <p className='text-sm text-gray-600'>
                        {getServerFunctions(selectedServerForModal.id).length}{" "}
                        available functions
                      </p>
                    </div>
                  </div>
                  <button
                    onClick={() => {
                      setShowFunctionModal(false);
                      setSelectedServerForModal(null);
                    }}
                    className='p-2 hover:bg-gray-100 rounded-full transition-colors'
                  >
                    <XCircle className='w-6 h-6 text-gray-500' />
                  </button>
                </div>
              </div>

              <div className='p-6 overflow-y-auto max-h-[60vh]'>
                <div className='space-y-4'>
                  {getServerFunctions(selectedServerForModal.id).map(
                    (func, index) => (
                      <div
                        key={index}
                        className='bg-gray-50 rounded-lg p-4 border border-gray-100'
                      >
                        <div className='flex items-start justify-between mb-2'>
                          <code className='text-base font-mono text-blue-600 font-medium'>
                            {func.name}
                          </code>
                          <span className='text-xs bg-primary/10 text-primary px-2 py-1 rounded-full'>
                            Function
                          </span>
                        </div>
                        <p className='text-sm text-gray-700 leading-relaxed'>
                          {func.description}
                        </p>
                      </div>
                    )
                  )}
                </div>
              </div>

              <div className='p-6 border-t border-gray-100 bg-gray-50'>
                <div className='flex justify-between items-center'>
                  <div className='flex items-center text-sm text-gray-600'>
                    <Lightbulb className='w-4 h-4 mr-2' />
                    <span>
                      These functions help ARIA solve specific types of problems
                    </span>
                  </div>
                </div>
              </div>
            </motion.div>
          </div>
        )}
      </div>
    );
  }

  const solveChallenge = () => {
    // Reset reload state when solving normally
    setIsReloading(false);
    const hasRequiredServers = challenge.requiredServers.every((serverId) =>
      enabledServers.includes(serverId)
    );

    const extraServers = enabledServers.filter(
      (serverId) => !challenge.requiredServers.includes(serverId)
    );

    const result = {
      challengeId: challenge.id,
      success: hasRequiredServers,
      selectedServers: [...enabledServers],
      requiredServers: [...challenge.requiredServers],
      extraServers,
    };

    // Remove any previous result for this challenge
    setChallengeResults((prev) =>
      prev.filter((r) => r.challengeId !== challenge.id)
    );
    // Add the new result
    setChallengeResults((prev) => [...prev, result]);
    setShowResult(true);
  };

  const nextChallenge = () => {
    if (currentChallenge < challenges.length - 1) {
      setCurrentChallenge((prev) => prev + 1);
      setShowResult(false);
      // Keep server selection for the next challenge
    } else {
      nextStep(); // Go to quiz
    }
  };

  const resetChallenge = async () => {
    // Show reload animation
    setIsReloading(true);

    // Brief delay to show the reload animation
    await new Promise((resolve) => setTimeout(resolve, 800));

    // Instead of just hiding results, re-run the challenge with current servers
    solveChallenge();

    // Hide reload animation after solving
    setIsReloading(false);
  };

  const currentResult = challengeResults.find(
    (r) => r.challengeId === challenge.id
  );

  // Main challenges page
  return (
    <div className='max-w-6xl mx-auto py-12'>
      {/* Current Challenge Progress */}
      <motion.div
        initial={{ opacity: 0, y: 20 }}
        animate={{ opacity: 1, y: 0 }}
        className='text-center mb-10'
      >
        <h2 className='text-3xl font-bold text-gray-800 mb-4'>
          📖 Sam's Senior Year Journey
        </h2>
        <p className='text-lg text-gray-600 mb-4'>
          Follow Sam through three critical moments where ARIA's MCP powers make
          all the difference!
        </p>
        <div className='bg-gradient-to-r from-blue-50 to-indigo-50 border border-blue-200 rounded-xl py-4 px-8 mb-6 text-center max-w-2xl mx-auto shadow-sm'>
          <div className='flex items-center justify-center mb-2'>
            <Target className='w-4 h-4 text-blue-600 mr-2' />
            <span className='font-semibold text-blue-800 text-sm'>
              Your Mission
            </span>
          </div>
          <p className='text-sm text-gray-700'>
            Enable the right MCP servers to help Sam succeed in each challenge.
          </p>
        </div>
        <div className='inline-flex items-center bg-primary/10 px-4 py-2 rounded-full text-sm'>
          <Target className='w-4 h-4 text-primary mr-2' />
          <span className='text-gray-700'>
            Challenge <strong>{currentChallenge + 1}</strong> of{" "}
            {challenges.length}
          </span>
        </div>
      </motion.div>

      <div className='grid lg:grid-cols-3 gap-6'>
        {/* Server Adjustment Panel - Always visible on left */}
        <motion.div
          initial={{ opacity: 0, x: -20 }}
          animate={{ opacity: 1, x: 0 }}
          className='lg:col-span-1'
        >
          <div className='bg-white rounded-xl shadow-lg p-6 sticky top-6'>
            <h3 className='font-semibold text-gray-800 mb-4 flex items-center'>
              <Target className='w-5 h-5 text-primary mr-2' />
              ARIA's Servers
            </h3>

            <p className='text-sm text-gray-600 mb-4'>
              Adjust which servers ARIA can use for this challenge.
            </p>

            <div className='space-y-3 max-h-96 overflow-y-auto'>
              {mcpServers.map((server) => {
                const Icon = server.icon;
                const isEnabled = enabledServers.includes(server.id);

                return (
                  <div
                    key={server.id}
                    className={`flex items-center justify-between p-3 rounded-lg transition-colors ${
                      isEnabled
                        ? "bg-green-50"
                        : "bg-gray-50"
                    }`}
                  >
                    <div className='flex items-center flex-1'>
                      <Icon className="w-4 h-4 mr-2 text-gray-400" />
                      <div className='flex-1'>
                        <div className="text-sm text-gray-600">
                          {server.name}
                        </div>
                        <button
                          onClick={() => {
                            setSelectedServerForModal(server);
                            setShowFunctionModal(true);
                          }}
                          className="text-xs text-gray-500 hover:text-gray-700 transition-colors cursor-pointer flex items-center"
                          title="Click to view server details"
                        >
                          {getServerFunctionCount(server.id)} functions
                          <ArrowRight className="w-3 h-3 ml-1" />
                        </button>
                      </div>
                    </div>

                    <button
                      onClick={(e) => {
                        e.stopPropagation();
                        handleServerToggle(server.id);
                      }}
                      className={`w-10 h-5 rounded-full transition-colors relative ml-3 ${
                        isEnabled ? "bg-primary" : "bg-gray-300"
                      }`}
                    >
                      <motion.div
                        animate={{ x: isEnabled ? 20 : 0 }}
                        transition={{
                          type: "spring",
                          stiffness: 500,
                          damping: 30,
                        }}
                        className='absolute top-0.5 left-0.5 w-4 h-4 bg-white rounded-full shadow-md'
                      />
                    </button>
                  </div>
                );
              })}
            </div>

            <div className='mt-4 text-center'>
              <p className='text-xs text-gray-500'>
                Selected: {enabledServers.length} servers
              </p>
            </div>
          </div>
        </motion.div>

        {/* Function Details Modal */}
        {showFunctionModal && selectedServerForModal && (
          <div className='fixed inset-0 bg-black bg-opacity-50 flex items-center justify-center z-50 p-4'>
            <motion.div
              initial={{ opacity: 0, scale: 0.9 }}
              animate={{ opacity: 1, scale: 1 }}
              exit={{ opacity: 0, scale: 0.9 }}
              className='bg-white rounded-2xl shadow-xl max-w-2xl w-full max-h-[80vh] overflow-hidden'
            >
              <div className='p-6 border-b border-gray-100'>
                <div className='flex items-center justify-between'>
                  <div className='flex items-center'>
                    <selectedServerForModal.icon className='w-8 h-8 text-primary mr-3' />
                    <div>
                      <h3 className='text-xl font-semibold text-gray-800'>
                        {selectedServerForModal.name} Functions
                      </h3>
                      <p className='text-sm text-gray-600'>
                        {getServerFunctions(selectedServerForModal.id).length}{" "}
                        available functions
                      </p>
                    </div>
                  </div>
                  <button
                    onClick={() => {
                      setShowFunctionModal(false);
                      setSelectedServerForModal(null);
                    }}
                    className='p-2 hover:bg-gray-100 rounded-full transition-colors'
                  >
                    <XCircle className='w-6 h-6 text-gray-500' />
                  </button>
                </div>
              </div>

              <div className='p-6 overflow-y-auto max-h-[60vh]'>
                <div className='space-y-4'>
                  {getServerFunctions(selectedServerForModal.id).map(
                    (func, index) => (
                      <div
                        key={index}
                        className='bg-gray-50 rounded-lg p-4 border border-gray-100'
                      >
                        <div className='flex items-start justify-between mb-2'>
                          <code className='text-base font-mono text-blue-600 font-medium'>
                            {func.name}
                          </code>
                          <span className='text-xs bg-primary/10 text-primary px-2 py-1 rounded-full'>
                            Function
                          </span>
                        </div>
                        <p className='text-sm text-gray-700 leading-relaxed'>
                          {func.description}
                        </p>
                      </div>
                    )
                  )}
                </div>
              </div>

              <div className='p-6 border-t border-gray-100 bg-gray-50'>
                <div className='flex justify-between items-center'>
                  <div className='flex items-center text-sm text-gray-600'>
                    <Lightbulb className='w-4 h-4 mr-2' />
                    <span>
                      These functions help ARIA solve specific types of problems
                    </span>
                  </div>
                </div>
              </div>
            </motion.div>
          </div>
        )}

        {/* Challenge Description */}
        <motion.div
          initial={{ opacity: 0, x: 20 }}
          animate={{ opacity: 1, x: 0 }}
          className='lg:col-span-2'
        >
          <div className='bg-white rounded-xl shadow-lg p-6 mb-6 sticky top-6'>
            <h3 className='text-xl font-semibold text-gray-800 mb-4'>
              {challenge.title}
            </h3>

            <div className='bg-yellow-50 border-l-4 border-yellow-400 p-4 rounded-lg mb-6'>
              <p className='text-gray-700 leading-relaxed'>
                <strong>Scenario:</strong> {challenge.scenario}
              </p>
            </div>

            <div className='bg-gray-50 rounded-lg p-4 mb-6'>
              <p className='text-sm font-medium text-gray-700 mb-2'>
                Sam asks ARIA:
              </p>
              <p className='text-gray-600 italic'>"{challenge.question}"</p>
            </div>

            {!showResult && (
              <div className='text-center'>
                <button
                  onClick={solveChallenge}
                  disabled={enabledServers.length === 0}
                  className='px-6 py-3 bg-gradient-to-r from-primary to-secondary text-white rounded-lg hover:shadow-lg transition-shadow disabled:opacity-50 disabled:cursor-not-allowed'
                >
                  Let ARIA Try to Solve This!
                </button>
              </div>
            )}

            {/* Reload Animation */}
            {isReloading && (
              <motion.div
                initial={{ opacity: 0, y: 10 }}
                animate={{ opacity: 1, y: 0 }}
                className='space-y-4'
              >
                <div className='bg-blue-50 border border-blue-200 rounded-lg p-6 text-center'>
                  <div className='flex items-center justify-center mb-4'>
                    <div className='animate-spin rounded-full h-8 w-8 border-b-2 border-primary'></div>
                  </div>
                  <p className='text-blue-700 font-medium'>
                    ARIA is analyzing with your current servers...
                  </p>
                  <p className='text-blue-600 text-sm mt-2'>
                    Generating a fresh response
                  </p>
                </div>
              </motion.div>
            )}

            {/* Result Display */}
            {showResult && currentResult && !isReloading && (
              <motion.div
                key={`result-${currentResult.challengeId}-${Date.now()}`}
                initial={{ opacity: 0, y: 10 }}
                animate={{ opacity: 1, y: 0 }}
                className='space-y-4'
              >
                <div
                  className={`p-4 rounded-lg ${
                    currentResult.success
                      ? "bg-green-50 border border-green-200"
                      : "bg-red-50 border border-red-200"
                  }`}
                >
                  <div className='flex items-center mb-3'>
                    {currentResult.success ? (
                      <CheckCircle className='w-5 h-5 text-green-600 mr-2' />
                    ) : (
                      <XCircle className='w-5 h-5 text-red-600 mr-2' />
                    )}
                    <h4 className='font-semibold'>
                      {currentResult.success
                        ? "🎉 Challenge Solved!"
                        : "❌ Challenge Failed"}
                    </h4>
                  </div>

                  <div className='bg-white rounded-lg p-4'>
                    <p className='text-sm font-medium text-gray-700 mb-2'>
                      ARIA responds:
                    </p>
                    <MarkdownText
                      text={
                        currentResult.success
                          ? challenge.optimalAnswer
                          : challenge.limitedAnswer
                      }
                      className='text-sm text-gray-600'
                    />
                  </div>
                </div>

                {/* Educational Content about MCP */}
                {currentResult.success && (
                  <div className='bg-purple-50 border border-purple-200 rounded-lg p-4'>
                    <div className='flex items-start'>
                      <Lightbulb className='w-5 h-5 text-purple-600 mr-2 mt-0.5' />
                      <div>
                        <h5 className='font-semibold text-purple-800 mb-2'>
                          🧠 How ARIA solved this challenge:
                        </h5>
                        <p className='text-sm text-purple-700 mb-2'>
                          From the enabled servers, ARIA used "function calling"
                          to select specific functions:
                        </p>
                        <ul className='text-xs text-purple-600 space-y-1'>
                          {currentResult.requiredServers.map((serverId) => {
                            const server = mcpServers.find(
                              (s) => s.id === serverId
                            );
                            const functions = getFunctionsForServer(serverId);
                            return (
                              <li key={serverId}>
                                • <strong>{server.name}:</strong> {functions}
                              </li>
                            );
                          })}
                        </ul>
                      </div>
                    </div>
                  </div>
                )}

                {currentResult.success &&
                  currentResult.extraServers.length > 0 && (
                    <div className='bg-blue-50 border border-blue-200 rounded-lg p-4'>
                      <div className='flex items-start'>
                        <Lightbulb className='w-5 h-5 text-blue-600 mr-2 mt-0.5' />
                        <div>
                          <h5 className='font-semibold text-blue-800 mb-2'>
                            Good thinking! Extra servers selected:
                          </h5>
                          <p className='text-sm text-blue-700 mb-2'>
                            You enabled {currentResult.extraServers.length}{" "}
                            additional server
                            {currentResult.extraServers.length > 1 ? "s" : ""}.
                            ARIA will automatically choose the right tools from
                            what's available, but having too many options can
                            sometimes slow down her decision-making process.
                          </p>
                          <p className='text-xs text-blue-600'>
                            <strong>How MCP really works:</strong> Each server
                            offers multiple "functions" (like checkCalendar,
                            getWeather, translateText). ARIA uses "function
                            calling" to pick the exact methods she needs from
                            enabled servers. It's like having a toolbox - she
                            chooses the right tool for each part of the job!
                          </p>
                        </div>
                      </div>
                    </div>
                  )}

                {!currentResult.success && (
                  <div className='bg-orange-50 border border-orange-200 rounded-lg p-4'>
                    <div className='flex items-start'>
                      <AlertTriangle className='w-5 h-5 text-orange-600 mr-2 mt-0.5' />
                      <div>
                        <h5 className='font-semibold text-orange-800 mb-2'>
                          Missing Required Servers:
                        </h5>
                        <div className='flex flex-wrap gap-2'>
                          {currentResult.requiredServers
                            .filter(
                              (serverId) =>
                                !currentResult.selectedServers.includes(
                                  serverId
                                )
                            )
                            .map((serverId) => {
                              const server = mcpServers.find(
                                (s) => s.id === serverId
                              );
                              return (
                                <span
                                  key={serverId}
                                  className='text-xs px-2 py-1 bg-orange-100 text-orange-700 rounded-full'
                                >
                                  {server.name}
                                </span>
                              );
                            })}
                        </div>
                      </div>
                    </div>
                  </div>
                )}

                <div className='flex justify-center space-x-3'>
                  {!currentResult.success && (
                    <button
                      onClick={resetChallenge}
                      className='px-4 py-2 text-gray-600 bg-gray-100 rounded-lg hover:bg-gray-200 transition-colors'
                    >
                      Try Again
                    </button>
                  )}
                  {currentChallenge < challenges.length - 1 && (
                    <button
                      onClick={nextChallenge}
                      className='px-6 py-3 bg-gradient-to-r from-primary to-secondary text-white rounded-lg hover:shadow-lg transition-shadow'
                    >
                      Continue Sam's Story
                    </button>
                  )}
                </div>
              </motion.div>
            )}
          </div>
        </motion.div>
      </div>

      {/* Navigation */}
      <div className='flex justify-between items-center mt-8'>
        <button
          onClick={() => {
            setShowIntro(true);
          }}
          className='inline-flex items-center px-6 py-3 text-gray-600 bg-gray-100 rounded-full hover:bg-gray-200 transition-colors'
        >
          <ArrowLeft className='mr-2 w-4 h-4' />
          Back to Intro
        </button>

        {/* Show completion button for Challenge 3 regardless of success, or for other challenges only on success */}
        {currentChallenge === challenges.length - 1 &&
          showResult &&
          (currentResult?.success || challenge.id === 3) && (
            <button
              onClick={nextStep}
              className='inline-flex items-center px-6 py-3 bg-gradient-to-r from-primary to-secondary text-white font-medium rounded-full hover:shadow-lg transition-shadow'
            >
              Complete Sam's Story
              <ArrowRight className='ml-2 w-4 h-4' />
            </button>
          )}
      </div>
    </div>
  );
};

export default Challenge;
