import React, { useState } from "react";
import { motion } from "framer-motion";
import {
  Calendar,
  Users,
  Clock,
  AlertCircle,
  CheckCircle,
  ArrowRight,
  ArrowLeft,
} from "lucide-react";
import useProgress from "../hooks/useProgress";

const Discovery = () => {
  const { nextStep, previousStep } = useProgress();
  const [showComparison, setShowComparison] = useState(false);
  const [currentScene, setCurrentScene] = useState(0);

  const scenes = [
    {
      title: "Sam's Challenge: Science Fair Research",
      icon: Users,
      description:
        "Sam needs help finding current renewable energy articles for tomorrow's science fair presentation.",
      color: "blue",
    },
    {
      title: "Basic ARIA: Limited Help",
      icon: AlertCircle,
      description: "Watch what happens when Sam asks basic ARIA for help...",
      color: "red",
      conversation: [
        {
          role: "user",
          text: "ARIA, I need current articles about renewable energy for my science fair presentation tomorrow!",
        },
        {
          role: "ai",
          text: "I can help explain renewable energy concepts from my training data, but I don't have access to:",
        },
        { role: "ai", text: "• Current news articles or reports" },
        { role: "ai", text: "• Real-time research databases" },
        { role: "ai", text: "• Ways to verify if sources are credible" },
        {
          role: "ai",
          text: "You'll need to search online yourself and hope you find good sources.",
        },
      ],
    },
    {
      title: "MCP-Powered ARIA: Real Help",
      icon: CheckCircle,
      description:
        "Now see how MCP transforms ARIA into Sam's perfect research assistant!",
      color: "green",
      conversation: [
        {
          role: "user",
          text: "ARIA, I need current articles about renewable energy for my science fair presentation tomorrow!",
        },
        {
          role: "ai",
          text: "Perfect! Let me help you with current research... 🔍",
        },
        { role: "ai", text: "Found 5 recent articles from credible sources:" },
        {
          role: "ai",
          text: "• 'Teen Solar Innovation Wins Science Prize' - MIT News (yesterday!)",
        },
        {
          role: "ai",
          text: "• 'High School Green Energy Success' - Science Daily (this week)",
        },
        {
          role: "ai",
          text: "✓ All sources verified and perfect for citations",
        },
        { role: "ai", text: "Your presentation will be amazing, Sam! 🌟" },
      ],
    },
  ];

  const handleNext = () => {
    if (currentScene < scenes.length - 1) {
      setCurrentScene(currentScene + 1);
      if (currentScene === 0) {
        setShowComparison(true);
      }
    } else {
      nextStep();
    }
  };

  const currentSceneData = scenes[currentScene];

  return (
    <div className='max-w-4xl mx-auto'>
      <motion.div
        initial={{ opacity: 0 }}
        animate={{ opacity: 1 }}
        transition={{ duration: 0.3 }}
      >
        <motion.div
          key={currentScene}
          initial={{ opacity: 0 }}
          animate={{ opacity: 1 }}
          transition={{ duration: 0.3 }}
          className='text-center mb-8'
        >
          <div
            className={`inline-flex items-center justify-center w-20 h-20 bg-${currentSceneData.color}-100 rounded-full mb-4`}
          >
            <currentSceneData.icon
              className={`w-10 h-10 text-${currentSceneData.color}-600`}
            />
          </div>

          <h2 className='text-3xl font-bold text-gray-800 mb-4'>
            {currentSceneData.title}
          </h2>

          <p className='text-lg text-gray-600'>
            {currentSceneData.description}
          </p>
        </motion.div>

        {currentSceneData.conversation && (
          <motion.div
            key={`conversation-${currentScene}`}
            initial={{ opacity: 0 }}
            animate={{ opacity: 1 }}
            transition={{ duration: 0.3 }}
            className='bg-white rounded-2xl shadow-xl p-6 mb-8'
          >
            <div className='space-y-4'>
              {currentSceneData.conversation.map((message, index) => (
                <motion.div
                  key={index}
                  initial={{ opacity: 0 }}
                  animate={{ opacity: 1 }}
                  transition={{ delay: index * 0.1, duration: 0.3 }}
                  className={`flex ${
                    message.role === "user" ? "justify-end" : "justify-start"
                  }`}
                >
                  <div
                    className={`max-w-xs px-4 py-3 rounded-2xl ${
                      message.role === "user"
                        ? "bg-primary text-white"
                        : currentScene === 1
                        ? "bg-gray-100 text-gray-700"
                        : "bg-green-100 text-gray-700"
                    }`}
                  >
                    {message.text}
                  </div>
                </motion.div>
              ))}
            </div>
          </motion.div>
        )}

        {showComparison && currentScene === 0 && (
          <motion.div
            initial={{ opacity: 0 }}
            animate={{ opacity: 1 }}
            transition={{ duration: 0.3 }}
            className='grid md:grid-cols-3 gap-4 mb-8'
          >
            <div className='bg-gray-50 rounded-lg p-4 text-center'>
              <Clock className='w-8 h-8 text-gray-400 mx-auto mb-2' />
              <p className='text-sm text-gray-600'>
                Hours of manual research needed
              </p>
            </div>
            <div className='bg-gray-50 rounded-lg p-4 text-center'>
              <Users className='w-8 h-8 text-gray-400 mx-auto mb-2' />
              <p className='text-sm text-gray-600'>
                Can't verify source credibility
              </p>
            </div>
            <div className='bg-gray-50 rounded-lg p-4 text-center'>
              <Calendar className='w-8 h-8 text-gray-400 mx-auto mb-2' />
              <p className='text-sm text-gray-600'>
                No access to current articles
              </p>
            </div>
          </motion.div>
        )}

        <div className='flex justify-between items-center mt-8'>
          <button
            onClick={previousStep}
            className='inline-flex items-center px-6 py-3 text-gray-600 bg-gray-100 rounded-full hover:bg-gray-200 transition-colors'
          >
            <ArrowLeft className='mr-2 w-4 h-4' />
            Back
          </button>

          <div className='flex space-x-2'>
            {scenes.map((_, index) => (
              <div
                key={index}
                className={`w-2 h-2 rounded-full transition-colors ${
                  index === currentScene ? "bg-primary" : "bg-gray-300"
                }`}
              />
            ))}
          </div>

          <button
            onClick={handleNext}
            className='inline-flex items-center px-6 py-3 bg-gradient-to-r from-primary to-secondary text-white font-medium rounded-full hover:shadow-lg transition-shadow'
          >
            {currentScene < scenes.length - 1 ? "Next" : "Learn How MCP Works"}
            <ArrowRight className='ml-2 w-4 h-4' />
          </button>
        </div>
      </motion.div>
    </div>
  );
};

export default Discovery;
