import React, { useState } from 'react';
import { motion } from 'framer-motion';
import { ArrowRight, ArrowLeft, Power, Info } from 'lucide-react';
import useProgress from '../hooks/useProgress';
import { mcpServers } from '../data/mcpServers';
import ServerCard from '../components/ServerCard';
import ComparisonModal from '../components/ComparisonModal';
import { getServerFunctionCount } from '../utils/serverFunctions';

const Exploration = () => {
  const { nextStep, previousStep, enabledServers, toggleServer } = useProgress();
  const [selectedServer, setSelectedServer] = useState(null);
  const [showModal, setShowModal] = useState(false);

  const handleServerClick = (server) => {
    setSelectedServer(server);
    setShowModal(true);
  };

  const handleToggle = (serverId) => {
    toggleServer(serverId);
  };

  return (
    <div className="max-w-6xl mx-auto py-12">
      <motion.div
        initial={{ opacity: 0, y: 20 }}
        animate={{ opacity: 1, y: 0 }}
        className="text-center mb-10"
      >
        <h2 className="text-3xl font-bold text-gray-800 mb-4">
          Choose MCP Servers for Sam
        </h2>
        <p className="text-lg text-gray-600 mb-6">
          Click on any server to see how it helps Sam. Toggle them on/off to build the perfect ARIA assistant!
        </p>
        
        <div className="inline-flex items-center bg-blue-50 px-4 py-2 rounded-full text-sm">
          <Power className="w-4 h-4 text-primary mr-2" />
          <span className="text-gray-700">
            {enabledServers.length} of {mcpServers.length} servers enabled • 
            <strong>{enabledServers.reduce((total, serverId) => {
              return total + getServerFunctionCount(serverId);
            }, 0)}</strong> functions available
          </span>
        </div>
      </motion.div>

      <div className="grid md:grid-cols-2 lg:grid-cols-3 gap-6 mb-10">
        {mcpServers.map((server, index) => (
          <motion.div
            key={server.id}
            initial={{ opacity: 0, scale: 0.9 }}
            animate={{ opacity: 1, scale: 1 }}
            transition={{ delay: index * 0.1 }}
          >
            <ServerCard
              server={server}
              isEnabled={enabledServers.includes(server.id)}
              onToggle={() => handleToggle(server.id)}
              onClick={() => handleServerClick(server)}
            />
          </motion.div>
        ))}
      </div>

      <motion.div
        initial={{ opacity: 0 }}
        animate={{ opacity: 1 }}
        transition={{ delay: 0.6 }}
        className="bg-gradient-to-r from-purple-50 to-blue-50 rounded-2xl p-6 mb-8"
      >
        <div className="flex items-start">
          <Info className="w-6 h-6 text-primary mr-3 flex-shrink-0 mt-1" />
          <div>
            <h3 className="font-semibold text-gray-800 mb-2">Quick Tip!</h3>
            <p className="text-gray-600">
              Each server gives your AI a specific superpower. The more servers you enable, 
              the more helpful your AI becomes. Try different combinations to see how they work together!
            </p>
          </div>
        </div>
      </motion.div>

      <div className="flex justify-between items-center">
        <button
          onClick={previousStep}
          className="inline-flex items-center px-6 py-3 text-gray-600 bg-gray-100 rounded-full hover:bg-gray-200 transition-colors"
        >
          <ArrowLeft className="mr-2 w-4 h-4" />
          Back
        </button>
        
        <button
          onClick={nextStep}
          className="inline-flex items-center px-6 py-3 bg-gradient-to-r from-primary to-secondary text-white font-medium rounded-full hover:shadow-lg transition-shadow"
        >
          Test ARIA for Sam
          <ArrowRight className="ml-2 w-4 h-4" />
        </button>
      </div>

      {showModal && selectedServer && (
        <ComparisonModal
          server={selectedServer}
          isOpen={showModal}
          onClose={() => setShowModal(false)}
          isEnabled={enabledServers.includes(selectedServer.id)}
          onToggle={() => handleToggle(selectedServer.id)}
        />
      )}
    </div>
  );
};

export default Exploration;