import React, { useState, useEffect, useCallback } from 'react';
import { motion } from 'framer-motion';
import { Send, ArrowRight, ArrowLeft, Zap, Shuffle } from 'lucide-react';
import useProgress from '../hooks/useProgress';
import { mcpServers } from '../data/mcpServers';
import ComparisonModal from '../components/ComparisonModal';
import EnhancedAIResponse from '../components/EnhancedAIResponse';
import { getFunctionsForServer } from '../utils/serverFunctions';

// Helper function to get which functions would be called for a specific question
const getFunctionsCalledForQuestion = (question, serverId) => {
  const functionCallMap = {
    'calendar': {
      'When can my study group meet this week before the chemistry test?': ['checkAvailability()', 'bookRoom()'],
      'When are you and your lab partner free to work on the science project?': ['checkAvailability()'],
      'When can my study group meet this week?': ['checkAvailability()'],
      'Book the library study room for tomorrow at 4pm': ['bookRoom()', 'sendInvites()'],
      'What time is my chemistry test next week?': ['checkAvailability()'],
      'Schedule a group meeting for our history project': ['checkAvailability()', 'bookRoom()', 'sendInvites()']
    },
    'weather': {
      'Should we move Friday\'s pep rally indoors or keep it at the football field?': ['getForecast()', 'checkConditions()'],
      'Is it good weather for our class photo shoot tomorrow?': ['getCurrentWeather()', 'getForecast()'],
      'Should we move the pep rally indoors or keep it outside?': ['getForecast()', 'checkConditions()'],
      'Will it rain during lunch break today?': ['getCurrentWeather()', 'getForecast()'],
      'What\'s the temperature for our outdoor biology lab?': ['getCurrentWeather()']
    },
    'maps': {
      'What\'s the quickest way to get from chemistry lab to the cafeteria before lunch ends?': ['findRoute()'],
      'How do I get from school to the new study café downtown?': ['findRoute()'],
      'What\'s the quickest way from chemistry lab to the cafeteria?': ['findRoute()'],
      'Show me the safest route to walk home after dark': ['findRoute()'],
      'Where\'s the nearest coffee shop from the library?': ['locatePlace()', 'findRoute()']
    },
    'music': {
      'I need music to help me focus while writing my English essay': ['createPlaylist()', 'analyzeGenre()'],
      'Make me a playlist for cramming before tomorrow\'s biology test': ['createPlaylist()', 'analyzeGenre()'],
      'I need focus music for writing my English essay': ['createPlaylist()', 'analyzeGenre()'],
      'Create upbeat music for our group study session': ['createPlaylist()', 'setVolume()'],
      'What\'s good background music for math homework?': ['createPlaylist()', 'analyzeGenre()']
    },
    'news': {
      'I need current info about climate change for my environmental science project': ['searchArticles()', 'verifySource()'],
      'What\'s the latest on AI in education for my tech presentation?': ['searchArticles()', 'verifySource()'],
      'Find current articles about climate change for my science project': ['searchArticles()', 'verifySource()'],
      'I need recent news about renewable energy for my essay': ['searchArticles()', 'verifySource()'],
      'What are the latest developments in space exploration?': ['searchArticles()', 'verifySource()']
    },
    'translation': {
      'Check my Spanish presentation script and help with pronunciation': ['translateText()', 'checkGrammar()', 'generateAudio()'],
      'How do I ask my exchange student friend about weekend plans in German?': ['translateText()', 'generateAudio()'],
      'Help me practice my Spanish presentation script': ['translateText()', 'checkGrammar()', 'generateAudio()'],
      'Translate this French homework into English': ['translateText()'],
      'Check my pronunciation for these Italian words': ['generateAudio()']
    },
    'calculator': {
      'Help me solve this quadratic equation and show all the work for my algebra test': ['solveEquation()', 'showSteps()'],
      'How much should each person pay if 4 friends split a $52.80 dinner bill with 18% tip?': ['solveEquation()', 'showSteps()'],
      'Solve this quadratic equation: x² + 5x - 6 = 0': ['solveEquation()', 'showSteps()', 'plotGraph()'],
      'What\'s 15% of 240?': ['solveEquation()'],
      'Help me calculate the area of a circle with radius 7cm': ['solveEquation()', 'showSteps()']
    },
    'wikipedia': {
      'Is it true that the Great Wall of China is visible from space? I need this for my history report.': ['searchFacts()', 'verifyClaims()', 'findCitations()'],
      'Do we really only use 10% of our brains?': ['searchFacts()', 'verifyClaims()'],
      'Is the Great Wall of China visible from space?': ['searchFacts()', 'verifyClaims()', 'findCitations()'],
      'How many chambers does a human heart have?': ['searchFacts()'],
      'What causes the northern lights?': ['searchFacts()', 'verifyClaims()']
    },
    'timer': {
      'Help me organize my homework for this week and set up study reminders': ['organizeDeadlines()', 'setReminder()'],
      'Start a study session for my upcoming biology exam': ['trackTime()', 'setReminder()'],
      'Remind me to submit my history essay tomorrow at 3pm': ['setReminder()'],
      'How long until my chemistry test next Friday?': ['organizeDeadlines()'],
      'Set a timer for 25 minutes of focused math practice': ['trackTime()', 'setReminder()']
    },
    'social': {
      'Should I post my college application portfolio online and when\'s the best time?': ['analyzeEngagement()', 'predictTiming()'],
      'When should I post about making honor roll to get the most positive response?': ['predictTiming()', 'analyzeEngagement()'],
      'Should I share my college application portfolio online?': ['analyzeEngagement()', 'predictTiming()'],
      'What\'s the best time to post my art project on Instagram?': ['predictTiming()', 'checkTrends()'],
      'Help me write a professional LinkedIn post about my internship': ['analyzeEngagement()']
    }
  };
  
  return functionCallMap[serverId]?.[question] || [];
};

const Playground = () => {
  const { nextStep, previousStep, enabledServers, toggleServer } = useProgress();
  const [query, setQuery] = useState('');
  const [currentAnswer, setCurrentAnswer] = useState('');
  const [showAnswer, setShowAnswer] = useState(false);
  const [selectedServer, setSelectedServer] = useState(null);
  const [showModal, setShowModal] = useState(false);
  const [displayedQuestions, setDisplayedQuestions] = useState([]);
  
  // Get all questions (both server-specific and generic)
  const getAllQuestions = () => {
    const serverQuestions = [];
    
    mcpServers.forEach(server => {
      // Add main playground question
      serverQuestions.push({
        question: server.playground.question,
        answer: server.playground.answer,
        serverId: server.id,
        serverName: server.name,
        serverIcon: server.icon,
        type: "server-specific"
      });
      
      // Add all sample questions if they exist
      if (server.playground.sampleQuestions) {
        server.playground.sampleQuestions.slice(1).forEach(sampleQ => { // Skip first one as it's the main question
          serverQuestions.push({
            question: sampleQ,
            answer: server.playground.answer, // Use same answer for now, could be customized later
            serverId: server.id,
            serverName: server.name,
            serverIcon: server.icon,
            type: "server-specific"
          });
        });
      }
    });
    
    const genericQuestions = [
      {
        question: "Tell me about artificial intelligence",
        answer: "Artificial Intelligence (AI) is a branch of computer science focused on creating systems that can perform tasks typically requiring human intelligence. This includes learning, reasoning, problem-solving, and understanding language.\n\nKey types of AI:\n• Machine Learning - learns from data\n• Natural Language Processing - understands text/speech\n• Computer Vision - processes images\n• Robotics - physical world interaction\n\nAI is used in everything from smartphones to self-driving cars!",
        serverId: null,
        type: "generic"
      },
      {
        question: "What are the main programming languages?",
        answer: "Popular programming languages include:\n\n**Beginner-friendly:**\n• Python - great for AI, data science\n• JavaScript - web development\n• Scratch - visual programming for kids\n\n**Widely used:**\n• Java - enterprise applications\n• C++ - gaming, systems programming\n• Swift - iOS app development\n\nEach language has its strengths for different types of projects!",
        serverId: null,
        type: "generic"
      },
      {
        question: "How do computers work?",
        answer: "Computers work by processing information through a series of basic operations:\n\n**Core Components:**\n• CPU (Central Processing Unit) - the 'brain' that executes instructions\n• Memory (RAM) - temporary storage for active programs\n• Storage - permanent data storage (hard drives, SSDs)\n• Input/Output devices - keyboard, mouse, screen\n\n**Basic Process:**\n1. Instructions are stored in memory\n2. CPU fetches and executes instructions one by one\n3. Results are stored back in memory or sent to output devices\n\nEverything is represented as binary code (1s and 0s)!",
        serverId: null,
        type: "generic"
      },
      {
        question: "What is the internet?",
        answer: "The internet is a global network of interconnected computers that can communicate with each other.\n\n**How it works:**\n• Millions of computers connected through cables, wireless signals, and satellites\n• Data is broken into packets and sent through the network\n• Protocols (like HTTP, HTTPS) govern how data is transmitted\n• Domain names (like google.com) map to IP addresses\n\n**Key services:**\n• World Wide Web (websites)\n• Email\n• File sharing\n• Video streaming\n• Social media\n\nIt's like a massive digital highway system connecting the world!",
        serverId: null,
        type: "generic"
      },
      {
        question: "Explain quantum computing",
        answer: "Quantum computing uses principles of quantum mechanics to process information differently than traditional computers.\n\n**Key Concepts:**\n• **Qubits** - quantum bits that can be 0, 1, or both simultaneously\n• **Superposition** - ability to exist in multiple states at once\n• **Entanglement** - qubits can be mysteriously connected\n• **Quantum interference** - amplifies correct answers, cancels wrong ones\n\n**Potential applications:**\n• Cryptography and security\n• Drug discovery\n• Weather prediction\n• Financial modeling\n\n**Current status:** Still experimental, but companies like IBM and Google are making progress!",
        serverId: null,
        type: "generic"
      },
      {
        question: "What is machine learning?",
        answer: "Machine learning is a subset of AI where computers learn patterns from data without being explicitly programmed for every scenario.\n\n**Types of machine learning:**\n• **Supervised learning** - learns from labeled examples\n• **Unsupervised learning** - finds hidden patterns in data\n• **Reinforcement learning** - learns through trial and error\n\n**Common applications:**\n• Image recognition (photo tagging)\n• Recommendation systems (Netflix, Spotify)\n• Language translation\n• Fraud detection\n• Medical diagnosis\n\nIt's like teaching a computer to recognize patterns the way humans do!",
        serverId: null,
        type: "generic"
      },
      {
        question: "How do smartphones work?",
        answer: "Smartphones are essentially powerful miniature computers with wireless communication capabilities.\n\n**Key components:**\n• **Processor** - runs apps and operating system\n• **Memory** - stores apps and data\n• **Battery** - powers all components\n• **Antennas** - for cellular, WiFi, Bluetooth, GPS\n• **Sensors** - accelerometer, gyroscope, camera, microphone\n\n**How they connect:**\n• Cell towers provide voice/data service\n• WiFi connects to local internet\n• GPS satellites provide location\n• Bluetooth connects to nearby devices\n\nApps are programs that use these features to provide functionality!",
        serverId: null,
        type: "generic"
      },
      {
        question: "What is cybersecurity?",
        answer: "Cybersecurity is the practice of protecting computer systems, networks, and data from digital attacks and unauthorized access.\n\n**Common threats:**\n• **Malware** - viruses, ransomware, spyware\n• **Phishing** - fake emails/websites to steal information\n• **Hacking** - unauthorized system access\n• **Data breaches** - stealing personal information\n\n**Protection methods:**\n• Strong passwords and two-factor authentication\n• Regular software updates\n• Antivirus software\n• Secure networks (VPNs)\n• User education and awareness\n\nIt's like having digital locks, alarms, and security guards for your data!",
        serverId: null,
        type: "generic"
      },
      {
        question: "What is blockchain technology?",
        answer: "Blockchain is a distributed ledger technology that maintains a secure, transparent record of transactions across multiple computers.\n\n**How it works:**\n• Data is stored in 'blocks' linked together in a 'chain'\n• Each block contains a cryptographic hash of the previous block\n• Network of computers validates new transactions\n• Once added, data cannot be easily altered\n\n**Key features:**\n• Decentralized (no single authority)\n• Transparent (all transactions visible)\n• Secure (cryptographically protected)\n• Immutable (difficult to change)\n\n**Applications:**\n• Cryptocurrencies (Bitcoin, Ethereum)\n• Supply chain tracking\n• Digital identity\n• Smart contracts",
        serverId: null,
        type: "generic"
      },
      {
        question: "How does cloud computing work?",
        answer: "Cloud computing delivers computing services (storage, processing power, software) over the internet instead of using local hardware.\n\n**Types of cloud services:**\n• **SaaS** - Software as a Service (Gmail, Netflix)\n• **PaaS** - Platform as a Service (app development tools)\n• **IaaS** - Infrastructure as a Service (virtual servers)\n\n**Benefits:**\n• Access from anywhere with internet\n• Automatic updates and maintenance\n• Scalable resources (pay for what you use)\n• No need for expensive hardware\n• Automatic backups and security\n\n**Examples:**\n• Google Drive for file storage\n• Netflix for video streaming\n• Office 365 for productivity apps\n\nIt's like renting computing power instead of buying it!",
        serverId: null,
        type: "generic"
      }
    ];
    
    return [...serverQuestions, ...genericQuestions];
  };

  const allQuestions = getAllQuestions();

  // Function to shuffle and select a subset of questions
  const shuffleQuestions = useCallback(() => {
    const genericQuestions = allQuestions.filter(q => q.type === "generic");
    const serverQuestions = allQuestions.filter(q => q.type === "server-specific");
    
    // Randomly select 3 generic questions from the 10 available
    const shuffledGenericQuestions = [...genericQuestions].sort(() => Math.random() - 0.5);
    const selectedGenericQuestions = shuffledGenericQuestions.slice(0, 3);
    
    // Randomly select 6-8 server questions
    const shuffledServerQuestions = [...serverQuestions].sort(() => Math.random() - 0.5);
    const selectedServerQuestions = shuffledServerQuestions.slice(0, Math.floor(Math.random() * 3) + 6); // 6-8 questions
    
    // Put server questions first, then generic questions at the end (no final shuffle)
    const shuffledQuestions = [...selectedServerQuestions, ...selectedGenericQuestions];
    
    setDisplayedQuestions(shuffledQuestions);
  }, [allQuestions]);

  // Initialize displayed questions on component mount
  useEffect(() => {
    shuffleQuestions();
  }, []);

  // Find matching question and get appropriate answer
  const handleSubmit = () => {
    if (!query.trim()) return;
    
    // Find exact or partial match
    const matchedQuestion = allQuestions.find(q => 
      q.question.toLowerCase() === query.toLowerCase().trim()
    );
    
    if (matchedQuestion) {
      // Get answer based on server state
      let answer;
      if (matchedQuestion.type === "generic") {
        answer = matchedQuestion.answer;
      } else {
        const isServerEnabled = enabledServers.includes(matchedQuestion.serverId);
        answer = isServerEnabled 
          ? matchedQuestion.answer 
          : "I understand you need help, but I don't have access to real-time information or external tools. I can only provide general advice based on my training data.";
      }
      setCurrentAnswer(answer);
    } else {
      // Default response for unmatched questions
      setCurrentAnswer("I can help with that! Try asking one of the suggested questions to see how different MCP servers enhance my capabilities.");
    }
    
    setShowAnswer(true);
  };

  // Update answer when servers change
  useEffect(() => {
    if (!showAnswer || !query.trim()) return;
    
    // Find the current question and update answer based on new server state
    const matchedQuestion = allQuestions.find(q => 
      q.question.toLowerCase() === query.toLowerCase().trim()
    );
    
    if (matchedQuestion && matchedQuestion.type !== "generic") {
      const isServerEnabled = enabledServers.includes(matchedQuestion.serverId);
      const answer = isServerEnabled 
        ? matchedQuestion.answer 
        : "I understand you need help, but I don't have access to real-time information or external tools. I can only provide general advice based on my training data.";
      setCurrentAnswer(answer);
    }
  }, [enabledServers, query, showAnswer, allQuestions]);
  

  return (
    <div className="max-w-5xl mx-auto py-12">
      <motion.div
        initial={{ opacity: 0, y: 20 }}
        animate={{ opacity: 1, y: 0 }}
        className="text-center mb-10"
      >
        <h2 className="text-3xl font-bold text-gray-800 mb-4">
          Build ARIA for Sam
        </h2>
        <p className="text-lg text-gray-600 mb-4">
          Toggle MCP servers on/off and test how ARIA can help Sam with different challenges!
        </p>
        
        <div className="inline-flex items-center bg-primary/10 px-4 py-2 rounded-full text-sm">
          <Zap className="w-4 h-4 text-primary mr-2" />
          <span className="text-gray-700">
            <strong>{enabledServers.length}</strong> of {mcpServers.length} servers enabled • 
            <strong>{enabledServers.reduce((total, serverId) => {
              const functions = getFunctionsForServer(serverId).split(', ').length;
              return total + functions;
            }, 0)}</strong> functions available
          </span>
        </div>
      </motion.div>

      <div className="grid lg:grid-cols-3 gap-6 mb-8">
        <motion.div
          initial={{ opacity: 0, x: -20 }}
          animate={{ opacity: 1, x: 0 }}
          className="lg:col-span-1"
        >
          <div className="bg-white rounded-xl shadow-lg p-6 pb-8 h-[568px] flex flex-col sticky top-6">
            <div className="flex items-center justify-between mb-4">
              <h3 className="font-semibold text-gray-800 flex items-center">
                <Zap className="w-5 h-5 text-primary mr-2" />
                MCP Servers
              </h3>
              <div className="flex items-center space-x-3">
                <span className="text-xs text-gray-600">All</span>
                <button
                  onClick={() => {
                    if (enabledServers.length === mcpServers.length) {
                      // Disable all
                      enabledServers.forEach(serverId => toggleServer(serverId));
                    } else {
                      // Enable all
                      mcpServers.forEach(server => {
                        if (!enabledServers.includes(server.id)) {
                          toggleServer(server.id);
                        }
                      });
                    }
                  }}
                  className={`w-10 h-5 rounded-full transition-colors relative ${
                    enabledServers.length === mcpServers.length ? 'bg-primary' : 'bg-gray-300'
                  }`}
                  title={enabledServers.length === mcpServers.length ? "Disable All Servers" : "Enable All Servers"}
                >
                  <motion.div
                    animate={{ x: enabledServers.length === mcpServers.length ? 20 : 0 }}
                    transition={{ type: "spring", stiffness: 500, damping: 30 }}
                    className="absolute top-0.5 left-0.5 w-4 h-4 bg-white rounded-full shadow-md"
                  />
                </button>
              </div>
            </div>
            
            {/* Learning Mode Info */}
            <div className="bg-blue-50 border border-blue-200 rounded-lg p-3 mb-3">
              <p className="text-xs text-blue-700">
                💡 <strong>Learning Mode:</strong> We use simple toggles here. In real apps, developers write code to connect these servers.
              </p>
            </div>
            
            <div className="space-y-3 overflow-y-auto flex-1">
              {mcpServers.map((server) => {
                const Icon = server.icon;
                const isEnabled = enabledServers.includes(server.id);
                
                return (
                  <div
                    key={server.id}
                    className={`flex items-center justify-between p-3 rounded-lg transition-colors ${
                      isEnabled ? 'bg-green-50' : 'bg-gray-50'
                    }`}
                  >
                    <div className="flex items-center flex-1">
                      <Icon className="w-4 h-4 mr-2 text-gray-400" />
                      <div className="flex-1">
                        <div className="text-sm text-gray-600">
                          {server.name}
                        </div>
                        <button
                          onClick={() => {
                            setSelectedServer(server);
                            setShowModal(true);
                          }}
                          className="text-xs text-gray-500 hover:text-gray-700 transition-colors cursor-pointer flex items-center"
                          title="Click to view server details"
                        >
                          {getFunctionsForServer(server.id).split('(), ').length} functions
                          <ArrowRight className="w-3 h-3 ml-1" />
                        </button>
                      </div>
                    </div>
                    
                    <button
                      onClick={(e) => {
                        e.stopPropagation();
                        toggleServer(server.id);
                      }}
                      className={`w-10 h-5 rounded-full transition-colors relative ${
                        isEnabled ? 'bg-primary' : 'bg-gray-300'
                      }`}
                    >
                      <motion.div
                        animate={{ x: isEnabled ? 20 : 0 }}
                        transition={{ type: "spring", stiffness: 500, damping: 30 }}
                        className="absolute top-0.5 left-0.5 w-4 h-4 bg-white rounded-full shadow-md"
                      />
                    </button>
                  </div>
                );
              })}
            </div>
          </div>
        </motion.div>

        <motion.div
          initial={{ opacity: 0, y: 20 }}
          animate={{ opacity: 1, y: 0 }}
          className="lg:col-span-2"
        >
          <div className="bg-white rounded-xl shadow-lg p-6">
            <h3 className="font-semibold text-gray-800 mb-4">Ask Your AI Assistant</h3>
            <p className="text-sm text-gray-600 mb-6">
              Click on a suggested question below. Toggle servers on/off to see how answers change!
            </p>
            
            {/* Suggested Questions */}
            <div className="mb-6">
              <div className="flex items-center justify-between mb-3">
                <p className="text-sm font-medium text-gray-700">
                  Suggested questions:
                </p>
                <button
                  onClick={shuffleQuestions}
                  className="inline-flex items-center px-3 py-1.5 text-xs bg-gray-100 hover:bg-gray-200 text-gray-600 rounded-full transition-colors"
                  title="Show new questions"
                >
                  <Shuffle className="w-3 h-3 mr-1" />
                  Show New Questions
                </button>
              </div>
              <div className="flex flex-wrap gap-2">
                {displayedQuestions.map((question, index) => {
                  const Icon = question.serverIcon;
                  const isServerEnabled = question.type === "generic" || enabledServers.includes(question.serverId);
                  const isServerDisabled = question.type !== "generic" && !enabledServers.includes(question.serverId);
                  
                  return (
                    <div key={index} className="relative group">
                      <button
                        onClick={() => {
                          setQuery(question.question);
                          setShowAnswer(false);
                        }}
                        className={`text-xs px-3 py-1.5 rounded-full transition-colors border flex items-center ${
                          query === question.question
                            ? question.type === "generic"
                              ? 'bg-gray-600 text-white border-gray-600 shadow-md'
                              : isServerEnabled
                              ? 'bg-green-600 text-white border-green-600 shadow-md'
                              : 'bg-red-600 text-white border-red-600 shadow-md'
                            : question.type === "generic"
                            ? 'bg-gray-50 text-gray-700 border-gray-200 hover:bg-gray-100'
                            : isServerEnabled
                            ? 'bg-green-50 text-green-700 border-green-200 hover:bg-green-100'
                            : 'bg-red-50 text-red-700 border-red-200 hover:bg-red-100'
                        }`}
                      >
                        {Icon && <Icon className="w-3 h-3 mr-1" />}
                        {question.question}
                      </button>
                      
                      {/* Tooltip for disabled servers */}
                      {isServerDisabled && (
                        <div className="absolute bottom-full left-1/2 transform -translate-x-1/2 mb-2 px-3 py-2 bg-gray-800 text-white text-xs rounded-lg opacity-0 group-hover:opacity-100 transition-opacity duration-200 pointer-events-none whitespace-nowrap z-10">
                          {question.serverName} is disabled
                          <div className="absolute top-full left-1/2 transform -translate-x-1/2 border-4 border-transparent border-t-gray-800"></div>
                        </div>
                      )}
                    </div>
                  );
                })}
              </div>
            </div>

            {/* Input Section */}
            <div className="mb-6">
              <div className="flex space-x-2 mb-2">
                <input
                  type="text"
                  value={query}
                  onChange={(e) => setQuery(e.target.value)}
                  onKeyPress={(e) => e.key === 'Enter' && handleSubmit()}
                  placeholder="Select a question from above to see how ARIA responds..."
                  className="flex-1 px-4 py-3 border border-gray-300 rounded-lg focus:outline-none focus:ring-2 focus:ring-primary bg-gray-50"
                  disabled={true}
                />
                <button
                  onClick={handleSubmit}
                  className="px-6 py-3 bg-gradient-to-r from-primary to-secondary text-white rounded-lg hover:shadow-lg transition-shadow disabled:opacity-50 disabled:cursor-not-allowed"
                  disabled={!query.trim()}
                >
                  <Send className="w-5 h-5" />
                </button>
              </div>
              <p className="text-xs text-gray-500">
                Choose from the suggested questions above to test ARIA's capabilities
              </p>
            </div>

            {/* Answer Section */}
            {showAnswer && (
              <motion.div
                initial={{ opacity: 0, y: 10 }}
                animate={{ opacity: 1, y: 0 }}
                className="border-t pt-6"
              >
                {(() => {
                  const matchedQuestion = allQuestions.find(q => 
                    q.question.toLowerCase() === query.toLowerCase().trim()
                  );
                  
                  const hasMCPFunctions = matchedQuestion && 
                    matchedQuestion.type !== "generic" && 
                    enabledServers.includes(matchedQuestion.serverId) &&
                    getFunctionsCalledForQuestion(matchedQuestion.question, matchedQuestion.serverId).length > 0;
                  
                  return (
                    <div className={`rounded-lg p-4 ${
                      hasMCPFunctions 
                        ? 'bg-green-50 border border-green-200' 
                        : 'bg-gray-50'
                    }`}>
                      <h4 className="font-medium text-gray-800 mb-2 flex items-center">
                        <span className="text-primary mr-2">🤖</span>
                        AI Assistant Response:
                      </h4>
                      <div className="text-sm text-gray-700">
                        <EnhancedAIResponse 
                          text={currentAnswer}
                          functionsUsed={hasMCPFunctions ? getFunctionsCalledForQuestion(matchedQuestion.question, matchedQuestion.serverId) : []}
                        />
                      </div>
                      
                      {/* Show which functions were called */}
                      {hasMCPFunctions && (() => {
                        const calledFunctions = getFunctionsCalledForQuestion(matchedQuestion.question, matchedQuestion.serverId);
                        return (
                          <div className="mt-3 p-2 bg-green-100 rounded border-l-4 border-green-400">
                            <p className="text-xs text-green-700 font-medium mb-1">🔧 Functions called:</p>
                            <p className="text-xs text-green-600">
                              {calledFunctions.join(', ')} from {matchedQuestion.serverName}
                            </p>
                          </div>
                        );
                      })()}
                    </div>
                  );
                })()}
                
                <p className="text-xs text-gray-500 mt-3 text-center">
                  {(() => {
                    // Check if this is a generic question by looking at the displayed questions
                    const genericQuestions = displayedQuestions.filter(q => q.type === "generic");
                    const isGenericQuestion = genericQuestions.some(q => q.question === query);
                    
                    return isGenericQuestion 
                      ? "ℹ️ This is an example of a question that doesn't require MCP servers"
                      : "💡 Try toggling servers on/off and asking the same question again to see different responses!";
                  })()}
                </p>
              </motion.div>
            )}
          </div>
        </motion.div>
      </div>

      <div className="flex justify-between items-center">
        <button
          onClick={previousStep}
          className="inline-flex items-center px-6 py-3 text-gray-600 bg-gray-100 rounded-full hover:bg-gray-200 transition-colors"
        >
          <ArrowLeft className="mr-2 w-4 h-4" />
          Back
        </button>
        
        <button
          onClick={nextStep}
          className="inline-flex items-center px-6 py-3 bg-gradient-to-r from-primary to-secondary text-white font-medium rounded-full hover:shadow-lg transition-shadow"
        >
          Try Live with ARIA
          <ArrowRight className="ml-2 w-4 h-4" />
        </button>
      </div>

      {/* Comparison Modal */}
      {selectedServer && (
        <ComparisonModal
          server={selectedServer}
          isOpen={showModal}
          onClose={() => {
            setShowModal(false);
            setSelectedServer(null);
          }}
          isEnabled={enabledServers.includes(selectedServer.id)}
          onToggle={() => toggleServer(selectedServer.id)}
        />
      )}
    </div>
  );
};

export default Playground;