import React, { useState } from "react";
import { motion } from "framer-motion";
import { Award, CheckCircle, Download, RefreshCw, Star } from "lucide-react";
import useProgress from "../hooks/useProgress";
import Quiz from "../components/Quiz";

const QuizPage = () => {
  const {
    resetProgress,
    enabledServers,
    quizScore,
    getTotalTimeSpent,
    userName,
    updateQuizScore,
  } = useProgress();
  const [showCertificate, setShowCertificate] = useState(false);
  const [quizCompleted, setQuizCompleted] = useState(false);

  const totalTimeSpent = getTotalTimeSpent();

  const formatTime = (milliseconds) => {
    const seconds = Math.floor(milliseconds / 1000);
    const minutes = Math.floor(seconds / 60);
    const hours = Math.floor(minutes / 60);

    if (hours > 0) {
      return `${hours}h ${minutes % 60}m`;
    } else if (minutes > 0) {
      return `${minutes}m ${seconds % 60}s`;
    } else {
      return `${seconds}s`;
    }
  };

  const handleQuizComplete = () => {
    setQuizCompleted(true);
    if (quizScore >= 3) {
      setShowCertificate(true);
    }
  };

  const handleSaveCertificate = () => {
    const certificateElement = document.getElementById("certificate");
    if (certificateElement) {
      // Create a new window with only the certificate content
      const printWindow = window.open("", "_blank");
      const certificateHTML = `
        <!DOCTYPE html>
        <html>
          <head>
            <title>MCP Explorer Certificate</title>
            <style>
              @page {
                size: landscape;
                margin: 0;
              }
              
              * {
                margin: 0;
                padding: 0;
                box-sizing: border-box;
              }
              
              body {
                width: 100vw;
                height: 100vh;
                font-family: -apple-system, BlinkMacSystemFont, 'Segoe UI', 'Roboto', sans-serif;
                background: linear-gradient(135deg, #dbeafe 0%, #e0e7ff 100%);
                display: flex;
                align-items: center;
                justify-content: center;
                overflow: hidden;
              }
              
              .certificate {
                width: 11in;
                height: 8.5in;
                background: white;
                border: 8px solid #8b5cf6;
                border-radius: 24px;
                position: relative;
                display: flex;
                flex-direction: column;
                align-items: center;
                justify-content: center;
                text-align: center;
                box-shadow: 0 25px 50px -12px rgba(0, 0, 0, 0.25);
              }
              
              .certificate::before {
                content: '';
                position: absolute;
                inset: 20px;
                border: 3px solid #a855f7;
                border-radius: 16px;
                z-index: 1;
                background: linear-gradient(135deg, rgba(139, 92, 246, 0.05) 0%, rgba(99, 102, 241, 0.05) 100%);
              }
              
              .certificate-content {
                position: relative;
                z-index: 2;
                padding: 40px 60px;
                width: 100%;
              }
              
              .header {
                margin-bottom: 30px;
              }
              
              .certificate-title {
                font-size: 48px;
                font-weight: 800;
                background: linear-gradient(135deg, #8b5cf6 0%, #6366f1 100%);
                -webkit-background-clip: text;
                -webkit-text-fill-color: transparent;
                background-clip: text;
                margin-bottom: 8px;
                text-transform: uppercase;
                letter-spacing: 2px;
              }
              
              .subtitle {
                font-size: 20px;
                color: #6b7280;
                font-weight: 500;
              }
              
              .main-text {
                margin: 40px 0;
                line-height: 1.8;
              }
              
              .certifies {
                font-size: 18px;
                color: #4b5563;
                margin-bottom: 15px;
                font-weight: 500;
              }
              
              .recipient-name {
                font-size: 40px;
                font-weight: 700;
                background: linear-gradient(135deg, #8b5cf6 0%, #6366f1 100%);
                -webkit-background-clip: text;
                -webkit-text-fill-color: transparent;
                background-clip: text;
                margin: 25px 0;
                border-bottom: 3px solid #e5e7eb;
                padding-bottom: 12px;
                display: inline-block;
                min-width: 450px;
              }
              
              .completion-text {
                font-size: 18px;
                color: #4b5563;
                margin-bottom: 10px;
                font-weight: 500;
              }
              
              .course-name {
                font-size: 28px;
                font-weight: 700;
                background: linear-gradient(135deg, #8b5cf6 0%, #6366f1 100%);
                -webkit-background-clip: text;
                -webkit-text-fill-color: transparent;
                background-clip: text;
                margin: 20px 0;
              }
              
              .description {
                font-size: 16px;
                color: #6b7280;
                margin: 30px 0;
                font-weight: 500;
                line-height: 1.6;
              }
              
              .footer {
                display: flex;
                justify-content: space-between;
                align-items: end;
                margin-top: 50px;
                width: 100%;
              }
              
              .date-section, .signature-section {
                text-align: center;
              }
              
              .date, .signature-line {
                border-bottom: 2px solid #8b5cf6;
                padding: 8px 0;
                margin-bottom: 12px;
                min-width: 220px;
                font-size: 15px;
                color: #4b5563;
                font-weight: 600;
              }
              
              .label {
                font-size: 12px;
                color: #6b7280;
                text-transform: uppercase;
                letter-spacing: 1px;
                font-weight: 600;
              }
              
              .logo {
                position: absolute;
                top: 40px;
                left: 60px;
                width: 70px;
                height: 70px;
                background: linear-gradient(135deg, #8b5cf6, #6366f1);
                border-radius: 50%;
                display: flex;
                align-items: center;
                justify-content: center;
                z-index: 3;
                box-shadow: 0 10px 25px rgba(139, 92, 246, 0.3);
              }
              
              .logo svg {
                width: 32px;
                height: 32px;
                fill: white;
              }
              
              @media print {
                body {
                  background: white !important;
                }
                .certificate {
                  box-shadow: none;
                }
              }
            </style>
          </head>
          <body>
            <div class="certificate">
              <div class="logo">
                <svg viewBox="0 0 24 24">
                  <path d="M9 12l2 2 4-4m6 2a9 9 0 11-18 0 9 9 0 0118 0z"/>
                </svg>
              </div>
              
              <div class="certificate-content">
                <div class="header">
                  <div class="certificate-title">Certificate</div>
                  <div class="subtitle">of Achievement</div>
                </div>
                
                <div class="main-text">
                  <div class="certifies">This certifies that</div>
                  
                  <div class="recipient-name">${
                    userName || "Recipient Name"
                  }</div>
                  
                  <div class="completion-text">has successfully completed the comprehensive</div>
                  
                  <div class="course-name">MCP Explorer Learning Experience</div>
                  
                  <div class="description">
                    Demonstrating mastery of Model Context Protocol fundamentals<br/>
                    and understanding of AI assistant enhancement techniques
                  </div>
                </div>
                
                <div class="footer">
                  <div class="date-section">
                    <div class="date">${new Date().toLocaleDateString("en-US", {
                      year: "numeric",
                      month: "long",
                      day: "numeric",
                    })}</div>
                    <div class="label">Date of Completion</div>
                  </div>
                  
                  <div class="signature-section">
                    <div class="signature-line">AI Explorers Program</div>
                    <div class="label">Certified By</div>
                  </div>
                </div>
              </div>
            </div>
          </body>
        </html>
      `;

      printWindow.document.write(certificateHTML);
      printWindow.document.close();
      printWindow.print();
    }
  };

  const handleRetakeQuiz = () => {
    setQuizCompleted(false);
    setShowCertificate(false);
    // Reset quiz score by subtracting current score
    updateQuizScore(-quizScore);
  };

  const keyTakeaways = [
    "MCP lets AI assistants connect to real tools and data sources",
    "Each MCP server provides specific capabilities (calendar, weather, etc.)",
    "AI with MCP can take actions, not just provide information",
    "You can customize your AI by choosing which servers to enable",
    "MCP makes AI more helpful for real-world tasks",
  ];

  return (
    <div className='max-w-4xl mx-auto'>
      {!quizCompleted ? (
        <Quiz onComplete={handleQuizComplete} />
      ) : (
        <>
          <motion.div
            initial={{ scale: 0.8, opacity: 0 }}
            animate={{ scale: 1, opacity: 1 }}
            className='text-center mb-10'
          >
            <div className='inline-flex items-center justify-center w-24 h-24 bg-gradient-to-br from-yellow-400 to-orange-500 rounded-full mb-6'>
              <Award className='w-12 h-12 text-white' />
            </div>

            <h2 className='text-4xl font-bold text-gray-800 mb-4'>
              Congratulations{userName ? `, ${userName.split(" ")[0]}` : ""}! 🎉
            </h2>

            <p className='text-xl text-gray-600'>
              You've mastered the basics of MCP!
            </p>
          </motion.div>

          <motion.div
            initial={{ y: 20, opacity: 0 }}
            animate={{ y: 0, opacity: 1 }}
            transition={{ delay: 0.3 }}
            className='bg-white rounded-2xl shadow-xl p-8 mb-8'
          >
            <h3 className='text-2xl font-semibold text-gray-800 mb-6 flex items-center'>
              <Star className='w-6 h-6 text-yellow-500 mr-2' />
              Your Achievement Summary
            </h3>

            <div className='grid md:grid-cols-4 gap-6 mb-8'>
              <div className='bg-blue-50 rounded-lg p-4 text-center'>
                <div className='text-3xl font-bold text-primary mb-2'>
                  {quizScore}/5
                </div>
                <p className='text-sm text-gray-600'>Quiz Score</p>
              </div>

              <div className='bg-green-50 rounded-lg p-4 text-center'>
                <div className='text-3xl font-bold text-green-600 mb-2'>
                  {enabledServers.length}
                </div>
                <p className='text-sm text-gray-600'>Servers Explored</p>
              </div>

              <div className='bg-orange-50 rounded-lg p-4 text-center'>
                <div className='text-3xl font-bold text-orange-600 mb-2'>
                  {formatTime(totalTimeSpent)}
                </div>
                <p className='text-sm text-gray-600'>Time Studied</p>
              </div>

              <div className='bg-purple-50 rounded-lg p-4 text-center'>
                <div className='text-3xl font-bold text-purple-600 mb-2'>
                  100%
                </div>
                <p className='text-sm text-gray-600'>Journey Complete</p>
              </div>
            </div>

            <div className='border-t pt-6'>
              <h4 className='font-semibold text-gray-800 mb-4'>
                Key Takeaways
              </h4>
              <div className='space-y-3'>
                {keyTakeaways.map((takeaway, index) => (
                  <motion.div
                    key={index}
                    initial={{ x: -20, opacity: 0 }}
                    animate={{ x: 0, opacity: 1 }}
                    transition={{ delay: 0.5 + index * 0.1 }}
                    className='flex items-start'
                  >
                    <CheckCircle className='w-5 h-5 text-green-500 mr-3 mt-0.5 flex-shrink-0' />
                    <p className='text-gray-700'>{takeaway}</p>
                  </motion.div>
                ))}
              </div>
            </div>
          </motion.div>

          {quizCompleted && quizScore < 3 && (
            <motion.div
              initial={{ opacity: 0, y: 20 }}
              animate={{ opacity: 1, y: 0 }}
              transition={{ delay: 0.6 }}
              className='bg-yellow-50 border border-yellow-200 rounded-2xl p-8 mb-8 text-center'
            >
              <div className='inline-flex items-center justify-center w-16 h-16 bg-yellow-100 rounded-full mb-4'>
                <Star className='w-8 h-8 text-yellow-600' />
              </div>

              <h3 className='text-2xl font-semibold text-gray-800 mb-4'>
                Almost There!
              </h3>

              <p className='text-gray-600 mb-2'>
                You scored{" "}
                <span className='font-semibold text-yellow-700'>
                  {quizScore}/5
                </span>{" "}
                on the quiz.
              </p>

              <p className='text-gray-600 mb-6'>
                To earn your certificate, you'll need at least 3 correct
                answers.
                <br />
                Don't worry - you can review the material and try again!
              </p>

              <div className='space-y-4'>
                <p className='text-sm text-gray-500'>
                  💡 <strong>Review Tip:</strong> Go back through the Discovery
                  and Try Live sections to refresh your knowledge about MCP
                  servers and their capabilities.
                </p>

                <button
                  onClick={handleRetakeQuiz}
                  className='inline-flex items-center px-6 py-3 bg-gradient-to-r from-primary to-secondary text-white font-semibold rounded-full shadow-lg hover:shadow-xl transform hover:-translate-y-1 transition-all duration-200'
                >
                  <RefreshCw className='mr-2 w-5 h-5' />
                  Retake Quiz
                </button>
              </div>
            </motion.div>
          )}

          {showCertificate && (
            <motion.div
              id='certificate'
              initial={{ opacity: 0 }}
              animate={{ opacity: 1 }}
              transition={{ delay: 1 }}
              className='bg-gradient-to-r from-primary to-secondary rounded-2xl p-8 text-white mb-8'
            >
              <div className='text-center'>
                <Award className='w-16 h-16 mx-auto mb-4' />
                <h3 className='text-2xl font-bold mb-2'>
                  Certificate of Completion
                </h3>
                <p className='mb-4'>
                  This certifies that{" "}
                  <span className='text-xl font-semibold'>
                    {userName || "you"}
                  </span>{" "}
                  {userName && <br />}
                  {userName ? "has" : "have"} successfully completed the
                  <br />
                  <span className='text-xl font-semibold'>
                    MCP Explorer Learning Experience
                  </span>
                </p>
                <p className='text-sm opacity-90 mb-2'>
                  You now understand how Model Context Protocol enhances AI
                  assistants
                </p>
                {userName && (
                  <p className='text-sm opacity-75'>
                    Awarded to {userName} • {new Date().toLocaleDateString()}
                  </p>
                )}
              </div>
            </motion.div>
          )}

          <div className='text-center space-y-4'>
            {showCertificate && (
              <button
                onClick={handleSaveCertificate}
                className='inline-flex items-center px-6 py-3 bg-white text-primary border-2 border-primary rounded-full hover:bg-primary hover:text-white transition-colors'
              >
                <Download className='mr-2 w-4 h-4' />
                Save Certificate
              </button>
            )}

            {showCertificate && <br />}

            <button
              onClick={resetProgress}
              className='inline-flex items-center px-6 py-3 text-gray-600 hover:text-gray-800 transition-colors'
            >
              <RefreshCw className='mr-2 w-4 h-4' />
              Start Over
            </button>
          </div>

          <motion.div
            initial={{ opacity: 0 }}
            animate={{ opacity: 1 }}
            transition={{ delay: 1.5 }}
            className='mt-12 p-6 bg-gray-50 rounded-lg text-center'
          >
            <h4 className='font-semibold text-gray-800 mb-2'>
              Want to Learn More?
            </h4>
            <p className='text-gray-600 mb-4'>
              Explore how developers are building with MCP and create your own
              AI tools!
            </p>
            <a
              href='https://modelcontextprotocol.io'
              target='_blank'
              rel='noopener noreferrer'
              className='text-primary hover:text-secondary transition-colors font-medium'
            >
              Visit MCP Documentation →
            </a>
          </motion.div>
        </>
      )}
    </div>
  );
};

export default QuizPage;
