import React, { useState, useRef, useEffect } from 'react';
import { motion, AnimatePresence } from 'framer-motion';
import { Send, ArrowRight, ArrowLeft, Zap, Bot, User, Loader } from 'lucide-react';
import useProgress from '../hooks/useProgress';
import { mcpServers } from '../data/mcpServers';
import EnhancedAIResponse from '../components/EnhancedAIResponse';
import ComparisonModal from '../components/ComparisonModal';
import { getFunctionsForServer, getServerFunctionCount } from '../utils/serverFunctions';

const RealAI = () => {
  const { nextStep, previousStep, enabledServers, toggleServer } = useProgress();
  const [query, setQuery] = useState('');
  const [conversations, setConversations] = useState([]);
  const [isLoading, setIsLoading] = useState(false);
  const [selectedServer, setSelectedServer] = useState(null);
  const [showModal, setShowModal] = useState(false);
  const messagesEndRef = useRef(null);
  const chatContainerRef = useRef(null);

  const scrollToBottom = () => {
    if (chatContainerRef.current) {
      chatContainerRef.current.scrollTop = chatContainerRef.current.scrollHeight;
    }
  };

  useEffect(() => {
    scrollToBottom();
  }, [conversations]);

  // Helper function to get enabled server info for prompt
  const getEnabledServerInfo = () => {
    const enabledServerInfo = enabledServers.map(serverId => {
      const server = mcpServers.find(s => s.id === serverId);
      if (!server) return null;
      
      return {
        id: server.id,
        name: server.name,
        description: server.description,
        useCase: server.useCase,
        functions: getFunctionsForServer(serverId).split(', ')
      };
    }).filter(Boolean);

    return enabledServerInfo;
  };


  // Client-side MCP simulation for fallback
  const generateMCPSimulation = (message, enabledServers) => {
    const msg = message.toLowerCase();
    
    if (enabledServers.length === 0) {
      return {
        response: "I don't have access to any MCP servers right now. Please enable some servers to unlock my enhanced capabilities!",
        functionsUsed: [],
        serversUsed: []
      };
    }

    // Determine what type of request this is and which servers can help
    const serverNames = enabledServers.map(s => s.name);
    
    if ((msg.includes('schedul') || msg.includes('calendar') || msg.includes('meet')) && serverNames.includes('Study Scheduler')) {
      return {
        response: "I'll help you schedule that! Using my calendar access, I can check everyone's availability and find the perfect time. Let me use checkAvailability() to see when everyone is free, then bookRoom() to reserve a study space.",
        functionsUsed: ['checkAvailability()', 'bookRoom()'],
        serversUsed: ['Study Scheduler']
      };
    }
    
    if ((msg.includes('playlist') || msg.includes('music') || msg.includes('beats')) && serverNames.includes('Study Beats')) {
      return {
        response: "Perfect! I'll create a study playlist for you. Using createPlaylist() to curate the perfect focus tracks, and analyzeGenre() to match your study mood.",
        functionsUsed: ['createPlaylist()', 'analyzeGenre()'],
        serversUsed: ['Study Beats']
      };
    }
    
    if ((msg.includes('math') || msg.includes('calculat') || msg.includes('solve')) && serverNames.includes('Math Solver')) {
      return {
        response: "I can solve that math problem for you! Using solveEquation() to work through the problem and showSteps() to explain each step clearly.",
        functionsUsed: ['solveEquation()', 'showSteps()'],
        serversUsed: ['Math Solver']
      };
    }
    
    if ((msg.includes('translat') || msg.includes('french') || msg.includes('spanish') || msg.includes('hi in')) && serverNames.includes('Language Tutor')) {
      return {
        response: "I can help with translation! Using translateText() to convert between languages and checkGrammar() to ensure accuracy.",
        functionsUsed: ['translateText()', 'checkGrammar()'],
        serversUsed: ['Language Tutor']
      };
    }
    
    if ((msg.includes('weather') || msg.includes('rain') || msg.includes('outdoor')) && serverNames.includes('Activity Planner')) {
      return {
        response: "Let me check the weather for your plans! Using getCurrentWeather() and getForecast() to give you the best activity recommendations.",
        functionsUsed: ['getCurrentWeather()', 'getForecast()'],
        serversUsed: ['Activity Planner']
      };
    }
    
    // General response using available servers
    const randomServer = enabledServers[Math.floor(Math.random() * enabledServers.length)];
    const functions = getFunctionsForServer(randomServer.id).split(', ').slice(0, 2);
    
    return {
      response: `I can help with that using my ${randomServer.name} capabilities! Let me assist you with the information you need.`,
      functionsUsed: functions,
      serversUsed: [randomServer.name]
    };
  };

  const handleSubmit = async () => {
    if (!query.trim() || isLoading) return;

    const userMessage = {
      id: Date.now(),
      type: 'user',
      content: query,
      timestamp: new Date()
    };

    setConversations(prev => [...prev, userMessage]);
    setIsLoading(true);
    
    const currentQuery = query;
    setQuery('');

    try {
      const enabledServerInfo = getEnabledServerInfo();
      let data;
      
      // Try API first, fallback to client-side simulation if it fails
      try {
        // Try the simpler API endpoint first
        const response = await fetch('/api/chat-simple', {
          method: 'POST',
          headers: {
            'Content-Type': 'application/json',
          },
          body: JSON.stringify({
            message: currentQuery,
            enabledServers: enabledServerInfo
          }),
        });

        if (!response.ok) {
          throw new Error('API not available');
        }

        data = await response.json();
      } catch (apiError) {
        // Fallback to client-side MCP simulation
        console.log('API failed, using simulation:', apiError);
        data = generateMCPSimulation(currentQuery, enabledServerInfo);
      }
      
      const aiMessage = {
        id: Date.now() + 1,
        type: 'ai',
        content: data.response,
        timestamp: new Date(),
        serversUsed: data.serversUsed || [],
        functionsUsed: data.functionsUsed || []
      };

      setConversations(prev => [...prev, aiMessage]);
    } catch (error) {
      console.error('Error:', error);
      const errorMessage = {
        id: Date.now() + 1,
        type: 'ai',
        content: 'Sorry, I encountered an error processing your request. Please try again.',
        timestamp: new Date(),
        serversUsed: [],
        functionsUsed: []
      };
      setConversations(prev => [...prev, errorMessage]);
    } finally {
      setIsLoading(false);
    }
  };


  return (
    <div className="max-w-5xl mx-auto py-12">
      <motion.div
        initial={{ opacity: 0, y: 20 }}
        animate={{ opacity: 1, y: 0 }}
        className="text-center mb-10"
      >
        <h2 className="text-3xl font-bold text-gray-800 mb-4">
          Try MCP Live
        </h2>
        <p className="text-lg text-gray-600 mb-2">
          Experience the power of MCP in real-time! Toggle servers to see how AI capabilities change instantly.
        </p>
        <p className="text-sm text-gray-500 mb-4 flex items-center justify-center">
          <span className="inline-flex items-center px-2 py-1 bg-green-100 text-green-700 rounded-full text-xs mr-2">
            ⚡ LIVE
          </span>
          ✨ Powered by real AI
        </p>
        
        <div className="inline-flex items-center bg-primary/10 px-4 py-2 rounded-full text-sm">
          <Zap className="w-4 h-4 text-primary mr-2" />
          <span className="text-gray-700">
            <strong>{enabledServers.length}</strong> of {mcpServers.length} servers enabled • 
            <strong>{enabledServers.reduce((total, serverId) => {
              return total + getServerFunctionCount(serverId);
            }, 0)}</strong> functions available
          </span>
        </div>
      </motion.div>

      <div className="grid lg:grid-cols-3 gap-6 mb-8">
        {/* Server Controls */}
        <motion.div
          initial={{ opacity: 0, x: -20 }}
          animate={{ opacity: 1, x: 0 }}
          className="lg:col-span-1"
        >
          <div className="bg-white rounded-xl shadow-lg p-6 pb-8 h-[568px] flex flex-col sticky top-6">
            <div className="flex items-center justify-between mb-4">
              <h3 className="font-semibold text-gray-800 flex items-center">
                <Zap className="w-5 h-5 text-primary mr-2" />
                MCP Servers
              </h3>
              <div className="flex items-center space-x-3">
                <span className="text-xs text-gray-600">All</span>
                <button
                  onClick={() => {
                    if (enabledServers.length === mcpServers.length) {
                      // Disable all
                      enabledServers.forEach(serverId => toggleServer(serverId));
                    } else {
                      // Enable all
                      mcpServers.forEach(server => {
                        if (!enabledServers.includes(server.id)) {
                          toggleServer(server.id);
                        }
                      });
                    }
                  }}
                  className={`w-10 h-5 rounded-full transition-colors relative ${
                    enabledServers.length === mcpServers.length ? 'bg-primary' : 'bg-gray-300'
                  }`}
                  title={enabledServers.length === mcpServers.length ? "Disable All Servers" : "Enable All Servers"}
                >
                  <motion.div
                    animate={{ x: enabledServers.length === mcpServers.length ? 20 : 0 }}
                    transition={{ type: "spring", stiffness: 500, damping: 30 }}
                    className="absolute top-0.5 left-0.5 w-4 h-4 bg-white rounded-full shadow-md"
                  />
                </button>
              </div>
            </div>
            
            <div className="space-y-3 overflow-y-auto flex-1">
              {mcpServers.map((server) => {
                const Icon = server.icon;
                const isEnabled = enabledServers.includes(server.id);
                
                return (
                  <div
                    key={server.id}
                    className={`flex items-center justify-between p-3 rounded-lg transition-colors ${
                      isEnabled ? 'bg-green-50' : 'bg-gray-50'
                    }`}
                  >
                    <div className="flex items-center flex-1">
                      <Icon className="w-4 h-4 mr-2 text-gray-400" />
                      <div className="flex-1">
                        <div className="text-sm text-gray-600">
                          {server.name}
                        </div>
                        <button
                          onClick={() => {
                            setSelectedServer(server);
                            setShowModal(true);
                          }}
                          className="text-xs text-gray-500 hover:text-gray-700 transition-colors cursor-pointer flex items-center"
                          title="Click to view server details"
                        >
                          {getServerFunctionCount(server.id)} functions
                          <ArrowRight className="w-3 h-3 ml-1" />
                        </button>
                      </div>
                    </div>
                    
                    <button
                      onClick={(e) => {
                        e.stopPropagation();
                        toggleServer(server.id);
                      }}
                      className={`w-10 h-5 rounded-full transition-colors relative ${
                        isEnabled ? 'bg-primary' : 'bg-gray-300'
                      }`}
                    >
                      <motion.div
                        animate={{ x: isEnabled ? 20 : 0 }}
                        transition={{ type: "spring", stiffness: 500, damping: 30 }}
                        className="absolute top-0.5 left-0.5 w-4 h-4 bg-white rounded-full shadow-md"
                      />
                    </button>
                  </div>
                );
              })}
            </div>
          </div>
        </motion.div>

        {/* Chat Interface */}
        <motion.div
          initial={{ opacity: 0, y: 20 }}
          animate={{ opacity: 1, y: 0 }}
          className="lg:col-span-2"
        >
          <div className="bg-white rounded-xl shadow-lg overflow-hidden flex flex-col">
            {/* Chat Messages */}
            <div ref={chatContainerRef} className="h-[460px] overflow-y-auto p-6 space-y-4">
              {conversations.length === 0 ? (
                <div className="text-center text-gray-500 mt-20">
                  <Bot className="w-12 h-12 mx-auto mb-4 text-gray-300" />
                  <p className="font-medium">Ready to try MCP live?</p>
                  <p className="text-sm mt-2">
                    Ask anything! Enable servers above to unlock powerful AI capabilities in real-time.
                  </p>
                  <p className="text-xs mt-2 text-gray-400">
                    Try: "Create a study playlist" or "Help me schedule a meeting"
                  </p>
                </div>
              ) : (
                <AnimatePresence>
                  {conversations.map((message) => (
                    <motion.div
                      key={message.id}
                      initial={{ opacity: 0, y: 10 }}
                      animate={{ opacity: 1, y: 0 }}
                      className={`flex ${message.type === 'user' ? 'justify-end' : 'justify-start'}`}
                    >
                      <div className={`flex max-w-xs lg:max-w-md ${
                        message.type === 'user' ? 'flex-row-reverse' : 'flex-row'
                      }`}>
                        <div className={`flex-shrink-0 ${
                          message.type === 'user' ? 'ml-2' : 'mr-2'
                        }`}>
                          {message.type === 'user' ? (
                            <User className="w-8 h-8 p-2 bg-primary text-white rounded-full" />
                          ) : (
                            <Bot className="w-8 h-8 p-2 bg-gray-600 text-white rounded-full" />
                          )}
                        </div>
                        <div className={`px-4 py-2 rounded-lg ${
                          message.type === 'user' 
                            ? 'bg-primary text-white' 
                            : message.functionsUsed && message.functionsUsed.length > 0
                            ? 'bg-green-50 text-gray-800 border border-green-200'
                            : 'bg-gray-100 text-gray-800'
                        }`}>
                          {message.type === 'ai' ? (
                            <EnhancedAIResponse 
                              text={message.content} 
                              functionsUsed={message.functionsUsed || []}
                            />
                          ) : (
                            <p className="text-sm whitespace-pre-wrap">{message.content}</p>
                          )}
                          {message.functionsUsed && message.functionsUsed.length > 0 && (
                            <div className="mt-3 p-2 bg-green-100 rounded border-l-4 border-green-400">
                              <p className="text-xs text-green-700 font-medium mb-1">🔧 Functions called:</p>
                              <p className="text-xs text-green-600">
                                {message.functionsUsed.join(', ')} from {message.serversUsed.join(', ')}
                              </p>
                            </div>
                          )}
                        </div>
                      </div>
                    </motion.div>
                  ))}
                </AnimatePresence>
              )}
              
              {isLoading && (
                <motion.div
                  initial={{ opacity: 0, y: 10 }}
                  animate={{ opacity: 1, y: 0 }}
                  className="flex justify-start"
                >
                  <div className="flex">
                    <Bot className="w-8 h-8 p-2 bg-gray-600 text-white rounded-full mr-2" />
                    <div className="px-4 py-2 bg-gray-100 rounded-lg">
                      <Loader className="w-4 h-4 animate-spin" />
                    </div>
                  </div>
                </motion.div>
              )}
              
              <div ref={messagesEndRef} />
            </div>

            {/* Input Section */}
            <div className="border-t p-4 bg-gray-50 flex-shrink-0">
              <div className="flex space-x-2">
                <input
                  type="text"
                  value={query}
                  onChange={(e) => setQuery(e.target.value)}
                  onKeyPress={(e) => e.key === 'Enter' && !e.shiftKey && handleSubmit()}
                  placeholder="Try MCP live - ask anything! (e.g., 'Create a study playlist')"
                  className="flex-1 px-4 py-3 border border-gray-300 rounded-lg focus:outline-none focus:ring-2 focus:ring-primary bg-white"
                  disabled={isLoading}
                />
                <button
                  onClick={handleSubmit}
                  className="px-6 py-3 bg-gradient-to-r from-primary to-secondary text-white rounded-lg hover:shadow-lg transition-shadow disabled:opacity-50"
                  disabled={!query.trim() || isLoading}
                >
                  <Send className="w-5 h-5" />
                </button>
              </div>
              <p className="text-xs text-gray-500 mt-2">
                Real AI responds based on your enabled MCP servers • Toggle servers to see capabilities change live
              </p>
            </div>
          </div>
        </motion.div>
      </div>

      {/* Navigation */}
      <div className="flex justify-between items-center">
        <button
          onClick={previousStep}
          className="inline-flex items-center px-6 py-3 text-gray-600 bg-gray-100 rounded-full hover:bg-gray-200 transition-colors"
        >
          <ArrowLeft className="mr-2 w-4 h-4" />
          Back
        </button>
        
        <button
          onClick={nextStep}
          className="inline-flex items-center px-6 py-3 bg-gradient-to-r from-primary to-secondary text-white font-medium rounded-full hover:shadow-lg transition-shadow"
        >
          Continue to Challenge
          <ArrowRight className="ml-2 w-4 h-4" />
        </button>
      </div>

      {/* Comparison Modal */}
      {selectedServer && (
        <ComparisonModal
          server={selectedServer}
          isOpen={showModal}
          onClose={() => {
            setShowModal(false);
            setSelectedServer(null);
          }}
          isEnabled={enabledServers.includes(selectedServer.id)}
          onToggle={() => toggleServer(selectedServer.id)}
        />
      )}
    </div>
  );
};

export default RealAI;