import React, { useState } from 'react';
import { motion } from 'framer-motion';
import { ArrowRight, ArrowLeft, Cpu, Server, Code, MessageSquare, Brain, Network, Settings } from 'lucide-react';
import useProgress from '../hooks/useProgress';
import { mcpServers } from '../data/mcpServers';
import { getFunctionsForServer } from '../utils/serverFunctions';

const ArchitectureSection = () => (
  <div className="space-y-6">
    <div className="bg-gradient-to-r from-blue-50 to-purple-50 rounded-xl p-6">
      <h3 className="text-xl font-semibold text-gray-800 mb-4">The Big Picture</h3>
      <div className="grid md:grid-cols-3 gap-4">
        <div className="bg-white rounded-lg p-4 text-center">
          <Brain className="w-8 h-8 text-blue-600 mx-auto mb-2" />
          <h4 className="font-semibold text-gray-800">ARIA (AI Assistant)</h4>
          <p className="text-sm text-gray-600">The smart brain that understands your requests</p>
        </div>
        <div className="bg-white rounded-lg p-4 text-center">
          <MessageSquare className="w-8 h-8 text-purple-600 mx-auto mb-2" />
          <h4 className="font-semibold text-gray-800">MCP Connection</h4>
          <p className="text-sm text-gray-600">The communication bridge</p>
        </div>
        <div className="bg-white rounded-lg p-4 text-center">
          <Server className="w-8 h-8 text-green-600 mx-auto mb-2" />
          <h4 className="font-semibold text-gray-800">MCP Servers</h4>
          <p className="text-sm text-gray-600">Specialized tools for different tasks</p>
        </div>
      </div>
    </div>
    
    <div className="bg-white border-l-4 border-blue-500 p-4 rounded-lg">
      <h4 className="font-semibold text-gray-800 mb-2">Think of it like this:</h4>
      <p className="text-gray-700">
        ARIA is like a super-smart student with a smartphone. MCP is like having the perfect app store where each app (server) gives ARIA specific superpowers - checking calendars, getting weather, solving math problems, and more!
      </p>
    </div>
    
    <div className="bg-white border border-gray-200 rounded-lg p-4">
      <h4 className="font-semibold text-gray-800 mb-3">How They Work Together</h4>
      <div className="space-y-2">
        <div className="bg-blue-50 p-3 rounded">
          <strong className="text-blue-800">🎯 ARIA gets your request:</strong> <span className="text-gray-700">"When can I meet with my study group?"</span>
        </div>
        <div className="bg-purple-50 p-3 rounded">
          <strong className="text-purple-800">🔄 MCP connects:</strong> <span className="text-gray-700">ARIA talks to the right servers</span>
        </div>
        <div className="bg-green-50 p-3 rounded">
          <strong className="text-green-800">⚡ Servers respond:</strong> <span className="text-gray-700">Calendar server finds free times</span>
        </div>
      </div>
    </div>
  </div>
);

const CommunicationSection = () => (
  <div className="space-y-6">
    <div className="bg-gradient-to-r from-green-50 to-blue-50 rounded-xl p-6">
      <h3 className="text-xl font-semibold text-gray-800 mb-4">How ARIA Talks to Servers</h3>
      <div className="space-y-4">
        <div className="flex items-center bg-white rounded-lg p-4">
          <div className="bg-blue-100 rounded-full p-2 mr-4">
            <span className="text-blue-600 font-bold">1</span>
          </div>
          <div>
            <h5 className="font-semibold text-gray-800">Say Hello!</h5>
            <p className="text-sm text-gray-600">ARIA introduces herself and asks "What can you do?"</p>
          </div>
        </div>
        
        <div className="flex items-center bg-white rounded-lg p-4">
          <div className="bg-purple-100 rounded-full p-2 mr-4">
            <span className="text-purple-600 font-bold">2</span>
          </div>
          <div>
            <h5 className="font-semibold text-gray-800">Server Shows Off</h5>
            <p className="text-sm text-gray-600">Each server lists its special abilities and functions</p>
          </div>
        </div>
        
        <div className="flex items-center bg-white rounded-lg p-4">
          <div className="bg-green-100 rounded-full p-2 mr-4">
            <span className="text-green-600 font-bold">3</span>
          </div>
          <div>
            <h5 className="font-semibold text-gray-800">Make the Request</h5>
            <p className="text-sm text-gray-600">ARIA asks the server to do something specific</p>
          </div>
        </div>
        
        <div className="flex items-center bg-white rounded-lg p-4">
          <div className="bg-orange-100 rounded-full p-2 mr-4">
            <span className="text-orange-600 font-bold">4</span>
          </div>
          <div>
            <h5 className="font-semibold text-gray-800">Get the Answer</h5>
            <p className="text-sm text-gray-600">Server sends back the information ARIA needs</p>
          </div>
        </div>
      </div>
    </div>
    
    <div className="bg-white border border-gray-200 rounded-lg p-4">
      <h4 className="font-semibold text-gray-800 mb-2">Real Example: Finding Study Time</h4>
      <div className="space-y-3">
        <div className="bg-blue-50 p-3 rounded">
          <p className="text-sm"><strong>You ask:</strong> "When can my study group meet this week?"</p>
        </div>
        <div className="bg-purple-50 p-3 rounded">
          <p className="text-sm"><strong>ARIA thinks:</strong> "I need to check calendars for this!"</p>
        </div>
        <div className="bg-green-50 p-3 rounded">
          <p className="text-sm"><strong>ARIA asks Calendar Server:</strong> "Show me free times for these people"</p>
        </div>
        <div className="bg-orange-50 p-3 rounded">
          <p className="text-sm"><strong>Calendar Server replies:</strong> "Tuesday 3-5pm: Everyone's free!"</p>
        </div>
      </div>
    </div>
  </div>
);

const ServersSection = () => (
  <div className="space-y-6">
    <div className="bg-gradient-to-r from-purple-50 to-pink-50 rounded-xl p-6">
      <h3 className="text-xl font-semibold text-gray-800 mb-4">Inside Each MCP Server</h3>
      <p className="text-gray-600 mb-4">
        Each server is like a specialized app with multiple functions. Here's what ARIA can do with each one:
      </p>
    </div>
    
    <div className="grid gap-4 max-h-96 overflow-y-auto">
      {mcpServers.slice(0, 6).map((server) => {
        const Icon = server.icon;
        const functions = getFunctionsForServer(server.id);
        
        return (
          <div key={server.id} className="bg-white border rounded-lg p-4">
            <div className="flex items-center mb-2">
              <Icon className="w-5 h-5 text-gray-600 mr-2" />
              <h4 className="font-semibold text-gray-800">{server.name}</h4>
            </div>
            <p className="text-sm text-gray-600 mb-2">{server.description}</p>
            <div className="bg-gray-50 rounded p-2">
              <code className="text-xs text-gray-700">{functions}</code>
            </div>
          </div>
        );
      })}
    </div>
  </div>
);

const FunctionCallingSection = () => (
  <div className="space-y-6">
    <div className="bg-gradient-to-r from-yellow-50 to-orange-50 rounded-xl p-6">
      <h3 className="text-xl font-semibold text-gray-800 mb-4">Why MCP is So Smart</h3>
      <div className="space-y-4">
        <div className="bg-white rounded-lg p-4">
          <h5 className="font-semibold text-gray-800 mb-2">🎯 ARIA Picks the Right Tools</h5>
          <p className="text-sm text-gray-600 mb-3">
            When you ask "When can my study group meet?", ARIA doesn't just randomly use everything. 
            She's smart about picking exactly what she needs:
          </p>
          <ul className="text-sm text-gray-600 space-y-1">
            <li>• <code className="bg-gray-100 px-1 rounded">checkAvailability()</code> - sees everyone's free time</li>
            <li>• <code className="bg-gray-100 px-1 rounded">findRooms()</code> - locates available study spaces</li>
            <li>• <code className="bg-gray-100 px-1 rounded">sendInvites()</code> - coordinates the meeting</li>
          </ul>
        </div>
        
        <div className="bg-white rounded-lg p-4">
          <h5 className="font-semibold text-gray-800 mb-2">🔄 Servers Can Update Themselves</h5>
          <p className="text-sm text-gray-600">
            MCP servers can tell ARIA "Hey, I just learned new tricks!" This means your AI assistant 
            can get better over time without you doing anything.
          </p>
        </div>
        
        <div className="bg-white rounded-lg p-4">
          <h5 className="font-semibold text-gray-800 mb-2">⚡ The Magic Balance</h5>
          <div className="grid md:grid-cols-2 gap-3">
            <div className="bg-green-50 p-3 rounded">
              <h6 className="font-medium text-green-800">Fewer Servers</h6>
              <p className="text-xs text-green-600">Super fast responses, but fewer abilities</p>
            </div>
            <div className="bg-orange-50 p-3 rounded">
              <h6 className="font-medium text-orange-800">More Servers</h6>
              <p className="text-xs text-orange-600">Amazing capabilities, but takes a bit longer</p>
            </div>
          </div>
        </div>
        
        <div className="bg-white rounded-lg p-4">
          <h5 className="font-semibold text-gray-800 mb-2">🚀 Why This Matters for You</h5>
          <p className="text-sm text-gray-600">
            Understanding MCP helps you become a power user of AI assistants. You'll know which 
            "apps" (servers) to enable for different tasks, making your AI helper incredibly efficient!
          </p>
        </div>
      </div>
    </div>
  </div>
);

const RealWorldSection = () => (
  <div className="space-y-6">
    <div className="bg-gradient-to-r from-indigo-50 to-purple-50 rounded-xl p-6">
      <h3 className="text-xl font-semibold text-gray-800 mb-4">Real World vs. Learning Mode</h3>
      <div className="space-y-4">
        <div className="bg-white rounded-lg p-4">
          <h5 className="font-semibold text-gray-800 mb-2">🎓 Learning Mode (This App)</h5>
          <p className="text-sm text-gray-600 mb-3">
            In this app, we use simple toggle switches to turn MCP servers on and off. It's like using training wheels 
            to learn how to ride a bike - it helps you understand the concepts without getting overwhelmed!
          </p>
          <div className="bg-gray-50 rounded p-3">
            <p className="text-xs text-gray-600">✅ Toggle Calendar Server → ARIA can check schedules</p>
            <p className="text-xs text-gray-600">✅ Toggle Weather Server → ARIA can check the forecast</p>
          </div>
        </div>
        
        <div className="bg-white rounded-lg p-4">
          <h5 className="font-semibold text-gray-800 mb-2">💻 Real World (What Developers Do)</h5>
          <p className="text-sm text-gray-600 mb-3">
            In real applications, developers write code to connect MCP servers. Think of it like the difference between 
            using Instagram (toggles) vs. building Instagram (coding). Here's a simplified example:
          </p>
          <div className="bg-gray-900 rounded p-3">
            <code className="text-xs text-green-400">
              <div>// Developers write code like this:</div>
              <div>const calendarServer = new MCPServer('calendar');</div>
              <div>aria.connect(calendarServer);</div>
            </code>
          </div>
        </div>
        
        <div className="bg-white rounded-lg p-4">
          <h5 className="font-semibold text-gray-800 mb-2">🌟 Why We Simplified It</h5>
          <p className="text-sm text-gray-600">
            Our toggle interface lets you focus on understanding <strong>what MCP does</strong> rather than <strong>how to code it</strong>. Once you understand the concepts, learning to code it becomes much easier!
          </p>
        </div>
        
        <div className="bg-blue-50 border border-blue-200 rounded-lg p-4">
          <h5 className="font-semibold text-blue-800 mb-2">🎯 The Important Part</h5>
          <p className="text-sm text-blue-700">
            Whether using toggles or code, the core idea is the same: MCP servers give AI assistants new abilities. 
            You're learning the same concepts that professional developers use - just in a friendlier way!
          </p>
        </div>
      </div>
    </div>
  </div>
);

const TechnicalOverview = () => {
  const { nextStep, previousStep } = useProgress();
  const [activeSection, setActiveSection] = useState('architecture');

  const sections = [
    { id: 'architecture', title: 'The Big Picture', icon: Network, component: ArchitectureSection },
    { id: 'communication', title: 'How They Talk', icon: MessageSquare, component: CommunicationSection },
    { id: 'servers', title: 'Server Powers', icon: Settings, component: ServersSection },
    { id: 'function-calling', title: 'Why It\'s Smart', icon: Code, component: FunctionCallingSection },
    { id: 'real-world', title: 'Real World vs. Learning', icon: Cpu, component: RealWorldSection }
  ];

  const activeContent = sections.find(section => section.id === activeSection);
  const ActiveComponent = activeContent.component;

  return (
    <div className="max-w-6xl mx-auto py-12">
      <motion.div
        initial={{ opacity: 0, y: 20 }}
        animate={{ opacity: 1, y: 0 }}
        className="text-center mb-10"
      >
        <h2 className="text-3xl font-bold text-gray-800 mb-4">
          🔧 How MCP Powers ARIA for Sam
        </h2>
        <p className="text-lg text-gray-600 mb-4">
          Let's explore how MCP works behind the scenes - think of it as AI's secret superpower system!
        </p>
      </motion.div>

      <div className="grid lg:grid-cols-4 gap-6">
        {/* Section Navigation */}
        <motion.div
          initial={{ opacity: 0, x: -20 }}
          animate={{ opacity: 1, x: 0 }}
          className="lg:col-span-1"
        >
          <div className="bg-white rounded-xl shadow-lg p-6 sticky top-6">
            <h3 className="font-semibold text-gray-800 mb-4">Technical Topics</h3>
            <div className="space-y-2">
              {sections.map((section) => {
                const Icon = section.icon;
                const isActive = activeSection === section.id;
                
                return (
                  <button
                    key={section.id}
                    onClick={() => setActiveSection(section.id)}
                    className={`w-full flex items-center p-3 rounded-lg transition-colors text-left ${
                      isActive 
                        ? 'bg-primary/10 border border-primary text-primary' 
                        : 'hover:bg-gray-50 text-gray-600'
                    }`}
                  >
                    <Icon className="w-4 h-4 mr-3" />
                    <span className="text-sm font-medium">{section.title}</span>
                  </button>
                );
              })}
            </div>
            
            <div className="mt-6 p-4 bg-blue-50 rounded-lg">
              <div className="flex items-start">
                <Cpu className="w-4 h-4 text-blue-600 mr-2 mt-0.5" />
                <div>
                  <p className="text-xs text-blue-700">
                    <strong>Pro tip:</strong> Understanding these concepts will help you become a better AI user and maybe even inspire you to build your own MCP servers someday!
                  </p>
                </div>
              </div>
            </div>
          </div>
        </motion.div>

        {/* Content Area */}
        <motion.div
          initial={{ opacity: 0, x: 20 }}
          animate={{ opacity: 1, x: 0 }}
          className="lg:col-span-3"
        >
          <div className="bg-white rounded-xl shadow-lg p-8">
            <div className="flex items-center mb-6">
              <activeContent.icon className="w-6 h-6 text-primary mr-3" />
              <h3 className="text-2xl font-bold text-gray-800">{activeContent.title}</h3>
            </div>
            
            <ActiveComponent />
          </div>
        </motion.div>
      </div>


      {/* Navigation */}
      <div className="flex justify-between items-center mt-8">
        <button
          onClick={previousStep}
          className="inline-flex items-center px-6 py-3 text-gray-600 bg-gray-100 rounded-full hover:bg-gray-200 transition-colors"
        >
          <ArrowLeft className="mr-2 w-4 h-4" />
          Back
        </button>
        
        <button
          onClick={nextStep}
          className="inline-flex items-center px-6 py-3 bg-gradient-to-r from-primary to-secondary text-white font-medium rounded-full hover:shadow-lg transition-shadow"
        >
          Choose Servers for Sam
          <ArrowRight className="ml-2 w-4 h-4" />
        </button>
      </div>
    </div>
  );
};

export default TechnicalOverview;