import random

import numpy as np
import pandas as pd
from scipy import optimize


COST_1 = True


def maximize_sharpe_ratio(all_returns, covariance_returns):
    # The sharpe ratio we want to optimize
    def f(x, returns, cov):
        portfolio_std = np.sqrt(np.matmul(np.matmul(x, cov), x.T))
        portfolio_returns = np.matmul(np.array(returns), x.T)
        sharpe = -portfolio_returns / portfolio_std
        return sharpe

    # The equality constraints
    # Either sum=1 or sum=0
    def equality_constraint(x):
        A = np.ones(x.shape)
        if COST_1:
            b = 1
        else:
            b = 0
        return np.matmul(A, x.T) - b

    # We start with all the weights at 0
    initial_weights = np.repeat(0.33, len(all_returns))
    all_constraints = ({'type': 'eq', 'fun': equality_constraint})
    # Weights between -1 and 1
    if COST_1:
        bounds = tuple([(0, 1) for _ in range(len(all_returns))])
    else:
        bounds = tuple([(-1, 1) for _ in range(len(all_returns))])

    # invoke minimize solver
    opt = optimize.minimize(f, x0=initial_weights, args=(all_returns, covariance_returns), method='SLSQP',
                            bounds=bounds, constraints=all_constraints, tol=10 ** -5)

    return opt


def get_optimal_weights(returns, covs):
    result = maximize_sharpe_ratio(returns, covs)
    return np.array(result.x)


if __name__ == '__main__':
    filename = "variance_portfolios/10-10-2022.csv"
    cov = pd.read_csv(filename, index_col="index")
    mean_returns = [random.random() * 2 - 1 for _ in range(len(cov))]
    optimal_weights = get_optimal_weights(mean_returns, cov)

    risk = np.matmul((np.matmul(optimal_weights, cov)), np.transpose(optimal_weights))
    expected_return = np.matmul(np.array(mean_returns), optimal_weights.T)
    annualized_risk = np.sqrt(risk * 252)
    annualized_return = 252 * np.array(expected_return)
    max_sharpe_ratio = annualized_return / annualized_risk

    # set precision for printing results
    np.set_printoptions(precision=3, suppress=True)

    # display results
    print('Maximal Sharpe Ratio: ', max_sharpe_ratio, '\nAnnualized Risk (%):  ',
          annualized_risk, '\nAnnualized Expected Portfolio Return(%):  ', annualized_return)
    print('\nOptimal weights (%):\n', optimal_weights.T * 100)
