import argparse


def get_config():
    """
    Parse command-line arguments for configuration.

    Returns:
        argparse.Namespace: Parsed arguments.
    """
    parser = argparse.ArgumentParser(
        description="Configuration for GAN training and evaluation."
    )

    # Dataset and evaluation parameters
    parser.add_argument(
        "--dataset",
        default="mnist",
        help="Dataset to use (options: 'mnist', 'fashionmnist', 'cifar10')",
    )
    parser.add_argument(
        "--eval_num", default=12800, type=int, help="Number of images for evaluation"
    )
    parser.add_argument(
        "--noise_std", type=float, default=0.0, help="Standard deviation for noise"
    )
    parser.add_argument(
        "--threshold", type=int, help="Threshold value for evaluation (if applicable)"
    )

    # General training settings
    parser.add_argument(
        "--manualSeed", type=int, default=1, help="Manual seed for reproducibility"
    )
    parser.add_argument(
        "--dataroot", default="/path/to/dataset", help="Path to the dataset directory"
    )
    parser.add_argument(
        "--workers", type=int, default=2, help="Number of data loading workers"
    )
    parser.add_argument("--batchSize", type=int, default=256, help="Input batch size")
    parser.add_argument(
        "--nz", type=int, default=100, help="Size of the latent z vector"
    )
    parser.add_argument(
        "--ngf", type=int, default=64, help="Number of generator filters"
    )
    parser.add_argument(
        "--ndf", type=int, default=64, help="Number of discriminator filters"
    )
    parser.add_argument(
        "--niter", type=int, default=1, help="Number of epochs to train for"
    )
    parser.add_argument("--lr", type=float, default=0.0005, help="Learning rate")
    parser.add_argument(
        "--beta1",
        type=float,
        default=0.5,
        help="Beta1 hyperparameter for Adam optimizer",
    )

    # GPU and device settings
    parser.add_argument("--cuda", action="store_true", default=True, help="Enable CUDA")
    parser.add_argument("--ngpu", type=int, default=1, help="Number of GPUs to use")
    parser.add_argument(
        "--mps", action="store_true", default=False, help="Enable macOS GPU training"
    )

    # Check options
    parser.add_argument(
        "--dry-run", action="store_true", help="Run a single training cycle for testing"
    )

    # Pre-trained models (for continuing training)
    parser.add_argument(
        "--netG", default="", help="Path to pre-trained generator model"
    )
    parser.add_argument(
        "--netD", default="", help="Path to pre-trained discriminator model"
    )

    return parser.parse_args()


if __name__ == "__main__":
    config = get_config()
    print("Parsed Configuration:")
    for arg in vars(config):
        print(f"{arg}: {getattr(config, arg)}")
