from torch import nn

class Generator(nn.Module):
    """
    Generator model for a GAN that transforms a latent vector into an output vector.
    
    The model is composed of a series of linear blocks, each optionally followed by
    batch normalization and a LeakyReLU activation. The final layer produces a 3-dimensional
    output.
    """
    def __init__(self):
        """
        Initialize the Generator.
        """
        super(Generator, self).__init__()

        def block(in_features, out_features, normalize=True):
            """
            Create a block consisting of a linear layer, optional batch normalization,
            and a LeakyReLU activation.
            
            Args:
                in_features (int): Number of input features.
                out_features (int): Number of output features.
                normalize (bool): If True, apply BatchNorm1d.
            
            Returns:
                list: A list of layers for this block.
            """
            layers = [nn.Linear(in_features, out_features)]
            if normalize:
                # Momentum is set to 0.8
                layers.append(nn.BatchNorm1d(out_features, momentum=0.8))
            layers.append(nn.LeakyReLU(0.2, inplace=True))
            return layers

        # Build the generator model as a sequence of blocks.
        # The architecture is:
        #   Input (3) -> Linear+ReLU -> 128 -> Linear+ReLU -> 256 -> Linear+ReLU -> 512 -> Linear -> 3
        self.model = nn.Sequential(
            *block(3, 128, normalize=False),    # First block (no normalization)
            *block(128, 256, normalize=False),   # Second block (no normalization)
            *block(256, 512, normalize=False),   # Third block (no normalization)
            nn.Linear(512, 3)                    # Final layer producing a 3-dimensional output
        )

    def forward(self, z):
        """
        Forward pass for the generator.
        
        Args:
            z (Tensor): Input latent vector.
        
        Returns:
            Tensor: Generated output.
        """
        img = self.model(z)
        return img


class Discriminator(nn.Module):
    """
    Discriminator model for a GAN that classifies input vectors as real or fake.
    
    The model consists of several linear layers with LeakyReLU activations, and ends with a Sigmoid
    activation to output a probability.
    """
    def __init__(self):
        """
        Initialize the Discriminator.
        """
        super(Discriminator, self).__init__()

        # Build the discriminator model as a sequence of linear layers.
        self.model = nn.Sequential(
            nn.Linear(3, 512),                 # First linear layer transforms input from 3 to 512 features
            nn.LeakyReLU(0.2, inplace=True),    # LeakyReLU activation
            nn.Linear(512, 512),                # Second linear layer
            nn.LeakyReLU(0.2, inplace=True),    # LeakyReLU activation
            nn.Linear(512, 1),                  # Final linear layer to output a single value
            nn.Sigmoid()                        # Sigmoid activation to produce a probability
        )

    def forward(self, img):
        """
        Forward pass for the discriminator.
        
        Args:
            img (Tensor): Input vector (either a real or generated sample).
        
        Returns:
            Tensor: Probability of the input being real.
        """
        validity = self.model(img)
        return validity
