import numpy as np
import gym
from gym import utils
from gym.envs.mujoco import mujoco_env
from gym.envs.registration import registry, register, make, spec
import os

DEFAULT_CAMERA_CONFIG = {
    'distance': 20.0,
}


class AntTrapEnv(mujoco_env.MujocoEnv, utils.EzPickle):
    def __init__(self):
        local_path = os.path.dirname(__file__)
        xml_file = local_path + "/mujoco_assets/ant_trap.xml"
        mujoco_env.MujocoEnv.__init__(self, xml_file, 5)
        utils.EzPickle.__init__(self)

    def step(self, a):
        xposbefore = self.get_body_com("torso")[0]
        self.do_simulation(a, self.frame_skip)
        xposafter = self.get_body_com("torso")[0]
        xy_position_after = self.get_body_com("torso")[:2].copy()

        forward_reward = (xposafter - xposbefore)/self.dt
        ctrl_cost = .5 * np.square(a).sum()
        contact_cost = 0.5 * 1e-3 * np.sum(
            np.square(np.clip(self.sim.data.cfrc_ext, -1, 1)))
        survive_reward = 1.0
        reward = forward_reward - ctrl_cost - contact_cost + survive_reward
        state = self.state_vector()
        notdone = np.isfinite(state).all() and state[2] >= 0.2 and state[2] <= 1.0
        done = not notdone
        ob = self._get_obs()
        return ob, reward, done, dict(
            reward_forward=forward_reward,
            reward_ctrl=-ctrl_cost,
            reward_contact=-contact_cost,
            reward_survive=survive_reward,
            x_position=np.maximum(0, xy_position_after[0]),        # modified for clipping the bc
            y_position=np.clip(xy_position_after[1], -8, 8))       # modified for clipping the bc

    def _get_obs(self):
        return np.concatenate([
            self.sim.data.qpos.flat,
            self.sim.data.qvel.flat,
            np.clip(self.sim.data.cfrc_ext, -1, 1).flat,
        ])

    def reset_model(self):
        qpos = self.init_qpos + self.np_random.uniform(size=self.model.nq, low=-.1, high=.1)
        qvel = self.init_qvel + self.np_random.randn(self.model.nv) * .1
        self.set_state(qpos, qvel)
        return self._get_obs()

    def viewer_setup(self):
        self.viewer.cam.distance = self.model.stat.extent * 0.5


register(
    id='AntTrap-v0',
    max_episode_steps=1000,
    entry_point='envs.ant_trap:AntTrapEnv',
)