#!/bin/bash

# 1. Extract model name and dataset name from config
mapfile -t config_values < <(python config_info.py)
MODEL_NAME=${config_values[0]}
MODEL_PATH=${config_values[1]}
DATA_NAME=${config_values[2]}

# 2. Set output directory and log file path (absolute paths)
OUTPUT_DIR="$HOME/FEAT/experiments/${MODEL_NAME}/${DATA_NAME}"
LOGFILE="$OUTPUT_DIR/latency.log"
mkdir -p "$OUTPUT_DIR"

# 3. Lock the current config by copying it to a unique directory
CONFIG_DIR="./configs"
LOCKED_CONFIG_DIR="./locked_configs/${MODEL_NAME}/${DATA_NAME}"
mkdir -p "$LOCKED_CONFIG_DIR"
cp "$CONFIG_DIR"/* "$LOCKED_CONFIG_DIR/"

# 4. Start logging
echo "Execution started: $(date)" > "$LOGFILE"

# 5. Function to run each Python script and log its execution time and config
run_and_log_time() {
    local script=$1
    echo "Running $script..." | tee -a "$LOGFILE"

    # Log fixed config info
    echo "  ↳ [CONFIG] model = $MODEL_NAME, path = $MODEL_PATH, data = $DATA_NAME" | tee -a "$LOGFILE"

    start=$(date +%s.%N)

    # Run Python script with locked config
    python "$script" --config-dir "$LOCKED_CONFIG_DIR"

    end=$(date +%s.%N)
    duration=$(echo "$end - $start" | bc)
    echo "  ↳ Execution time for $script: ${duration} seconds" | tee -a "$LOGFILE"
    echo "" >> "$LOGFILE"
}

# 6. Record total execution time
total_start=$(date +%s.%N)

run_and_log_time "extract_attention.py"
run_and_log_time "calculating_feature.py"
run_and_log_time "calculating_barcode.py"
run_and_log_time "calculating_risper.py"
run_and_log_time "calculating_template.py"

total_end=$(date +%s.%N)
total_duration=$(echo "$total_end - $total_start" | bc)
echo "Total execution time: ${total_duration} seconds" | tee -a "$LOGFILE"
echo "Execution finished: $(date)" >> "$LOGFILE"

# 7. Remove locked config after execution to avoid clutter
rm -rf "$LOCKED_CONFIG_DIR"