
import os
import time
import pandas as pd
from transformers import AutoTokenizer
import json
import vllm
from vllm import LLM, SamplingParams

def load_math_questions(file_path=None, tokenizer = None):
    """
    读取 JSON 文件，提取 'first_dialog' 和 'answer' 字段，并返回。
    """
    # 读取 JSON 文件
    with open(file_path, 'r') as f:
        data = json.load(f)
    student_message = \
            "You are a student with poor knowledge mastery and calculation ability. Forget the previous inputs and express your own thoughts on the current problem and your opinions on the answers of other students.\n" + \
            "You are responsible for playing the role of Student B. Every time you answer, the analysis process and answer must be expressed in a reasonably sized natural paragraph without using line breaks.\n" + \
            "Your response must include the complete steps to obtain the result, listing all steps, and provide the final answer at the end. Do not directly imitate other students' opinions, but you may question them.\n" + \
            "You are limited to playing the role of the student agent, focusing on the topic debate.\n"+\
            "You must respond in English."
    # 提取 'first_dialog' 和 'answer' 字段
    first_dialogs = [item.get('first_dialog') for item in data[:3000]]  # 取前 5000 个
    answers = [item.get('answer') for item in data[:3000]]  # 取前 5000 个
    
    # 将 teacher_message 拼接到每个 first_dialog 的前面
    prompt_first_dialogs = []
    for dialog in first_dialogs:
        messages = [
            {"role": "system", "content": student_message},
            {"role": "user", "content": dialog}
        ]
        # 使用 tokenizer 格式化
        formatted_question = format_prompt(messages, tokenizer)
        prompt_first_dialogs.append(formatted_question)
       
    return prompt_first_dialogs, first_dialogs, answers

def solve_math_problems(prompt_first_dialogs, first_dialogs, answers, llm, batch_size=3000):
    """
    以批次处理数学题目，调用模型生成解答，并返回题目、答案和历史记录的对。
    """
    history = []
    total_dialogs = len(prompt_first_dialogs)
    sampling_params = vllm.SamplingParams(temperature=0.6, top_p=0.95, max_tokens = 4096)
    for i in range(0, total_dialogs, batch_size):
        # 提取当前批次的问题
        batch_prompt_first_dialogs = prompt_first_dialogs[i:i + batch_size]
        batch_answers = answers[i:i + batch_size]
        batch_first_dialogs = first_dialogs[i:i + batch_size]
        # 为模型准备输入的批次（拼接问题文本）
        prompts = batch_prompt_first_dialogs  # 直接使用问题批次
        
        # 生成每个批次的解答
        batch_responses = llm.generate(prompts,sampling_params)
        
        # 处理每个问题的模型反馈并保存历史记录
        for idx, (correct_answer, first_dialog, response) in enumerate(zip(batch_answers, batch_first_dialogs, batch_responses)):
            dialog_history = {
                "second_dialog": first_dialog + "<StudentB>" +response.outputs[0].text ,
                "answer": correct_answer
            }
            history.append(dialog_history)
            
            # 输出每个问题的处理进度
            print(f"Processing question {i + idx + 1}/{total_dialogs}...")

        # 显示当前批次的处理进度
        print(f"Batch {i // batch_size + 1} processed, total {len(history)} dialog histories collected.")
        
    return history

def save_to_json(data, file_name="./output/second_turn.json"):
    """
    将对话历史和答案保存为 JSON 文件
    """
    # 确保目录存在
    os.makedirs(os.path.dirname(file_name), exist_ok=True)
    
    with open(file_name, 'w') as f:
        json.dump(data, f, indent=4)
    print(f"Data saved to {file_name}.")

def format_prompt(messages, tokenizer):
    """将消息列表转换为合规的文本输入"""
    try:
        # 使用分词器的聊天模板格式化
        text = tokenizer.apply_chat_template(
            messages,
            tokenize=False,
            add_generation_prompt=True
        )
        return text
    except Exception as e:
        print(f"格式化提示词时出错: {e}")
        # 回退到手动格式
        return "\n".join([f"{m['role']}: {m['content']}" for m in messages])
    
def main():
# 创建采样参数对象
    print("start......")
    # 加载本地模型（请确保模型路径正确）
    engine_7B = vllm.LLM(model="AI_School_main_vllm/quantize_model/qwen2__5_7B")
    # 初始化分词器
    tokenizer = AutoTokenizer.from_pretrained(
        "AI_School_main_vllm/quantize_model/qwen2__5_7B",
        trust_remote_code=True
    )
    print("LLM is ready")
    # 加载上轮对话数据
    file_path = "AI_School_main_vllm/agentclass/debate/openhermes/sbatch/output/first_turn.json"
    prompt_first_dialogs, first_dialogs, answers = load_math_questions(file_path,tokenizer)
    print("dialogs are ready")
    # 批量解题，获取历史记录
    print("Starting to solve code problems...")
    history = solve_math_problems(prompt_first_dialogs, first_dialogs, answers, engine_7B)
    
    # 保存历史记录到 JSON 文件
    save_to_json(history)

if __name__ == "__main__":
    print ("start")
    main() 