
import os
import time
import pandas as pd
from transformers import AutoTokenizer
import json
import vllm
from vllm import LLM, SamplingParams

def load_math_questions(file_path=None, tokenizer = None):
    """
    读取 JSON 文件，提取 'first_dialog' 和 'answer' 字段，并返回。
    """
    # 读取 JSON 文件
    with open(file_path, 'r') as f:
        data = json.load(f)
        teacher_message = \
            "You are a teacher responsible for guiding the student’s learning. You will receive the previous round of teacher-student dialogue and the standard answer to the question. Based on the following rules, generate your response:\n" + \
            "(1) When you receive the previous round of teacher-student dialogue, you need to correct the student’s answer based on the standard answer. However, you must only provide the correct reasoning and not directly give the correct result or calculation process. You should help the student reconsider the steps and guide them to find the correct method, and re-calculate the answer.\n"+\
            "(2) You must avoid providing or hinting at any irrelevant information. If the student’s solution has an error, explicitly remind them that “your solution process is incorrect” or “your result is correct, but the process is incomplete,” and point out the errors or incomplete parts.\n"+\
            "(3) You are limited to playing the teacher agent role and should focus solely on providing the question paraphrasing and guiding the student to correct the knowledge errors. Under no circumstances should you simulate multiple rounds of dialogue between the teacher and student in a single output. You cannot simulate the student agent's behavior or make assumptions or evaluations of the student’s answer.\n"+\
            "(4) You must respond in English."
    # 提取 'first_dialog' 和 'answer' 字段
    first_dialogs = [item.get('first_dialog') for item in data[:5000]]  # 取前 5000 个
    answers = [item.get('answer') for item in data[:5000]]  # 取前 5000 个
    
    # 将 teacher_message 拼接到每个 first_dialog 的前面
    modified_first_dialogs = []
    for dialog in first_dialogs:
        messages = [
            {"role": "system", "content": teacher_message},
            {"role": "user", "content": dialog}
        ]
        # 使用 tokenizer 格式化
        formatted_question = format_prompt(messages, tokenizer)
        modified_first_dialogs.append(formatted_question)
       
    return modified_first_dialogs, first_dialogs, answers

def solve_math_problems(modified_dialogs, dialogs, answers, llm, batch_size=50):
    """
    以批次处理数学题目，调用模型生成解答，并返回题目、答案和历史记录的对。
    """
    history = []
    total_dialogs = len(dialogs)
    sampling_params = vllm.SamplingParams(temperature=0.2, top_p=0.95, max_tokens = 4096)
    for i in range(0, total_dialogs, batch_size):
        # 提取当前批次的第一轮对话
        batch_dialogs = dialogs[i:i + batch_size]
        batch_answers = answers[i:i + batch_size]
        batch_modified_dialogs = modified_dialogs[i:i + batch_size]
        # 为模型准备输入的批次（拼接问题文本）
        prompts = batch_modified_dialogs  # 直接使用问题批次
        
        # 生成每个批次的解答
        batch_responses = llm.generate(prompts,sampling_params)
        
        # 处理每个问题的模型反馈并保存历史记录
        for idx, (dialog, correct_answer, response) in enumerate(zip(batch_dialogs, batch_answers, batch_responses)):
            dialog_history = {
                "first_dialog": dialog ,
                "teacher_correct": "<Teacher>" + response.outputs[0].text,
                "answer": correct_answer
            }
            history.append(dialog_history)
            
            # 输出每个问题的处理进度
            print(f"Processing question {i + idx + 1}/{total_dialogs}...")

        # 显示当前批次的处理进度
        print(f"Batch {i // batch_size + 1} processed, total {len(history)} dialog histories collected.")
        
    return history

def save_to_json(data, file_name="./output/second_turn.json"):
    """
    将对话历史和答案保存为 JSON 文件
    """
    # 确保目录存在
    os.makedirs(os.path.dirname(file_name), exist_ok=True)
    
    with open(file_name, 'w') as f:
        json.dump(data, f, indent=4)
    print(f"Data saved to {file_name}.")
def format_prompt(messages, tokenizer):
    """将消息列表转换为合规的文本输入"""
    try:
        # 使用分词器的聊天模板格式化
        text = tokenizer.apply_chat_template(
            messages,
            tokenize=False,
            add_generation_prompt=True
        )
        return text
    except Exception as e:
        print(f"格式化提示词时出错: {e}")
        # 回退到手动格式
        return "\n".join([f"{m['role']}: {m['content']}" for m in messages])
    
def main():
# 创建采样参数对象
    
    print("start......")
    # 加载本地模型（请确保模型路径正确）
    engine_14B = vllm.LLM(model="AI_School_main_vllm/quantize_model/qwen2__5_14B")
    # 初始化分词器
    tokenizer = AutoTokenizer.from_pretrained(
        "AI_School_main_vllm/quantize_model/qwen2__5_14B",
        trust_remote_code=True
    )
    print("LLM is ready")
    # 加载上轮对话数据
    file_path = "AI_School_main_vllm/agentclass/make_error/code/sbatch/output/first_turn.json"
    modified_dialogs,dialogs,answers = load_math_questions(file_path, tokenizer)
    print("dialogs are ready")
    # 批量解题，获取历史记录
    print("Starting to solve code problems...")
    history = solve_math_problems(modified_dialogs, dialogs, answers, engine_14B)
    
    # 保存历史记录到 JSON 文件
    save_to_json(history)

if __name__ == "__main__":
    print ("start")
    main() 