import json
import re

def extract_answers_from_json(answers_file_path, student_field):
    # 读取答案 JSON 文件
    with open(answers_file_path, 'r') as file:
        data = json.load(file)
    
    # 定义正则表达式，匹配左括号、左方括号或左花括号后跟字母 A 到 Z（不区分大小写）
    pattern = re.compile(r'[\(\[\{]\s*(\d{1,2}|[A-Z])', re.IGNORECASE)
    
    extracted_answers = []
    none_count = 0  # 计数 None 的数量
    
    for entry in data:
        answer = entry.get(student_field, '')  # 获取指定学生字段的答案
        
        # 查找所有匹配的字符串
        matches = pattern.findall(answer)
        
        if matches:
            # 如果找到了匹配项，取最后一个匹配项并转换为大写
            last_match = matches[-1].strip()
            if last_match.isdigit():
                num = int(last_match)
                if 1 <= num <= 26:
                    # 将数字转换为对应的大写字母
                    extracted_answers.append(chr(num + 64))
                else:
                    extracted_answers.append(None)
                    none_count += 1
            else:
                extracted_answers.append(last_match.upper())
        else:
            extracted_answers.append(None)  # 没有匹配到时返回 None
            none_count += 1  # 增加 None 计数
    
    return extracted_answers, none_count

def get_correct_answers(correct_file_path):
    """
    读取JSONL格式的正确答案文件，提取每行的Answer字段并转换为大写
    
    参数:
        correct_file_path: JSONL文件路径，每行格式为{"A":"...", "B":"...", "Answer":"..."}
    
    返回:
        List[str]: 包含所有Answer字段值（大写）的列表
    """
    correct_answers = []
    
    with open(correct_file_path, 'r', encoding='utf-8') as file:
        for line in file:
            try:
                entry = json.loads(line.strip())  # 解析单行JSON[3,5](@ref)
                answer = entry.get('Answer', '')  # 获取Answer字段[1,4](@ref)
                correct_answers.append(answer.upper())  # 转换为大写[6](@ref)
            except json.JSONDecodeError:
                continue  # 跳过无效行[2](@ref)
    
    return correct_answers

def calculate_accuracy(extracted_answers, correct_answers):
    # 计算准确率
    total = len(extracted_answers)
    correct_count = 0
    
    for extracted, correct in zip(extracted_answers, correct_answers):
        if extracted and extracted == correct[-1].upper():  # 统一转换为大写进行比较
            correct_count += 1
    
    accuracy = correct_count / total if total > 0 else 0
    return accuracy

def compare_student_answers(studentA_answers, studentC_answers, correct_answers):
    A_correct_C_wrong = 0
    A_wrong_C_correct = 0

    for a_ans, c_ans, correct in zip(studentA_answers, studentC_answers, correct_answers):
        a_correct = a_ans and a_ans == correct[-1].upper()
        c_correct = c_ans and c_ans == correct[-1].upper()

        if a_correct and not c_correct:
            A_correct_C_wrong += 1
        elif not a_correct and c_correct:
            A_wrong_C_correct += 1

    return A_correct_C_wrong, A_wrong_C_correct

# 调用函数并打印提取的结果、准确率和 None 的数量
answers_file_path = "LLAMA3_MATHQA_EVAL.json"  # 替换为你的答案 JSON 文件路径
correct_file_path = "AI_School_main_vllm/data/eval_datasets/math-mc/test.jsonl"  # 替换为你的正确答案 JSON 文件路径

# 提取答案和正确答案
studentA_answers, none_count_A = extract_answers_from_json(answers_file_path, '<StudentA>')
studentC_answers, none_count_C = extract_answers_from_json(answers_file_path, '<StudentC>')
correct_answers = get_correct_answers(correct_file_path)

# 计算准确率
accuracy_A = calculate_accuracy(studentA_answers, correct_answers)
accuracy_C = calculate_accuracy(studentC_answers, correct_answers)

# 比较两位学生的答案
A_correct_C_wrong, A_wrong_C_correct = compare_student_answers(studentA_answers, studentC_answers, correct_answers)

print(f"Accuracy of StudentA: {accuracy_A * 100:.2f}%")
print(f"Accuracy of StudentC: {accuracy_C * 100:.2f}%")
print(f"Number of None entries for StudentA: {none_count_A}")
print(f"Number of None entries for StudentC: {none_count_C}")
print(f"Number of questions where StudentA is correct and StudentC is wrong: {A_correct_C_wrong}")
print(f"Number of questions where StudentA is wrong and StudentC is correct: {A_wrong_C_correct}")
