import json
import re
from datasets import load_dataset, concatenate_datasets
def extract_answers_from_json(answers_file_path):
    # 读取答案 JSON 文件
    with open(answers_file_path, 'r') as file:
        data = json.load(file)
    
    # 定义正则表达式，匹配左括号、左方括号或左花括号后跟字母 A 到 Z（不区分大小写）
    pattern = re.compile(r'[\(\[\{]\s*(\d{1,2}|[A-Z])\s*[\)\]\}]|([A-Z])\s*\.', re.IGNORECASE)
    
    extracted_answers = []
    none_count = 0  # 计数 None 的数量
    
    for entry in data:
        answer = entry.get('answer', '')  # 获取 answer 字段
        
        # 查找所有匹配的字符串
        matches = pattern.findall(answer)
        
        if matches:
            # 获取最后一个匹配项，注意 matches[-1] 是个 tuple
            match = matches[-1]
            val = match[0] if match[0] else match[1]  # 选出有效的捕获内容

            if val and val.strip().isdigit():
                num = int(val.strip())
                if 1 <= num <= 26:
                    extracted_answers.append(chr(num + 64))  # 转换为大写字母
                else:
                    extracted_answers.append(None)
                    none_count += 1
            else:
                extracted_answers.append(val.strip().upper())
        else:
            extracted_answers.append(None)
            none_count += 1
    
    return extracted_answers, none_count

def get_correct_answers(correct_file_path):
    # 读取正确答案 JSON 文件
    subcategories = [
        'abstract_algebra',
        'anatomy',
        'astronomy',
        'business_ethics',
        'clinical_knowledge',
        'college_biology',
        'college_chemistry',
        'college_computer_science',
        'college_mathematics',
        'college_medicine',
        'college_physics',
        'computer_security',
        'conceptual_physics',
        'econometrics',
        'electrical_engineering',
        'elementary_mathematics',
        'formal_logic',
        'global_facts',
        'high_school_biology',
        'high_school_chemistry',
        'high_school_computer_science',
        'high_school_european_history',
        'high_school_geography',
        'high_school_government_and_politics',
        'high_school_macroeconomics',
        'high_school_mathematics',
        'high_school_microeconomics',
        'high_school_physics',
        'high_school_psychology',
        'high_school_statistics',
        'high_school_us_history',
        'high_school_world_history',
        'human_aging',
        'human_sexuality',
        'international_law',
        'jurisprudence',
        'logical_fallacies',
        'machine_learning',
        'management',
        'marketing',
        'medical_genetics',
        'miscellaneous',
        'moral_disputes',
        'moral_scenarios',
        'nutrition',
        'philosophy',
        'prehistory',
        'professional_accounting',
        'professional_law',
        'professional_medicine',
        'professional_psychology',
        'public_relations',
        'security_studies',
        'sociology',
        'us_foreign_policy',
        'virology',
        'world_religions'
    ]

    datasets_list = []
    for subcat in subcategories:
        try:
            dataset = load_dataset(
                path=correct_file_path,
                name=subcat,
                split='test'
            )
            datasets_list.append(dataset)
        except Exception as e:
            print(f"加载子类别 {subcat} 时出错: {e}")

    if not datasets_list:
        print("未能加载任何子类别的测试集。请检查数据目录和子类别名称是否正确。")
        return []

    # 合并所有子类别的测试集
    data = concatenate_datasets(datasets_list)
    
    correct_answers = []
    
    for entry in data:
        correct = entry.get('answer', '')
        if isinstance(correct, int):
            if 0 <= correct <= 25:
                correct = chr(ord('A') + correct)
            else:
                print(f"Warning: 数值 {correct} 超出有效范围（0-25），无法转换为字母。")
        elif isinstance(correct, str):
            correct = correct.strip().upper()
        else:
            print(f"Warning: 无法识别的 answer 类型：{type(correct)}")
        correct_answers.append(correct)

    
    return correct_answers

def calculate_accuracy(extracted_answers, correct_answers):
    # 计算准确率
    total = len(extracted_answers)
    correct_count = 0
    
    for extracted, correct in zip(extracted_answers, correct_answers):
        if extracted and extracted == correct[-1].upper():  # 统一转换为大写进行比较
            correct_count += 1
    
    accuracy = correct_count / total if total > 0 else 0
    return accuracy


# 调用函数并打印提取的结果、准确率和 None 的数量
answers_file_path = "LLAMA3_MATHQA_EVAL.json"  # 替换为你的答案 JSON 文件路径
correct_file_path = "AI_School_main_vllm/data/eval_datasets/mmlu"  # 替换为你的正确答案 JSON 文件路径

# 提取答案和正确答案
extracted_answers, none_count = extract_answers_from_json(answers_file_path)
correct_answers = get_correct_answers(correct_file_path)

# 计算准确率
accuracy = calculate_accuracy(extracted_answers, correct_answers)

print(f"Accuracy: {accuracy * 100:.2f}%")
print(f"Number of None entries: {none_count}")
