import torch.nn as nn
import torch.nn.functional as F
from utils.builder import get_builder
from args import args as parser_args


class BasicBlock(nn.Module):
    expansion = 1

    def __init__(self, builder, in_planes, planes, stride=1):
        super(BasicBlock, self).__init__()
        self.conv1 = builder.conv3x3(in_planes, planes, stride=stride)
        self.bn1 = nn.BatchNorm2d(planes)
        self.conv2 = builder.conv3x3(planes, planes, stride=1)
        self.bn2 = nn.BatchNorm2d(planes)

        self.shortcut = nn.Sequential()
        if parser_args.conv_type == "ReinforceLOO" or parser_args.conv_type == "ReinforceLOOVR" or parser_args.conv_type == "Reinforce":
            self.shortcut = nn.Sequential(
                builder.conv1x1(in_planes, self.expansion * planes, stride=stride),
                nn.BatchNorm2d(self.expansion * planes),
            )
        else:
            if stride != 1 or in_planes != self.expansion * planes:
                self.shortcut = nn.Sequential(
                    builder.conv1x1(in_planes, self.expansion * planes, stride=stride),
                    nn.BatchNorm2d(self.expansion * planes),
                )

    def forward(self, x):
        out = F.relu(self.bn1(self.conv1(x)))
        out = self.bn2(self.conv2(out))
        out += self.shortcut(x)
        out = F.relu(out)
        return out


class ResNet(nn.Module):
    def __init__(self, block, num_blocks, num_classes):
        super(ResNet, self).__init__()
        self.builder = get_builder()
        _outputs = [32, 64, 128]
        self.in_planes = _outputs[0]
        self.conv1 = self.builder.conv3x3(3, 32, stride=1, first_layer=True)
        self.bn = nn.BatchNorm2d(_outputs[0])
        self.layer1 = self._make_layer(block, 32, num_blocks[0], stride=1)
        self.layer2 = self._make_layer(block, 64, num_blocks[1], stride=2)
        self.layer3 = self._make_layer(block, 128, num_blocks[2], stride=2)
        self.linear = self.builder.conv1x1(128, num_classes)
        self.wl = []
        for n, m in self.named_modules():
            if hasattr(m, "scores"):
                self.wl.append(m.weight)

    def weight_list(self):
        return self.wl

    def _make_layer(self, block, planes, num_blocks, stride):
        strides = [stride] + [1] * (num_blocks - 1)
        layers = []
        for stride in strides:
            layers.append(block(self.builder, self.in_planes, planes, stride))
            self.in_planes = planes * block.expansion

        return nn.Sequential(*layers)

    def forward(self, x):
        out = F.relu(self.bn(self.conv1(x)))
        out = self.layer1(out)
        out = self.layer2(out)
        out = self.layer3(out)
        out = F.avg_pool2d(out, out.size()[3])
        out = self.linear(out)
        out = out.flatten(1)
        return out

def resnet20():
    num_classes = 10 if parser_args.set == "CIFAR10" else 100
    return ResNet(BasicBlock, [3, 3, 3], num_classes)

def resnet32():
    num_classes = 10 if parser_args.set == "CIFAR10" else 100
    return ResNet(BasicBlock, [5, 5, 5], num_classes)