"""
Pure Python implementation of the Rudy Graph Generator.
This implements the same algorithms as the C code without any C dependencies.
All functions return graphs as symmetric adjacency matrices.
Includes binary and unitary operations from the original rudy_lib.c.
"""
import random
import numpy as np
from typing import Optional, Tuple


def create_planar_graph(size: int, density: float, seed: int) -> np.ndarray:
    """Create a planar graph using the triangulation algorithm.
    
    Args:
        size: Number of nodes
        density: Percentage of edges to keep (0-100)
        seed: Random seed
        
    Returns:
        A symmetric adjacency matrix representing the graph
    """
    if size <= 0:
        raise ValueError("Size must be positive")
    
    random.seed(seed)
    
    # Initialize adjacency matrix with zeros
    adj_matrix = np.zeros((size, size), dtype=np.int32)
    
    # Special cases for small graphs
    if size == 1:
        # No edges for a single node
        return adj_matrix
    
    if size == 2:
        # Just one edge for two nodes
        adj_matrix[0, 1] = adj_matrix[1, 0] = 1
        return adj_matrix
    
    # Initialize with a triangle (first 3 vertices)
    edges = [(0, 1), (0, 2), (1, 2)]
    
    # List of faces (triangles) - each face is (i, j, k) indices
    faces = [(0, 1, 2)]
    
    # Add vertices one by one, connecting to a random face
    for new_vertex in range(3, size):
        # Choose a random face
        face_idx = random.randrange(len(faces))
        a, b, c = faces[face_idx]
        
        # Connect new vertex to all corners of the face
        edges.append((a, new_vertex))
        edges.append((b, new_vertex))
        edges.append((c, new_vertex))
        
        # Update faces: remove chosen face, add three new faces
        del faces[face_idx]
        faces.append((a, b, new_vertex))
        faces.append((b, c, new_vertex))
        faces.append((c, a, new_vertex))
    
    # Remove edges randomly based on density
    max_edges = 3 * (size - 2)  # Maximum number of edges in a planar graph
    target_edges = int((density * max_edges) / 100.0)
    
    # Shuffle edges and take the first 'target_edges' of them
    random.shuffle(edges)
    edges = edges[:target_edges]
    
    # Fill adjacency matrix with 1s for edges
    for src, tgt in edges:
        adj_matrix[src, tgt] = adj_matrix[tgt, src] = 1
    
    return adj_matrix


def create_circuit_graph(length: int) -> np.ndarray:
    """Create a circuit graph (cycle) of specified length.
    
    Args:
        length: Number of nodes in the circuit
        
    Returns:
        A symmetric adjacency matrix representing the graph
    """
    if length <= 0:
        raise ValueError("Length must be positive")
    
    # Initialize adjacency matrix with zeros
    adj_matrix = np.zeros((length, length), dtype=np.int32)
    
    # Add edges to form a circuit
    for i in range(length):
        next_node = (i + 1) % length
        adj_matrix[i, next_node] = adj_matrix[next_node, i] = 1
    
    return adj_matrix


def create_clique_graph(size: int) -> np.ndarray:
    """Create a complete graph (clique) of specified size.
    
    Args:
        size: Number of nodes
        
    Returns:
        A symmetric adjacency matrix representing the graph
    """
    if size <= 0:
        raise ValueError("Size must be positive")
    
    # Initialize adjacency matrix with zeros
    adj_matrix = np.zeros((size, size), dtype=np.int32)
    
    # Fill in all edges with weights |i-j|
    for i in range(size):
        for j in range(i+1, size):
            weight = abs(i - j)
            adj_matrix[i, j] = adj_matrix[j, i] = weight
    
    return adj_matrix


def create_random_graph(size: int, density: float, seed: int) -> np.ndarray:
    """Create a random graph with given size and density.
    
    Args:
        size: Number of nodes
        density: Percentage of density (0-100)
        seed: Random seed
        
    Returns:
        A symmetric adjacency matrix representing the graph
    """
    if size <= 0:
        raise ValueError("Size must be positive")
    #print("seed", seed)
    random.seed(seed)
    
    # Initialize adjacency matrix with zeros
    adj_matrix = np.zeros((size, size), dtype=np.int32)
    
    # Calculate number of edges based on density
    max_edges = (size * (size - 1)) // 2
    n_edges = int((max_edges * density) / 100.0 + 0.5)  # Round to nearest integer
    n_edges = min(n_edges, max_edges)  # Ensure we don't exceed max_edges
    
    # Generate all possible edges
    all_edges = [(i, j) for i in range(size) for j in range(i + 1, size)]
    
    # Randomly select n_edges from all possible edges
    selected_edges = random.sample(all_edges, n_edges)
    
    # Fill in adjacency matrix
    for src, tgt in selected_edges:
        adj_matrix[src, tgt] = adj_matrix[tgt, src] = 1
    
    return adj_matrix


def create_grid_2d(height: int, width: int) -> np.ndarray:
    """Create a 2D grid graph.
    
    Args:
        height: Number of rows
        width: Number of columns
        
    Returns:
        A symmetric adjacency matrix representing the graph
    """
    if height <= 0 or width <= 0:
        raise ValueError("Height and width must be positive")
    
    size = height * width
    adj_matrix = np.zeros((size, size), dtype=np.int32)
    
    # Add horizontal edges
    for i in range(height):
        for j in range(width - 1):
            node1 = i * width + j
            node2 = i * width + j + 1
            adj_matrix[node1, node2] = adj_matrix[node2, node1] = 1
    
    # Add vertical edges
    for i in range(height - 1):
        for j in range(width):
            node1 = i * width + j
            node2 = (i + 1) * width + j
            adj_matrix[node1, node2] = adj_matrix[node2, node1] = 1
    
    return adj_matrix


def create_toroidal_grid_2d(height: int, width: int) -> np.ndarray:
    """Create a 2D grid graph with toroidal (periodic) boundary conditions.
    
    Args:
        height: Number of rows
        width: Number of columns
        
    Returns:
        A symmetric adjacency matrix representing the graph
    """
    if height <= 0 or width <= 0:
        raise ValueError("Height and width must be positive")
    
    size = height * width
    adj_matrix = np.zeros((size, size), dtype=np.int32)
    
    # Add horizontal edges (including wrap-around)
    for i in range(height):
        for j in range(width):
            node1 = i * width + j
            node2 = i * width + (j + 1) % width
            adj_matrix[node1, node2] = adj_matrix[node2, node1] = 1
    
    # Add vertical edges (including wrap-around)
    for i in range(height):
        for j in range(width):
            node1 = i * width + j
            node2 = ((i + 1) % height) * width + j
            adj_matrix[node1, node2] = adj_matrix[node2, node1] = 1
    
    return adj_matrix


# ------------------- BINARY OPERATORS -------------------

def graph_union(g1: np.ndarray, g2: np.ndarray) -> np.ndarray:
    """Combine two graphs with the same number of nodes by taking the union of their edges.
    This is the '+' operator in rudy_lib.c.
    
    Args:
        g1: Adjacency matrix of first graph
        g2: Adjacency matrix of second graph
        
    Returns:
        Adjacency matrix of the combined graph
    """
    if g1.shape != g2.shape:
        raise ValueError("Graphs must have the same number of nodes")
    
    # Take the maximum weight for each edge
    return np.maximum(g1, g2)


def cartesian_product(g1: np.ndarray, g2: np.ndarray) -> np.ndarray:
    """Create the Cartesian product of two graphs.
    This is the 'x' operator in rudy_lib.c.
    
    For each node pair (u1,u2) where u1 is from g1 and u2 is from g2:
    - Connect to (v1,u2) if u1 is connected to v1 in g1
    - Connect to (u1,v2) if u2 is connected to v2 in g2
    
    Args:
        g1: Adjacency matrix of first graph
        g2: Adjacency matrix of second graph
        
    Returns:
        Adjacency matrix of the product graph
    """
    n1 = g1.shape[0]
    n2 = g2.shape[0]
    n = n1 * n2  # Total nodes in product graph
    
    # Initialize the product adjacency matrix
    product_adj = np.zeros((n, n), dtype=np.int32)
    
    # For each node (u1,u2) in the product graph
    for u1 in range(n1):
        for u2 in range(n2):
            u = u1 * n2 + u2  # Node index in product graph
            
            # Add edges from g1 (connect to (v1,u2) for all v1 adjacent to u1)
            for v1 in range(n1):
                if g1[u1, v1] > 0:
                    v = v1 * n2 + u2
                    product_adj[u, v] = product_adj[v, u] = g1[u1, v1]
            
            # Add edges from g2 (connect to (u1,v2) for all v2 adjacent to u2)
            for v2 in range(n2):
                if g2[u2, v2] > 0:
                    v = u1 * n2 + v2
                    product_adj[u, v] = product_adj[v, u] = g2[u2, v2]
    
    return product_adj


def complete_bipartite(g1: np.ndarray, g2: np.ndarray) -> np.ndarray:
    """Create a graph where nodes from g1 and g2 form a complete bipartite graph.
    This is the ':' operator in rudy_lib.c.
    
    Args:
        g1: Adjacency matrix of first graph
        g2: Adjacency matrix of second graph
        
    Returns:
        Adjacency matrix of the resulting graph
    """
    n1 = g1.shape[0]
    n2 = g2.shape[0]
    n = n1 + n2  # Total nodes
    
    # Initialize the combined adjacency matrix
    result_adj = np.zeros((n, n), dtype=np.int32)
    
    # Copy g1 to the top-left corner
    result_adj[:n1, :n1] = g1
    
    # Copy g2 to the bottom-right corner
    result_adj[n1:, n1:] = g2
    
    # Add edges between all nodes of g1 and g2 (complete bipartite)
    for i in range(n1):
        for j in range(n2):
            result_adj[i, n1 + j] = result_adj[n1 + j, i] = 1
    
    return result_adj


# ------------------- UNARY OPERATORS -------------------

def graph_complement(g: np.ndarray) -> np.ndarray:
    """Create the complement of a graph.
    This is the '-complement' operator in rudy_lib.c.
    
    Args:
        g: Adjacency matrix of the graph
        
    Returns:
        Adjacency matrix of the complement graph
    """
    n = g.shape[0]
    
    # Start with a matrix of all 1s (except diagonal)
    complement_adj = np.ones((n, n), dtype=np.int32)
    np.fill_diagonal(complement_adj, 0)  # No self-loops
    
    # Remove edges that exist in the original graph
    for i in range(n):
        for j in range(n):
            if g[i, j] > 0:
                complement_adj[i, j] = 0
    
    return complement_adj


def randomize_weights(g: np.ndarray, lower_weight: int, upper_weight: int, seed: int) -> np.ndarray:
    """Assign random weights to the edges of a graph.
    This is the '-random' operator in rudy_lib.c.
    
    Args:
        g: Adjacency matrix of the graph
        lower_weight: Minimum weight
        upper_weight: Maximum weight
        seed: Random seed
        
    Returns:
        Adjacency matrix with randomized weights
    """
    random.seed(seed)
    n = g.shape[0]
    result = g.copy()
    
    for i in range(n):
        for j in range(i+1, n):  # Only upper triangle due to symmetry
            if g[i, j] > 0:
                # Assign random weight in [lower_weight, upper_weight]
                weight = random.randint(lower_weight, upper_weight)
                result[i, j] = result[j, i] = weight
    
    return result


def multiply_weights(g: np.ndarray, scalar: int) -> np.ndarray:
    """Multiply all edge weights by a scalar.
    This is the '-times' operator in rudy_lib.c.
    
    Args:
        g: Adjacency matrix of the graph
        scalar: Factor to multiply weights by
        
    Returns:
        Adjacency matrix with scaled weights
    """
    return g * scalar


def add_to_weights(g: np.ndarray, scalar: int) -> np.ndarray:
    """Add a scalar to all edge weights.
    This is the '-plus' operator in rudy_lib.c.
    
    Args:
        g: Adjacency matrix of the graph
        scalar: Value to add to weights
        
    Returns:
        Adjacency matrix with increased weights
    """
    result = g.copy()
    n = g.shape[0]
    
    for i in range(n):
        for j in range(i+1, n):  # Only upper triangle due to symmetry
            if g[i, j] > 0:
                result[i, j] = result[j, i] = g[i, j] + scalar
    
    return result


def line_graph(g: np.ndarray) -> np.ndarray:
    """Create the line graph of a graph.
    This is the '-line' operator in rudy_lib.c.
    
    In a line graph, each edge of the original graph becomes a node,
    and two nodes are adjacent if the corresponding edges in the original
    graph share a vertex.
    
    Args:
        g: Adjacency matrix of the graph
        
    Returns:
        Adjacency matrix of the line graph
    """
    n = g.shape[0]
    
    # Extract edges from adjacency matrix
    edges = []
    for i in range(n):
        for j in range(i+1, n):
            if g[i, j] > 0:
                edges.append((i, j))
    
    num_edges = len(edges)
    if num_edges == 0:
        return np.zeros((0, 0), dtype=np.int32)  # Empty graph
    
    # Create adjacency matrix for line graph
    line_adj = np.zeros((num_edges, num_edges), dtype=np.int32)
    
    # Two edges are adjacent in the line graph if they share a vertex in the original graph
    for i in range(num_edges):
        u1, v1 = edges[i]
        for j in range(i+1, num_edges):
            u2, v2 = edges[j]
            # Check if edges share a vertex
            if u1 == u2 or u1 == v2 or v1 == u2 or v1 == v2:
                line_adj[i, j] = line_adj[j, i] = 1
    
    return line_adj


# Helper function to display graph information
def print_graph_info(name: str, adj_matrix: np.ndarray):
    """Print information about a graph from its adjacency matrix."""
    n = adj_matrix.shape[0]
    edges = []
    
    # Extract edges from adjacency matrix
    for i in range(n):
        for j in range(i+1, n):
            if adj_matrix[i, j] > 0:
                edges.append((i, j, adj_matrix[i, j]))
    
    print(f"{name}: {n} nodes, {len(edges)} edges")
    for idx, (i, j, w) in enumerate(edges):
        print(f"Edge {idx+1}: {i+1} -> {j+1} (weight: {w})")


# Example usage
if __name__ == "__main__":
    # Create a random planar graph
    planar_adj = create_planar_graph(10, 50.0, 42)
    print(planar_adj)
    print_graph_info("Planar Graph", planar_adj)
    
    # Create a circuit
    circuit_adj = create_circuit_graph(5)
    print("\n")
    print_graph_info("Circuit", circuit_adj)
    
    # Create a clique
    clique_adj = create_clique_graph(4)
    print("\n")
    print_graph_info("Clique", clique_adj)
    
    # Create a random graph
    random_adj = create_random_graph(8, 30.0, 42)
    print("\n")
    print_graph_info("Random Graph", random_adj)
    
    # Create a 2D grid
    grid_adj = create_grid_2d(3, 4)
    print("\n")
    print_graph_info("2D Grid", grid_adj)
    
    # Create a toroidal 2D grid
    toroidal_adj = create_toroidal_grid_2d(3, 3)
    print("\n")
    print_graph_info("Toroidal 2D Grid", toroidal_adj)
    
    # Demonstrate binary operators
    print("\n====== BINARY OPERATORS EXAMPLES ======")
    
    # Graph union ('+' operator)
    small_circuit = create_circuit_graph(4)
    small_grid = create_grid_2d(2, 2)
    union_result = graph_union(small_circuit, small_grid)
    print("\nGraph Union Example:")
    print_graph_info("Circuit + Grid", union_result)
    
    # Cartesian product ('x' operator)
    path3 = create_circuit_graph(3)  # Path with 3 nodes
    path3[0, 2] = path3[2, 0] = 0  # Remove edge to make it a path not a cycle
    path2 = np.zeros((2, 2), dtype=np.int32)
    path2[0, 1] = path2[1, 0] = 1
    product_result = cartesian_product(path2, path3)
    print("\nCartesian Product Example:")
    print_graph_info("P2 × P3", product_result)
    
    # Complete bipartite (':' operator)
    small_clique1 = create_clique_graph(3)
    small_clique2 = create_clique_graph(2)
    bipartite_result = complete_bipartite(small_clique1, small_clique2)
    print("\nComplete Bipartite Example:")
    print_graph_info("K3:K2", bipartite_result)
    
    # Demonstrate unary operators
    print("\n====== UNARY OPERATORS EXAMPLES ======")
    
    # Graph complement
    comp_result = graph_complement(path3)
    print("\nGraph Complement Example:")
    print_graph_info("Complement of P3", comp_result)
    
    # Randomize weights
    rand_weights = randomize_weights(small_grid, 1, 10, 123)
    print("\nRandomized Weights Example:")
    print_graph_info("Grid with random weights", rand_weights)
    
    # Multiply weights
    scaled_weights = multiply_weights(rand_weights, 2)
    print("\nMultiplied Weights Example:")
    print_graph_info("Weights multiplied by 2", scaled_weights)
    
    # Add to weights
    added_weights = add_to_weights(rand_weights, 5)
    print("\nAdded to Weights Example:")
    print_graph_info("Weights with 5 added", added_weights)
    
    # Line graph
    line_graph_result = line_graph(path3)
    print("\nLine Graph Example:")
    print_graph_info("Line graph of P3", line_graph_result) 


    g = graph_union(create_planar_graph(100, 50.0, 42), create_planar_graph(100, 50.0, 43))
    print(g)

