from collections import defaultdict

def parse_hierarchy(file_path):
    """Parse the hypernyms.txt file to build a parent-to-children mapping."""
    parent_to_children = defaultdict(set)
    with open(file_path, 'r') as f:
        for line in f:
            child, parent = line.strip().split(', ')
            parent_to_children[parent].add(child)
    return parent_to_children

def parse_objects(file_path):
    """Parse the object.txt file to get the set of concepts."""
    with open(file_path, 'r') as f:
        return set(line.strip() for line in f)

def find_pairs(parent_to_children, concepts):
    """Find overlap and exclusive pairs based on shared direct parents."""
    overlap_pairs = set()
    exclusive_pairs = set()

    for parent, children in parent_to_children.items():
        # Filter children to only include those in the concepts set
        valid_children = children & concepts
        valid_children = list(valid_children)

        # Compare all pairs of children under the same parent
        for i in range(len(valid_children)):
            for j in range(i + 1, len(valid_children)):
                a, b = valid_children[i], valid_children[j]
                subclasses_a = parent_to_children.get(a, set())
                subclasses_b = parent_to_children.get(b, set())
                
                # Check if the intersection of subclasses is empty
                if subclasses_a & subclasses_b:
                    overlap_pairs.add((a, b))
                else:
                    exclusive_pairs.add((a, b))
    
    return overlap_pairs, exclusive_pairs

def write_pairs(file_path, pairs):
    """Write pairs to a file."""
    with open(file_path, 'w') as f:
        for a, b in sorted(pairs):
            f.write(f"{a}, {b}\n")

def main():
    hypernyms_file = "/Users/boxiong/projects/concept_analysis/datasets/animal/hypernyms.txt"
    objects_file = "/Users/boxiong/projects/concept_analysis/datasets/animal/object.txt"
    overlap_file = "/Users/boxiong/projects/concept_analysis/datasets/animal/overlap.txt"
    exclusive_file = "/Users/boxiong/projects/concept_analysis/datasets/animal/exclusive.txt"

    # Parse input files
    parent_to_children = parse_hierarchy(hypernyms_file)
    concepts = parse_objects(objects_file)

    # Find overlap and exclusive pairs
    overlap_pairs, exclusive_pairs = find_pairs(parent_to_children, concepts)

    # Write results to files
    write_pairs(overlap_file, overlap_pairs)
    write_pairs(exclusive_file, exclusive_pairs)

if __name__ == "__main__":
    main()
