# Copyright 2024 The HuggingFace Team. All rights reserved.
#
# Licensed under the Apache License, Version 2.0 (the "License");
# you may not use this file except in compliance with the License.
# You may obtain a copy of the License at
#
#     http://www.apache.org/licenses/LICENSE-2.0
#
# Unless required by applicable law or agreed to in writing, software
# distributed under the License is distributed on an "AS IS" BASIS,
# WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
# See the License for the specific language governing permissions and
# limitations under the License.


from typing import List, Optional, Tuple, Union

import torch

from ...utils.torch_utils import randn_tensor
from ..pipeline_utils import DiffusionPipeline, ImagePipelineOutput

import torchvision.transforms as transforms

transform_normalize = transforms.Compose([
                transforms.Normalize(mean=[0.5, 0.5, 0.5], std=[0.5, 0.5, 0.5]),
            ])

def take_every_m_element(tensor, m):
    indices = torch.arange(0, len(tensor), m)
    last_index = tensor.size(0) - 1
    indices = torch.cat((indices, torch.tensor([last_index])))
    return tensor[indices]

class Guided_DDPMPipeline(DiffusionPipeline):
    model_cpu_offload_seq = "unet"

    def __init__(self, unet, scheduler):
        super().__init__()
        self.register_modules(unet=unet, scheduler=scheduler)

    #TODO： Get grad of this function, remove the torch.no_grad()
    # @torch.no_grad()
    def __call__(
        self,
        input_noise_image: Optional[torch.tensor] = None,
        input_noise_step: Optional[int] = None,
        noise_step_state: Optional[List[int]] = 1,
        batch_size: int = 1,
        generator: Optional[Union[torch.Generator, List[torch.Generator]]] = None,
        num_inference_steps: int = 1000,
        output_type: Optional[str] = "pil",
        return_dict: bool = True,
    ) -> Union[ImagePipelineOutput, Tuple]:
        torch_device = 'cuda' if torch.cuda.is_available() else 'cpu'
        self.unet.to(torch_device)
        if isinstance(self.unet.config.sample_size, int):
            image_shape = (
                batch_size,
                self.unet.config.in_channels,
                self.unet.config.sample_size,
                self.unet.config.sample_size,
            )
        else:
            image_shape = (batch_size, self.unet.config.in_channels, *self.unet.config.sample_size)

        if input_noise_image == None:
            if self.device.type == "mps":
                # randn does not work reproducibly on mps
                image = randn_tensor(image_shape, generator=generator)
                image = image.to(self.device)
            else:
                image = randn_tensor(image_shape, generator=generator, device=self.device)
        else:
            image = input_noise_image.to(torch_device)

        # set step values
        self.scheduler.set_timesteps(num_inference_steps)

        if input_noise_step != None:
            self.sample_times = self.scheduler.timesteps[-input_noise_step:]
            if noise_step_state != 1:
                self.sample_times = take_every_m_element(self.sample_times, noise_step_state)
        else:
            self.sample_times = self.scheduler.timesteps
            if noise_step_state != 1:
                self.sample_times = take_every_m_element(self.sample_times, noise_step_state)
        
        # for t in self.progress_bar(self.scheduler.timesteps):
        for t in self.sample_times:
            # 1. predict noise model_output
            model_output = self.unet(image, t).sample

            # 2. compute previous image: x_t -> x_t-1
            image = self.scheduler.step(model_output, t, image, generator=generator).prev_sample

        image = (image / 2 + 0.5).clamp(0, 1)
        if input_noise_image != None:
            return image