# coding=utf-8
# Copyright 2020 The Google Research Authors.
#
# Licensed under the Apache License, Version 2.0 (the "License");
# you may not use this file except in compliance with the License.
# You may obtain a copy of the License at
#
#     http://www.apache.org/licenses/LICENSE-2.0
#
# Unless required by applicable law or agreed to in writing, software
# distributed under the License is distributed on an "AS IS" BASIS,
# WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
# See the License for the specific language governing permissions and
# limitations under the License.

# pylint: skip-file
# pytype: skip-file
"""Various sampling methods."""
import functools

import torch
import numpy as np
import abc
import logging
import utils
from models.utils import get_score_fn
from scipy import integrate
import sde_lib
from models import utils as mutils

_CORRECTORS = {}
_PREDICTORS = {}


def register_predictor(cls=None, *, name=None):
  """A decorator for registering predictor classes."""

  def _register(cls):
    if name is None:
      local_name = cls.__name__
    else:
      local_name = name
    if local_name in _PREDICTORS:
      raise ValueError(f'Already registered model with name: {local_name}')
    _PREDICTORS[local_name] = cls
    return cls

  if cls is None:
    return _register
  else:
    return _register(cls)


def register_corrector(cls=None, *, name=None):
  """A decorator for registering corrector classes."""

  def _register(cls):
    if name is None:
      local_name = cls.__name__
    else:
      local_name = name
    if local_name in _CORRECTORS:
      raise ValueError(f'Already registered model with name: {local_name}')
    _CORRECTORS[local_name] = cls
    return cls

  if cls is None:
    return _register
  else:
    return _register(cls)


def get_predictor(name):
  return _PREDICTORS[name]


def get_corrector(name):
  return _CORRECTORS[name]


def get_sampling_fn(config, sde, eps, ode_sampling=False):

  predictor = get_predictor(config.sampling.predictor.lower())
  corrector = get_corrector(config.sampling.corrector.lower())
  if ode_sampling:
    sampling_fn = get_ode_sampler(sde=sde, denoise=False, rtol=1e-5, atol=1e-5, method='RK45', eps=eps, device=config.device)
  else:
    sampling_fn = get_pc_sampler(sde=sde, predictor=predictor, corrector=corrector,
                                  snr=config.sampling.snr, n_steps=config.sampling.n_steps_each, eps=eps, device=config.device)

  return sampling_fn


class Predictor(abc.ABC):
  """The abstract class for a predictor algorithm."""

  def __init__(self, sde, score_fn):
    super().__init__()
    self.sde = sde
    self.rsde = sde.reverse(score_fn)
    self.score_fn = score_fn

  @abc.abstractmethod
  def update_fn(self, x, t):
    """One update of the predictor.

    Args:
      x: A PyTorch tensor representing the current state
      t: A Pytorch tensor representing the current time step.

    Returns:
      x: A PyTorch tensor of the next state.
      x_mean: A PyTorch tensor. The next state without random noise. Useful for denoising.
    """
    pass

  @abc.abstractmethod
  def conditional_update_fn(self, h, x, t):
    """One update of the corrector.

    Args:
      x: A PyTorch tensor representing the current state
      t: A PyTorch tensor representing the current time step.

    Returns:
      x: A PyTorch tensor of the next state.
      x_mean: A PyTorch tensor. The next state without random noise. Useful for denoising.
    """
    pass


class Corrector(abc.ABC):
  """The abstract class for a corrector algorithm."""

  def __init__(self, sde, score_fn, snr, n_steps):
    super().__init__()
    self.sde = sde
    self.score_fn = score_fn
    self.snr = snr
    self.n_steps = n_steps

  @abc.abstractmethod
  def update_fn(self, x, t):
    """One update of the corrector.

    Args:
      x: A PyTorch tensor representing the current state
      t: A PyTorch tensor representing the current time step.

    Returns:
      x: A PyTorch tensor of the next state.
      x_mean: A PyTorch tensor. The next state without random noise. Useful for denoising.
    """
    pass

  @abc.abstractmethod
  def conditional_update_fn(self, h, x, t):
    """One update of the corrector.

    Args:
      x: A PyTorch tensor representing the current state
      t: A PyTorch tensor representing the current time step.

    Returns:
      x: A PyTorch tensor of the next state.
      x_mean: A PyTorch tensor. The next state without random noise. Useful for denoising.
    """
    pass


@register_predictor(name='euler_maruyama')
class EulerMaruyamaPredictor(Predictor):
  def __init__(self, sde, score_fn):
    super().__init__(sde, score_fn)

  def update_fn(self, x, t):
    dt = -1. / self.rsde.N
    z = torch.randn_like(x)
    drift, diffusion = self.rsde.sde(x, t)
    x_mean = x + drift * dt
    x = x_mean + diffusion[:, None] * np.sqrt(-dt) * z
    return x, x_mean

  def conditional_update_fn(self, h, x, t):
    dt = -1. / self.rsde.N
    z = torch.randn_like(x)
    drift, diffusion = self.rsde.conditional_sde(h, x, t)
    x_mean = x + drift * dt
    x = x_mean + diffusion[:, None] * np.sqrt(-dt) * z
    return x, x_mean


@register_predictor(name='reverse_diffusion')
class ReverseDiffusionPredictor(Predictor):
  def __init__(self, sde, score_fn):
    super().__init__(sde, score_fn)

  def update_fn(self, x, t):
    f, G = self.rsde.discretize(x, t)
    z = torch.randn_like(x)
    x_mean = x - f
    x = x_mean + G[:, None] * z
    return x, x_mean

  def conditional_update_fn(self, h, x, t):
    f, G = self.rsde.conditional_discretize(h, x, t)
    z = torch.randn_like(x)
    x_mean = x - f
    x = x_mean + G[:, None] * z
    return x, x_mean


@register_predictor(name='ancestral_sampling')
class AncestralSamplingPredictor(Predictor):
  """The ancestral sampling predictor. Currently only supports VE/VP SDEs."""

  def __init__(self, sde, score_fn):
    super().__init__(sde, score_fn)
    if not isinstance(sde, sde_lib.VPSDE) and not isinstance(sde, sde_lib.VESDE):
      raise NotImplementedError(f"SDE class {sde.__class__.__name__} not yet supported.")

  def vesde_update_fn(self, x, t):
    sde = self.sde
    timestep = (t * (sde.N - 1) / sde.T).long()
    sigma = sde.discrete_sigmas[timestep]
    adjacent_sigma = torch.where(timestep == 0, torch.zeros_like(t), sde.discrete_sigmas.to(t.device)[timestep - 1])
    score = self.score_fn(x, t)
    x_mean = x + score * (sigma ** 2 - adjacent_sigma ** 2)[:, None]
    std = torch.sqrt((adjacent_sigma ** 2 * (sigma ** 2 - adjacent_sigma ** 2)) / (sigma ** 2))
    noise = torch.randn_like(x)
    x = x_mean + std[:, None] * noise
    return x, x_mean
  def conditional_vesde_update_fn(self, h, x, t):
    sde = self.sde
    timestep = (t * (sde.N - 1) / sde.T).long()
    sigma = sde.discrete_sigmas[timestep]
    adjacent_sigma = torch.where(timestep == 0, torch.zeros_like(t), sde.discrete_sigmas.to(t.device)[timestep - 1])
    score = self.score_fn(h, x, t)
    x_mean = x + score * (sigma ** 2 - adjacent_sigma ** 2)[:, None]
    std = torch.sqrt((adjacent_sigma ** 2 * (sigma ** 2 - adjacent_sigma ** 2)) / (sigma ** 2))
    noise = torch.randn_like(x)
    x = x_mean + std[:, None] * noise
    return x, x_mean

  def vpsde_update_fn(self, x, t):
    sde = self.sde
    timestep = (t * (sde.N - 1) / sde.T).long()
    beta = sde.discrete_betas.to(t.device)[timestep]
    score = self.score_fn(x, t)
    x_mean = (x + beta[:, None] * score) / torch.sqrt(1. - beta)[:, None]
    noise = torch.randn_like(x)
    x = x_mean + torch.sqrt(beta)[:, None] * noise
    return x, x_mean
  def conditional_vpsde_update_fn(self, h, x, t):
    sde = self.sde
    timestep = (t * (sde.N - 1) / sde.T).long()
    beta = sde.discrete_betas.to(t.device)[timestep]
    score = self.score_fn(h, x, t)
    x_mean = (x + beta[:, None] * score) / torch.sqrt(1. - beta)[:, None]
    noise = torch.randn_like(x)
    x = x_mean + torch.sqrt(beta)[:, None] * noise
    return x, x_mean

  def update_fn(self, x, t):
    if isinstance(self.sde, sde_lib.VESDE):
      return self.vesde_update_fn(x, t)
    elif isinstance(self.sde, sde_lib.VPSDE):
      return self.vpsde_update_fn(x, t)

  def conditional_update_fn(self, h, x, t):
    if isinstance(self.sde, sde_lib.VESDE):
      return self.conditional_vesde_update_fn(h, x, t)
    elif isinstance(self.sde, sde_lib.VPSDE):
      return self.conditional_vpsde_update_fn(h, x, t)


@register_predictor(name='none')
class NonePredictor(Predictor):
  """An empty predictor that does nothing."""

  def __init__(self, sde, score_fn):
    pass

  def update_fn(self, x, t):
    return x, x
  def conditional_update_fn(self, h, x, t):
    return x, x


@register_corrector(name='langevin')
class LangevinCorrector(Corrector):
  def __init__(self, sde, score_fn, snr, n_steps):
    super().__init__(sde, score_fn, snr, n_steps)
    if not isinstance(sde, sde_lib.VPSDE) \
        and not isinstance(sde, sde_lib.VESDE) \
        and not isinstance(sde, sde_lib.subVPSDE):
      raise NotImplementedError(f"SDE class {sde.__class__.__name__} not yet supported.")

  def update_fn(self, x, t):
    sde = self.sde
    score_fn = self.score_fn
    n_steps = self.n_steps
    target_snr = self.snr
    if isinstance(sde, sde_lib.VPSDE) or isinstance(sde, sde_lib.subVPSDE):
      timestep = (t * (sde.N - 1) / sde.T).long()
      alpha = sde.alphas.to(t.device)[timestep]
    else:
      alpha = torch.ones_like(t)

    for i in range(n_steps):
      grad = score_fn(x, t)
      noise = torch.randn_like(x)
      grad_norm = torch.norm(grad.reshape(grad.shape[0], -1), dim=-1).mean()
      noise_norm = torch.norm(noise.reshape(noise.shape[0], -1), dim=-1).mean()
      step_size = (target_snr * noise_norm / grad_norm) ** 2 * 2 * alpha
      x_mean = x + step_size[:, None] * grad
      x = x_mean + torch.sqrt(step_size * 2)[:, None] * noise

    return x, x_mean

  def conditional_update_fn(self, h, x, t):
    sde = self.sde
    score_fn = self.score_fn
    n_steps = self.n_steps
    target_snr = self.snr
    if isinstance(sde, sde_lib.VPSDE) or isinstance(sde, sde_lib.subVPSDE):
      timestep = (t * (sde.N - 1) / sde.T).long()
      alpha = sde.alphas.to(t.device)[timestep]
    else:
      alpha = torch.ones_like(t)

    for i in range(n_steps):
      grad = score_fn(h, x, t)
      noise = torch.randn_like(x)
      grad_norm = torch.norm(grad.reshape(grad.shape[0], -1), dim=-1).mean()
      noise_norm = torch.norm(noise.reshape(noise.shape[0], -1), dim=-1).mean()
      step_size = (target_snr * noise_norm / grad_norm) ** 2 * 2 * alpha
      x_mean = x + step_size[:, None] * grad
      x = x_mean + torch.sqrt(step_size * 2)[:, None] * noise

    return x, x_mean


@register_corrector(name='ald')
class AnnealedLangevinDynamics(Corrector):
  """The original annealed Langevin dynamics predictor in NCSN/NCSNv2.

  We include this corrector only for completeness. It was not directly used in our paper.
  """

  def __init__(self, sde, score_fn, snr, n_steps):
    super().__init__(sde, score_fn, snr, n_steps)
    if not isinstance(sde, sde_lib.VPSDE) \
        and not isinstance(sde, sde_lib.VESDE) \
        and not isinstance(sde, sde_lib.subVPSDE):
      raise NotImplementedError(f"SDE class {sde.__class__.__name__} not yet supported.")

  def update_fn(self, x, t):
    sde = self.sde
    score_fn = self.score_fn
    n_steps = self.n_steps
    target_snr = self.snr
    if isinstance(sde, sde_lib.VPSDE) or isinstance(sde, sde_lib.subVPSDE):
      timestep = (t * (sde.N - 1) / sde.T).long()
      alpha = sde.alphas.to(t.device)[timestep]
    else:
      alpha = torch.ones_like(t)

    std = self.sde.marginal_prob(x, t)[1]

    for i in range(n_steps):
      grad = score_fn(x, t)
      noise = torch.randn_like(x)
      step_size = (target_snr * std) ** 2 * 2 * alpha
      x_mean = x + step_size[:, None] * grad
      x = x_mean + noise * torch.sqrt(step_size * 2)[:, None]

    return x, x_mean

  def conditional_update_fn(self, h, x, t):
    sde = self.sde
    score_fn = self.score_fn
    n_steps = self.n_steps
    target_snr = self.snr
    if isinstance(sde, sde_lib.VPSDE) or isinstance(sde, sde_lib.subVPSDE):
      timestep = (t * (sde.N - 1) / sde.T).long()
      alpha = sde.alphas.to(t.device)[timestep]
    else:
      alpha = torch.ones_like(t)

    std = self.sde.marginal_prob(x, t)[1]

    for i in range(n_steps):
      grad = score_fn(h, x, t)
      noise = torch.randn_like(x)
      step_size = (target_snr * std) ** 2 * 2 * alpha
      x_mean = x + step_size[:, None] * grad
      x = x_mean + noise * torch.sqrt(step_size * 2)[:, None]

    return x, x_mean


@register_corrector(name='none')
class NoneCorrector(Corrector):
  """An empty corrector that does nothing."""

  def __init__(self, sde, score_fn, snr, n_steps):
    pass

  def update_fn(self, x, t):
    return x, x

  def conditional_update_fn(self, h, x, t):
    return x, x


def shared_predictor_update_fn(sde, model, predictor, eps, conditional):
  """A wrapper that configures and returns the update function of predictors."""
  if conditional:
    score_fn = mutils.get_conditional_score_fn(sde, model, train=False)
  else:
    score_fn = mutils.get_score_fn(sde, model, train=False)

  if predictor is None:
    # Corrector-only sampler
    predictor_obj = NonePredictor(sde, score_fn)
  else:
    predictor_obj = predictor(sde, score_fn)
  if conditional:
    return predictor_obj.conditional_update_fn
  else:
    return predictor_obj.update_fn


def shared_corrector_update_fn(sde, model, corrector, snr, n_steps, conditional):
  """A wrapper tha configures and returns the update function of correctors."""
  if conditional:
    score_fn = mutils.get_conditional_score_fn(sde, model, train=False)
  else:
    score_fn = mutils.get_score_fn(sde, model, train=False)
  if corrector is None:
    # Predictor-only sampler
    corrector_obj = NoneCorrector(sde, score_fn, snr, n_steps)
  else:
    corrector_obj = corrector(sde, score_fn, snr, n_steps)
  if conditional:
    return corrector_obj.conditional_update_fn
  else:
    return corrector_obj.update_fn

def to_flattened_numpy(x):
  """Flatten a torch tensor `x` and convert it to numpy."""
  return x.detach().cpu().numpy().reshape((-1,))


def from_flattened_numpy(x, shape):
  """Form a torch tensor with the given `shape` from a flattened numpy array `x`."""
  return torch.from_numpy(x.reshape(shape))

def get_pc_sampler(sde, predictor, corrector, snr,
                   n_steps=1, eps = 1e-5, device='cuda'):

  def pc_sampler(conditional_model,sampling_shape):
    """ The PC sampler funciton.

    Args:
      model: A score model.
    Returns:
      Samples, number of function evaluations.
    """
    conditional_predictor_update_fn = shared_predictor_update_fn(sde=sde,
                                                    model=conditional_model,
                                                    predictor=predictor,
                                                    eps = eps,
                                                    conditional=True)
    conditional_corrector_update_fn = shared_corrector_update_fn(sde=sde,
                                                    model=conditional_model,
                                                    corrector=corrector,
                                                    snr=snr,
                                                    n_steps=n_steps,
                                                    conditional=True)
    with torch.no_grad():
      leng = sampling_shape[1]
      shape = [sampling_shape[0],sampling_shape[2]]
      x = sde.prior_sampling(shape).to(device)
      prev = torch.zeros(shape).to(device)
      timesteps = torch.linspace(sde.T, eps, sde.N, device=device)
      for i in range(sde.N):
        t = timesteps[i]
        vec_t = torch.ones(shape[0], device=t.device) * t
        x, x_mean = conditional_corrector_update_fn(prev, x, vec_t)
        x, x_mean = conditional_predictor_update_fn(prev, x, vec_t)
      prev = x
      result = x.unsqueeze(dim=1)
      for j in range(leng-1):
        x = sde.prior_sampling(shape).to(device)
        timesteps = torch.linspace(sde.T, eps, sde.N, device=device)
        for i in range(sde.N):
          t = timesteps[i]
          vec_t = torch.ones(shape[0], device=t.device) * t
          x, x_mean = conditional_corrector_update_fn(prev, x, vec_t)
          x, x_mean = conditional_predictor_update_fn(prev, x, vec_t)
        prev = x
        result = torch.cat([result,x.unsqueeze(dim=1)], dim=1)
    return result, sde.N * (n_steps + 1)
  return pc_sampler

def get_ode_sampler(sde, denoise=False, rtol=1e-5, atol=1e-5, method='RK45', eps=1e-3, device='cuda'):

  def denoise_update_fn(model, prev, x):
    score_fn = mutils.get_conditional_score_fn(sde, model, train=False)
    # Reverse diffusion predictor for denoising
    predictor_obj = ReverseDiffusionPredictor(sde, score_fn)
    vec_eps = torch.ones(x.shape[0], device=x.device) * eps
    _, x = predictor_obj.conditional_update_fn(prev, x, vec_eps)
    return x

  def drift_fn(model, prev, x, t):
    score_fn = mutils.get_conditional_score_fn(sde, model, train=False)
    rsde = sde.reverse(score_fn)
    return rsde.conditional_sde(prev, x, t)[0]

  def ode_sampler(model):

    with torch.no_grad():
      leng = 24
      shape = [3661,24]
      x = sde.prior_sampling(shape).to(device)
      prev = torch.zeros(shape).to(device)
      result_nfe = 0

      def ode_func(t, x):
        x = from_flattened_numpy(x, shape).to(device).type(torch.float32)
        vec_t = torch.ones(shape[0], device=x.device) * t
        drift = drift_fn(model, prev, x, vec_t)
        return to_flattened_numpy(drift)

      # Black-box ODE solver for the probability flow ODE
      solution = integrate.solve_ivp(ode_func, (sde.T, eps), to_flattened_numpy(x), rtol=rtol, atol=atol, method=method)
      nfe = solution.nfev
      x = torch.tensor(solution.y[:, -1]).reshape(shape).to(device).type(torch.float32)
      if denoise:
        x = denoise_update_fn(model, prev, x)

      result_nfe += nfe
      prev = x
      result = x.unsqueeze(dim=1)

      for j in range(leng-1):
        x = sde.prior_sampling(shape).to(device)
        solution = integrate.solve_ivp(ode_func, (sde.T, eps), to_flattened_numpy(x), rtol=rtol, atol=atol, method=method)
        nfe = solution.nfev
        x = torch.tensor(solution.y[:, -1]).reshape(shape).to(device).type(torch.float32)
        if denoise:
          x = denoise_update_fn(model, prev, x)
        result_nfe += nfe
        prev = x
        result = torch.cat([result,x.unsqueeze(dim=1)], dim=1)
        print(j)

      return result, result_nfe

  return ode_sampler