

#include <gudhi/Simplex_tree.h>
#include <Python.h>
#include <pybind11/pybind11.h>
#include <pybind11/stl.h>
#include "approximation.h"
#include "vineyards_trajectories.h" 

namespace py = pybind11;
using namespace pybind11::literals;



std::vector<std::vector<unsigned int> > _get_boundaries(Gudhi::Simplex_tree<> &simplexTree){
	std::vector<std::vector<unsigned int> > simplexList(simplexTree.num_simplices());
	unsigned int i = 0;
	for (auto& simplex : simplexTree.filtration_simplex_range()) {
		for (auto vertex : simplexTree.simplex_vertex_range(simplex))
			simplexList[i].push_back(vertex);
		i++;
	}
	for (unsigned int i = 0; i < simplexTree.num_simplices(); i++)
		std::sort(simplexList[i].begin(), simplexList[i].end());
	std::sort(simplexList.begin(), simplexList.end(), &is_strictly_smaller_simplex);

	return simplexList;
}





PYBIND11_MODULE(mma, m) {
	m.doc() = "Yet another Multi-parameter persistence Module Approximation library.";
	py::class_<Vineyard::Box>(m,"Box")
		.def(py::init<Vineyard::corner_type &, Vineyard::corner_type &>())
		.def("get_bottom_corner", &Vineyard::Box::get_bottom_corner, "Gets the lower corner of the box")
		.def("get_upper_corner", &Vineyard::Box::get_upper_corner, "Gets the upper corner of the box")
		.def("inflate", &Vineyard::Box::inflate)
		.def("contains", &Vineyard::Box::contains)
		// .def("__repr__",
        // [](const Vineyard::Box &box) {
        //     return box.get_bottom_corner(), box.get_upper_corner() ;
        // }
		;
	py::class_<Vineyard::Summand>(m,"Summand")
		// .def(py::init());
		.def("get_birth_list",&Vineyard::Summand::get_birth_list)
		.def("get_death_list", &Vineyard::Summand::get_death_list)
		.def("show", [](Vineyard::Summand &self,  std::vector<std::vector<double>> box){
			py::object pyf = py::module::import("python_fct");
			pyf.attr("plot_2d_summand")(
				self.get_birth_list(), 
				self.get_death_list(),
				box
				);
		},
		py::arg("box"), 
		"Shows the summand in the box.")
		;
	
	m.def("simplex_tree_to_boundary_matrix", [](intptr_t pointer){
		Gudhi::Simplex_tree<> simplextree = *(Gudhi::Simplex_tree<>*)(pointer); //Jardinage
		return Vineyard::simplex_tree_to_boundary_matrix(simplextree);
	}, "Turns a simplextree (pointer) to a boundary matrix.");
	m.def("get_boundaries",[](intptr_t splx_ptr){
		Gudhi::Simplex_tree<> &simplexTree = *(Gudhi::Simplex_tree<>*)(splx_ptr); //Jardinage

		return _get_boundaries(simplexTree);
	}
	);
	m.def("splxff", &build_boundary_matrix_from_simplex_list,
		py::arg("boundaries"),
		py::arg("filtrations"),
		py::arg("to_permute")=std::vector<unsigned int>(1), "old splx to boundary");
	m.def("simplextree_filtration_format", [](
			intptr_t splx_ptr
			// const std::vector<filtration_type> &filtrations, 
			// std::vector<unsigned int>& indices
			)
		{
			Gudhi::Simplex_tree<> simplexTree = *(Gudhi::Simplex_tree<>*)(splx_ptr); //Jardinage
			unsigned int numberOfSimplices = simplexTree.num_simplices();
			filtration_type complex_filtration(numberOfSimplices);
			std::vector<Gudhi::Simplex_tree<>::Simplex_handle> simplices(numberOfSimplices);

			std::vector<std::vector<unsigned int>> simplexList(numberOfSimplices);
			unsigned int count = 0;
			for (auto &simplex : simplexTree.filtration_simplex_range()){
				for (auto vertex : simplexTree.simplex_vertex_range(simplex)){
					simplexList[count].push_back(vertex);
				}
				complex_filtration[count] = simplexTree.filtration(simplex);
				simplices[count] = simplex;
				// simplexTree.assign_key(simplex, count++);
				count++;
			}
			for (unsigned int i = 0; i < numberOfSimplices; i++)
        		std::sort(simplexList[i].begin(), simplexList[i].end());
    		
			permutation_type p = Combinatorics::sort_and_return_permutation<boundary_type>(
                simplexList, &is_strictly_smaller_simplex);

			permutation_type q(numberOfSimplices); //the inverse of p
			for (unsigned int i =0; i < numberOfSimplices; i++){
				q[p[i]] = i;
			}

			Combinatorics::compose(complex_filtration, p);
			unsigned int scale = std::pow(10, std::ceil(std::log10(numberOfSimplices)));

			boundary_matrix boundaries(numberOfSimplices);
			
			for (unsigned int key = 0; key < numberOfSimplices; key++){
				const auto &simplex = simplices[q[key]];
				simplexTree.assign_key(simplex, key);

				for (auto &boundary_simplex : simplexTree.boundary_simplex_range(simplex)){
					std::cout << simplexTree.key(boundary_simplex) << " ";
					boundaries[key].push_back(simplexTree.key(boundary_simplex));
				}
				std::cout << "\n";
			}
			
			
			
			// std::unordered_map<unsigned int, unsigned int> simplexID;
			// for (unsigned int i = 0; i < numberOfSimplices; i++){
			// 	simplexID.emplace(
            //         hash_simplex_into_unsigned_int(simplexList[i], scale),
            //         i);
			// 	// If simplex is of dimension 0, there is no boundary
        	// 	if (simplexList[i].size() <= 1) continue;

			// 	// Fills the output matrix with the boundary of simplex cursor,
			// 	// and computes filtration of the simplex
			// 	for (unsigned int j = 0; j < simplexList[i].size(); j++){
			// 		// computes the id of the child
			// 		unsigned int childID =
			// 				simplexID[hash_simplex_face_into_unsigned_int(
			// 					simplexList[i], j, scale
			// 					)];

			// 		// add this child to the boundary
			// 		boundaries[i].push_back(childID);
     		//    }
			// }
			// for (unsigned int i = 0; i < numberOfSimplices; i++){
        	// 	std::sort(boundaries[i].begin(), boundaries[i].end());
    		// }
			// unsigned int count = 0;
			// for (auto splx : simplexTree.filtration_simplex_range()){
			// 	complex_filtration[count] = simplexTree.filtration(splx);
			// 	simplexTree.assign_key(splx, count++); // Thats a copy, we can reassign keys
			// }
			
			// std::vector<std::vector<unsigned int> > boundaries(numberOfSimplices);
			// unsigned int i = 0;
			// for (auto& simplex : simplexTree.filtration_simplex_range()) {
			// 	for (auto &boundary_simplex : simplexTree.boundary_simplex_range(simplex))
			// 		boundaries[i].push_back(simplexTree.key(boundary_simplex));
			// 	i++;
			// }

			// for (unsigned int i = 0; i < numberOfSimplices; i++){
        	// 	std::sort(boundaries[i].begin(), boundaries[i].end());
    		// }

			// permutation_type p = Combinatorics::sort_and_return_permutation<boundary_type>(
            //     boundaries, &is_strictly_smaller_simplex);
			// Combinatorics::compose(complex_filtration, p);
			return std::make_pair(boundaries, complex_filtration);
		},
		py::arg("simplexptr"),
		"Converts simplextree to boundary matrix and filtration vector."
		);
	// m.def("_boundary_filtration_format",[](
	// 	vector<vector<double>> &boundaries,
	// 	filtration_type &filter_to_permute
	// ){
	// 	for (unsigned int i = 0; i < numberOfSimplices; i++){
	// 		std::sort(boundaries[i].begin(), boundaries[i].end());
	// 	}
	// 	permutation_type p = Combinatorics::sort_and_return_permutation<boundary_type>(
	// 		boundaries, &is_strictly_smaller_simplex);
	// 	Combinatorics::compose(filter_to_permute, p);
	// 	return std::make_pair(boundaries, filter_to_permute);
	// });
	py::class_<Vineyard::Module>(m, "Module")	
		.def("__iter__", [](Vineyard::Module &mod){ 
				return py::make_iterator(mod.begin(), mod.end());
			}, py::keep_alive<0, 1>()
		)
		.def("__getitem__", [](Vineyard::Module &mod, unsigned int i) -> Vineyard::Summand{
			return mod.at(i);
		})
		.def("get_summands", [](Vineyard::Module &self, int dimension) -> std::vector<std::pair<Vineyard::Module::image_type,Vineyard::Module::image_type>>{
			std::vector<std::pair<Vineyard::Module::image_type,Vineyard::Module::image_type>> list;
			for (Vineyard::Summand &summand : self){
				if (summand.get_dimension() == dimension)
					list.push_back({summand.get_birth_list(), summand.get_death_list()});
			}
			return list;
		}, "Returns the summands of specified dimension.",
		py::arg("dimension")=0)
		.def("get_box", &Vineyard::Module::get_box, "Gets the box.")
		.def("get_vectorization", py::overload_cast<double,unsigned int,unsigned int, const Vineyard::Box&>(&Vineyard::Module::get_vectorization))
		.def("get_vectorization_in_dimension", py::overload_cast<const Vineyard::dimension_type, double,unsigned int,unsigned int, const Vineyard::Box&>(&Vineyard::Module::get_vectorization_in_dimension))
		.def("dimension", &Vineyard::Module::dimension, "Returns the maximum dimension of the module.")
		.def("plot2d",[](
			Vineyard::Module &self,
			unsigned int dimension, 
			bool separated,
			double min_interleaving,
			double alpha)
			{
				if(self.get_box().get_bottom_corner().size() != 2 ){
					std::cout << "#parameters is " << self.get_box().get_bottom_corner().size() << " != 2\n";
					return;
				}
				std::vector<std::pair<Vineyard::Module::image_type,Vineyard::Module::image_type>> corners;
				for (Vineyard::Summand &summand : self)
					if (summand.get_dimension() == dimension)
						corners.push_back({summand.get_birth_list(), summand.get_death_list()});
				const std::vector<double> lCorner = self.get_box().get_bottom_corner();
				const std::vector<double> uCorner = self.get_box().get_upper_corner();

				const std::vector<double> xlim = {lCorner[0], uCorner[0]};
				const std::vector<double> ylim = {lCorner.back(), uCorner.back()};
				py::object pyf = py::module_::import("python_fct");
				pyf.attr("plot_module")(
					corners,
					xlim, ylim,
					separated, 
					min_interleaving,
					alpha);
			},
			py::arg("dimension")=0,
			py::arg("separated")=false,
			py::arg("min_interleaving")=0,
			py::arg("alpha")=1,
			"Plots the module"
		)
		;
	m.def("_approx",[](
		boundary_matrix &B,
        std::vector<filtration_type> &f,
        const double p,
// 		Vineyard::Box &box,
		std::pair<filtration_type,filtration_type> bbox,
        const bool threshold,
        const bool complete,
        // const bool multithread,
        const bool verbose){
			Vineyard::Box box(bbox.first, bbox.second);
			if (box.get_bottom_corner().size() <= 1 || box.get_upper_corner().size() <= 1){
				std::cout << "Box not given.\n";
				box.infer_from_filters(f);
			}
			if(f.size() != box.get_bottom_corner().size() || f.size() != box.get_upper_corner().size()){
				std::cout<< "Filters and box must be of the same dimension! " << f.size() << " and " << box.get_upper_corner().size() << std::endl;
				return Vineyard::Module(box);
			}
			Vineyard::Module module = Vineyard::compute_vineyard_barcode_approximation(B,f, p, box, threshold, true, false, verbose);
			
			return module;
		},
		"Approximate a module",
		py::arg("boundary"),
		py::arg("filters_list"),
		py::arg("precision") = 0.01,
		py::arg("box") = std::make_pair(filtration_type(), filtration_type()),
		py::arg("threshold") = true,
		py::arg("complete") = true,
		py::arg("verbose") = false
	);
	
}	




