"""
Base Language Model Interface

This module defines the base interface for language models.
"""

import json
import logging
import os
from abc import ABC, abstractmethod
from pathlib import Path
from typing import Any, Dict, List

from src.utils.decorator_utils import with_logger

API_URL = "OPENAI_BASE_URL"
API_KEY = "OPENAI_API_KEY"
CREDENTIALS_PATH = "credentials/cred.json"


class EmbeddingsInterface(ABC):
    """
    Interface for Embeddings models.

    This abstract class defines the interface that all language model
    implementations must adhere to.
    """

    def __init__(self):
        """Initialise the LLM interface."""
        pass

    @with_logger
    def load_api_info(self) -> tuple[str, str]:
        """
        Load API URL and API key from environment variables or credentials file.

        Returns:
            tuple: (api_url, api_key)

        Raises:
            ValueError: If API URL or API key could not be loaded
        """

        # Try to load from environment variables first
        api_url = os.environ.get(API_URL)
        api_key = os.environ.get(API_KEY)

        # If environment variables not set, try to load from credentials file
        if not api_url or not api_key:
            logger.warning(
                "OPENAI_API_KEY & OPENAI_BASE_URL environment variables not set"
            )
            try:
                logger.info(f"Attempting to load credentials from {CREDENTIALS_PATH}")
                cred_path = Path(CREDENTIALS_PATH)
                if cred_path.exists():
                    with open(cred_path, "r") as f:
                        creds = json.load(f)
                    api_url = api_url or creds.get(API_URL)
                    api_key = api_key or creds.get(API_KEY)
            except FileNotFoundError:
                logger.warning(f"Credentials file {CREDENTIALS_PATH} not found")
            except json.JSONDecodeError:
                logger.warning("Failed to parse credentials file as JSON")
            except Exception as e:
                logger.warning(f"Error loading credentials: {str(e)}")

        if not api_url or not api_key:
            logger.warning(
                "Could not load API URL or API key from environment or credentials file"
            )
            raise ValueError(
                "Missing API credentials: Could not load API URL or API key from environment variables or credentials file"
            )

        return api_url, api_key

    @with_logger
    def configure_environment(self, base_url: str, api_key: str) -> None:
        os.environ.update(
            {
                "OPENAI_API_KEY": f"{api_key}",
                "OPENAI_BASE_URL": f"{base_url}",
            }
        )

    @abstractmethod
    def generate(self, messages: List[Dict[str, Any]], **kwargs: Any) -> str:
        """
        Generate text based on the provided messages.

        Args:
            messages: A list of message dictionaries, each with 'role' and 'content' keys
            **kwargs: Additional keyword arguments for the generation process

        Returns:
            The generated text
        """
        pass

    @with_logger
    def log_request(self, items: List[str], **kwargs: Any) -> None:
        """
        Log information about a generation request.

        Args:
            messages: A list of message dictionaries, each with 'role' and 'content' keys
            **kwargs: Additional keyword arguments for the generation process
        """
        num_messages = len(items)

        logger.info(f"Request: {num_messages} messages")
        logger.info(f"Request parameters: {kwargs}")

        if logger.isEnabledFor(logging.DEBUG):
            for i, msg in enumerate(items):
                # role = msg.get("role", "unknown")
                # content = msg.get("content", "")
                content_preview = msg[:50] + "..." if len(msg) > 50 else msg
                logger.info(f"Embeddings comparison Prompt {i} - {content_preview}")

    @with_logger
    def log_response(self, item: str, duration: float) -> None:
        """
        Log information about a generation response.

        Args:
            response: The generated text
            duration: The time taken to generate the response in seconds
        """
        item_tokens = len(item) // 4  # Rough estimate

        logger.info(f"Response: ~{item_tokens} tokens in {duration:.2f}s")

        if logger.isEnabledFor(logging.DEBUG):
            item_preview = item[:100] + "..." if len(item) > 100 else item
            logger.info(f"Response preview: {item_preview}")

    @property
    @abstractmethod
    def model_info(self) -> Dict[str, Any]:
        """
        Get information about the model.

        Returns:
            A dictionary containing information about the model
        """
        pass
