"""
Evaluator Implementation

This module provides a concrete implementation of the evaluation interface.
"""

from typing import Any, Callable, Dict, List, Optional

import pandas as pd

from src.core.registry import evaluator_registry
from src.evaluation.base import BaseEvaluator


@evaluator_registry.register("default")
class Evaluator(BaseEvaluator):
    """
    Default evaluator implementation.
    
    This class provides a concrete implementation of the evaluation interface
    with common metrics.
    """
    
    def __init__(self):
        """
        initialise the evaluator.
        """
        super().__init__()
        
        # Register common metrics
        self.register_metric("mean", self._calculate_mean)
        self.register_metric("sum", self._calculate_sum)
        self.register_metric("min", self._calculate_min)
        self.register_metric("max", self._calculate_max)
    
    def _calculate_mean(self, results: Any) -> float:
        """
        Calculate the mean of the results.
        
        Args:
            results: The results to evaluate
            
        Returns:
            The mean value
        """
        if isinstance(results, pd.DataFrame):
            if "score" in results.columns:
                return results["score"].mean()
            else:
                return 0.0
        elif isinstance(results, (list, tuple)):
            return sum(results) / len(results) if results else 0.0
        else:
            return 0.0
    
    def _calculate_sum(self, results: Any) -> float:
        """
        Calculate the sum of the results.
        
        Args:
            results: The results to evaluate
            
        Returns:
            The sum value
        """
        if isinstance(results, pd.DataFrame):
            if "score" in results.columns:
                return results["score"].sum()
            else:
                return 0.0
        elif isinstance(results, (list, tuple)):
            return sum(results)
        else:
            return 0.0
    
    def _calculate_min(self, results: Any) -> float:
        """
        Calculate the minimum of the results.
        
        Args:
            results: The results to evaluate
            
        Returns:
            The minimum value
        """
        if isinstance(results, pd.DataFrame):
            if "score" in results.columns:
                return results["score"].min()
            else:
                return 0.0
        elif isinstance(results, (list, tuple)):
            return min(results) if results else 0.0
        else:
            return 0.0
    
    def _calculate_max(self, results: Any) -> float:
        """
        Calculate the maximum of the results.
        
        Args:
            results: The results to evaluate
            
        Returns:
            The maximum value
        """
        if isinstance(results, pd.DataFrame):
            if "score" in results.columns:
                return results["score"].max()
            else:
                return 0.0
        elif isinstance(results, (list, tuple)):
            return max(results) if results else 0.0
        else:
            return 0.0